/* Logwi
   Copyright (C) 2007 Herve Quatremain
 
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * lwc_linkedlist.c -- Linked list functions
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_linkedlist.h>


/*
 * Create a new linked list
 *
 * Return:
 *   A pointer to the newly allocated linked list object or
 *   NULL in case of malloc error
 */
lwc_LL *
lwc_newLL ()
{
	lwc_LL *ptr;

	ptr = (lwc_LL *)malloc (sizeof (lwc_LL));
	if (ptr == (lwc_LL *)0) {
		return (lwc_LL *)0;
	}

	ptr->start = (struct _lwc_linkedNode *)0;
	ptr->end   = (struct _lwc_linkedNode *)0;
	ptr->ptr   = (struct _lwc_linkedNode *)0;
	ptr->num   = 0;

	return ptr;
}

/*
 * Destroy a linked list object and all its items (if deleteNote function
 * is not NULL)
 */
void
lwc_delLL (lwc_LL *ptr, void (*deleteNode)(const void *))
{
	struct _lwc_linkedNode *node, *save;

	if (ptr != (lwc_LL *)0) {
		node = ptr->start;
		while (node != (struct _lwc_linkedNode *)0) {
			save = node;
			node = node->next;
			if (	   deleteNode != (void (*)(const void *))0
				&& save->obj != (void *)0)
			{
				(*deleteNode) (save->obj);
			}
			free (save);
		}
		free (ptr);
	}
}

/*
 * Destroy all nodes of a linked list (if deleteNote function
 * is not NULL, object associated with each node is also destroyed)
 */
void
lwc_emptyLL (lwc_LL *ptr, void (*deleteNode)(const void *))
{
	struct _lwc_linkedNode *node, *save;

	if (ptr != (lwc_LL *)0) {
		node = ptr->start;
		while (node != (struct _lwc_linkedNode *)0) {
			save = node;
			node = node->next;
			if (	   deleteNode != (void (*)(const void *))0
				&& save->obj != (void *)0)
			{
				(*deleteNode) (save->obj);
			}
			free (save);
		}
		ptr->start = (struct _lwc_linkedNode *)0;
		ptr->end   = (struct _lwc_linkedNode *)0;
		ptr->ptr   = (struct _lwc_linkedNode *)0;
		ptr->num   = 0;
	}
}



/*
 * Search and return an item in a sorter list
 * The linked list must be in ascending sorted order (ie. items must be
 * added with the lwc_addLL() or lwc_replaceLL() functions and not with the
 * lwc_addStartLL() or addEndLL() functions).
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * less than, equal to, or greater than zero if the key object is found,
 * respectively, to be less than, to match, or be greater than the list item.
 * 
 * Return:
 *   A pointer to the object found or
 *   NULL if the item is not found
 */
void *
lwc_searchSortLL (lwc_LL *ptr, const void *key,
				int (*compar)(const void *, const void *))
{
	struct _lwc_linkedNode *node;
	int comp_result;

	if (ptr == (lwc_LL *) 0) {
		return (void *)0;
	}

	for (	node = ptr->start;
		node != (struct _lwc_linkedNode *)0;
		node = node->next)
	{
		comp_result = (*compar) (key, node->obj);
		if (comp_result == 0) {
			return node->obj;
		}
		if (comp_result < 0) {
			return (void *)0;
		}
	}
	return (void *)0;
}

/*
 * Search and return an item in an unsorted list
 * The linked list do not have to be in ascending sorted order (ie. items can be
 * added with the lwc_addLL(), lwc_replaceLL(), lwc_addStartLL() or addEndLL()
 * functions).
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * equal to zero if the key object is found to match a list item or non zero
 * if it does not match.
 *
 * Return:
 *   A pointer to the object found or
 *   NULL if the item is not found
 */
void *
lwc_searchLL (lwc_LL *ptr, const void *key,
				int (*compar)(const void *, const void *))
{
	struct _lwc_linkedNode *node;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	for (	node = ptr->start;
		node != (struct _lwc_linkedNode *)0;
		node = node->next)
	{
		if ((*compar) (key, node->obj) == 0) {
			return node->obj;
		}
	}
	return (void *)0;
}


/*
 * Add a new item in a sorted linked list. The item added must not already
 * exists in the linked list.
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * less than, equal to, or greater than zero if the key object is found,
 * respectively, to be less than, to match, or be greater than the list item.
 * 
 * Return:
 *    0 --> Item successfully added
 *   -1 --> malloc error
 *   -2 --> Item already in list
 */
int
lwc_addLL (lwc_LL *ptr, void *key,
		int (*compar)(const void *, const void *))
{
	struct _lwc_linkedNode *nextNode, *prevNode, *newNode;
	int comp_result;

#if HAVE_ASSERT_H
	assert (ptr != (lwc_LL *)0);
#endif

	prevNode = (struct _lwc_linkedNode *)0;
	for (	nextNode = ptr->start;
		nextNode != (struct _lwc_linkedNode *)0;
		nextNode = nextNode->next)
	{
		comp_result = (*compar) (key, nextNode->obj);
		if (comp_result == 0) {
			return -2; /* Already in list */
		}
		if (comp_result < 0) {
			break;
		}
		prevNode = nextNode;
	}
	newNode = (struct _lwc_linkedNode *) malloc
					(sizeof (struct _lwc_linkedNode));
	if (newNode == (struct _lwc_linkedNode *)0) {
		return -1;
	}
	ptr->num++;
	newNode->obj  = key;
	newNode->next = nextNode;
	newNode->prev = prevNode;
	if (prevNode == (struct _lwc_linkedNode *)0) {
		ptr->start = newNode;
	}
	else {
		prevNode->next = newNode;
	}
	if (nextNode == (struct _lwc_linkedNode *)0) {
		ptr->end = newNode;
	}
	else {
		nextNode->prev = newNode;
	}
	return 0;
}


/*
 * Add a new item in a sorted linked list. If the item is already in list, it
 * is replaced with the new item (key) and the old one is saved in the old
 * parameter (if old is not NULL). If the old parameter is NULL, then the old
 * item is lost (this can lead to memory leak !).
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * less than, equal to, or greater than zero if the key object is found,
 * respectively, to be less than, to match, or be greater than the list item.
 * 
 * Return:
 *    0 --> Item successfully added (*old is NULL if no old item is replaced.
 *          Else *old is set to the old object).
 *   -1 --> malloc error
 */
int
lwc_replaceLL (lwc_LL *ptr, void *key,
			int (*compar)(const void *, const void *),
			void **old)
{
	struct _lwc_linkedNode *nextNode, *prevNode, *newNode;
	int comp_result;

#if HAVE_ASSERT_H
	assert (ptr != (lwc_LL *)0);
#endif

	prevNode = (struct _lwc_linkedNode *)0;
	for (	nextNode = ptr->start;
		nextNode != (struct _lwc_linkedNode *)0;
		nextNode = nextNode->next)
	{
		comp_result = (*compar) (key, nextNode->obj);
		if (comp_result == 0) {
			if (old != (void **)0) {
				*old = nextNode->obj;
			}
			nextNode->obj = key;
			return 0; 
		}
		if (comp_result < 0) {
			break;
		}
		prevNode = nextNode;
	}
	newNode = (struct _lwc_linkedNode *) malloc
					(sizeof (struct _lwc_linkedNode));
	if (newNode == (struct _lwc_linkedNode *)0) {
		return -1;
	}
	ptr->num++;
	newNode->obj  = key;
	newNode->next = nextNode;
	newNode->prev = prevNode;
	if (prevNode == (struct _lwc_linkedNode *)0) {
		ptr->start = newNode;
	}
	else {
		prevNode->next = newNode;
	}
	if (nextNode == (struct _lwc_linkedNode *)0) {
		ptr->end = newNode;
	}
	else {
		nextNode->prev = newNode;
	}
	if (old != (void **)0) {
		*old = (void *)0;
	}
	return 0;
}


/*
 * Add a new item in an unsorted linked list. If the item is already in list,
 * it is replaced with the new item (key) and the old one is saved in the old
 * parameter (if old is not NULL). If the old parameter is NULL, then the old
 * item is lost (this can lead to memory leak !).
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * equal to, or different from zero if the key object is found,
 * respectively, to match, or to not match the list item.
 * 
 * Return:
 *    0 --> Item successfully added (*old is NULL if no old item is replaced.
 *          Else *old is set to the old object).
 *   -1 --> malloc error
 */
int
lwc_replaceUnsortedLL (lwc_LL *ptr, void *key,
			int (*compar)(const void *, const void *),
			void **old)
{
	struct _lwc_linkedNode *nextNode, *prevNode, *newNode;
	int comp_result;

#if HAVE_ASSERT_H
	assert (ptr != (lwc_LL *)0);
#endif

	prevNode = (struct _lwc_linkedNode *)0;
	for (	nextNode = ptr->start;
		nextNode != (struct _lwc_linkedNode *)0;
		nextNode = nextNode->next)
	{
		comp_result = (*compar) (key, nextNode->obj);
		if (comp_result == 0) {
			if (old != (void **)0) {
				*old = nextNode->obj;
			}
			nextNode->obj = key;
			return 0; 
		}
		prevNode = nextNode;
	}
	newNode = (struct _lwc_linkedNode *) malloc
					(sizeof (struct _lwc_linkedNode));
	if (newNode == (struct _lwc_linkedNode *)0) {
		return -1;
	}
	ptr->num++;
	newNode->obj  = key;
	newNode->next = (struct _lwc_linkedNode *)0;
	newNode->prev = prevNode;
	ptr->end      = newNode;
	if (prevNode == (struct _lwc_linkedNode *)0) {
		ptr->start = newNode;
	}
	else {
		prevNode->next = newNode;
	}
	if (old != (void **)0) {
		*old = (void *)0;
	}
	return 0;
}


/*
 * Add an item at the begining of the linked list
 *
 * Return:
 *   0 --> Item successfully added
 *  -1 --> malloc error
 */
int
lwc_addStartLL (lwc_LL *ptr, void *key)
{
	struct _lwc_linkedNode * newNode;

#if HAVE_ASSERT_H
	assert (ptr != (lwc_LL *)0);
#endif

	newNode = (struct _lwc_linkedNode *) malloc
					(sizeof (struct _lwc_linkedNode));
	if (newNode == (struct _lwc_linkedNode *)0) {
		return -1;
	}
	ptr->num++;
	newNode->obj  = key;
	newNode->next = ptr->start;
	ptr->start    = newNode;
	newNode->prev = (struct _lwc_linkedNode *)0;
	if (newNode->next != (struct _lwc_linkedNode *)0) {
		newNode->next->prev = newNode;
	}
	else {
		ptr->end = newNode;
	}
	return 0;
}

/*
 * Add an item at the end of the linked list
 *
 * Return:
 *   0 --> Item successfully added
 *  -1 --> malloc error
 */
int
lwc_addEndLL (lwc_LL *ptr, void *key)
{
	struct _lwc_linkedNode *newNode;

#if HAVE_ASSERT_H
	assert (ptr != (lwc_LL *)0);
#endif

	newNode = (struct _lwc_linkedNode *) malloc
					(sizeof (struct _lwc_linkedNode));
	if (newNode == (struct _lwc_linkedNode *)0) {
		return -1;
	}
	ptr->num++;
	newNode->obj  = key;

	newNode->prev = ptr->end;
	ptr->end      = newNode;
	newNode->next = (struct _lwc_linkedNode *)0;
	if (newNode->prev != (struct _lwc_linkedNode *)0) {
		newNode->prev->next = newNode;
	}
	else {
		ptr->start = newNode;
	}
	return 0;
}

/*
 * Delete the first node of the linked list and return the associated object
 *
 * Return:
 *   A pointer to the object associated with the deleted node or
 *   NULL if there is no item in the list
 */
void *
lwc_delStartLL (lwc_LL *ptr)
{
	void *item;
	struct _lwc_linkedNode *delNode;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	delNode = ptr->start;
	if (delNode != (struct _lwc_linkedNode *)0) {
		item = delNode->obj;
		ptr->start = delNode->next;
		if (ptr->start != (struct _lwc_linkedNode *)0) {
			ptr->start->prev = (struct _lwc_linkedNode *)0;
			ptr->num--;
			if (ptr->ptr == delNode) {
				ptr->ptr = ptr->start;
			}
		}
		else {
			ptr->end = (struct _lwc_linkedNode *)0;
			ptr->ptr = (struct _lwc_linkedNode *)0;
			ptr->num = 0;
		}
		free (delNode);
		return item;
	}
	else {
		return (void *)0;
	}

}

/*
 * Delete the last node of the linked list and return the associated object
 *
 * Return:
 *   A pointer to the object associated with the deleted node or
 *   NULL if there is no item in the list
 */
void *
lwc_delEndLL (lwc_LL *ptr)
{
	void *item;
	struct _lwc_linkedNode *delNode;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	delNode = ptr->end;
	if (delNode != (struct _lwc_linkedNode *)0) {
		item = delNode->obj;
		ptr->end = delNode->prev;
		if (ptr->end != (struct _lwc_linkedNode *)0) {
			ptr->end->next = (struct _lwc_linkedNode *)0;
			ptr->num--;
			if (ptr->ptr == delNode) {
				ptr->ptr = (struct _lwc_linkedNode *)0;
			}
		}
		else {
			ptr->start = (struct _lwc_linkedNode *)0;
			ptr->ptr   = (struct _lwc_linkedNode *)0;
			ptr->num   = 0;
		}
		free (delNode);
		return item;
	}
	else {
		return (void *)0;
	}

}

/*
 * Delete an item from a sorted linked list and return the associated object
 * (or NULL if the key cannot be found).
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * less than, equal to, or greater than zero if the key object is found,
 * respectively, to be less than, to match, or be greater than the list item.
 *
 * Return:
 *  The object associated with the deleted item or
 *  NULL if the key is not found
 */
void *
lwc_delNodeLL (lwc_LL *ptr, const void *key,
			int (*compar)(const void *, const void *))
{
	struct _lwc_linkedNode *node;
	int comp_result;
	void *item = (void *)0;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	for (	node = ptr->start;
		node != (struct _lwc_linkedNode *)0;
		node = node->next)
	{
		comp_result = (*compar) (key, node->obj);
		if (comp_result == 0) {
			item = node->obj;
			break;
		}
		if (comp_result < 0) {
			return (void *)0;
		}
	}
	if (node == (struct _lwc_linkedNode *)0) {
		return (void *)0;
	}
	if (node->prev == (struct _lwc_linkedNode *)0) {
		ptr->start = node->next;
	}
	else {
		node->prev->next = node->next;
	}
	if (node->next == (struct _lwc_linkedNode *)0) {
		ptr->end = node->prev;
	}
	else {
		node->next->prev = node->prev;
	}
	if (ptr->ptr == node) {
		ptr->ptr = node->next;
	}
	free (node);
	ptr->num--;
	return item;
}

/*
 * Delete an item from an unsorted linked list and return the associated object
 * (or NULL if the key cannot be found).
 * The compar routine is expected to have two arguments which point to the key
 * object and to an array member, in that order, and should return an integer
 * equal to zero if the key object match the list item.
 *
 * Return:
 *  The object associated with the deleted item or
 *  NULL if the key is not found
 */
void *
lwc_delUnsortedNodeLL (lwc_LL *ptr, const void *key,
			int (*compar)(const void *, const void *))
{
	struct _lwc_linkedNode *node;
	int comp_result;
	void *item = (void *)0;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	for (	node = ptr->start;
		node != (struct _lwc_linkedNode *)0;
		node = node->next)
	{
		comp_result = (*compar) (key, node->obj);
		if (comp_result == 0) {
			item = node->obj;
			break;
		}
	}
	if (node == (struct _lwc_linkedNode *)0) {
		return (void *)0;
	}
	if (node->prev == (struct _lwc_linkedNode *)0) {
		ptr->start = node->next;
	}
	else {
		node->prev->next = node->next;
	}
	if (node->next == (struct _lwc_linkedNode *)0) {
		ptr->end = node->prev;
	}
	else {
		node->next->prev = node->prev;
	}
	if (ptr->ptr == node) {
		ptr->ptr = node->next;
	}
	free (node);
	ptr->num--;
	return item;
}

/*
 * Set the access pointer to the begining of the list.
 * This function with the next one (lwc_nextLL()) is used to sequentially
 * access all items of the linked list.
 * Deletion, insertion and searching of items (lwc_delNodeLL(), lwc_addLL(),
 * lwc_searchLL(), ...) are allowed while 'visiting' the linked list with
 * lwc_rewindLL() and lwc_nextLL().
 */
void
lwc_rewindLL (lwc_LL *ptr)
{
	if (ptr != (lwc_LL *)0) {
		ptr->ptr = ptr->start;
	}
}

/*
 * Return the next object of the linked list
 *
 * Return:
 *   A pointer to the next object or
 *   NULL if no more objects are found (end of list is reached).
 */
void *
lwc_nextLL (lwc_LL *ptr)
{
	struct _lwc_linkedNode *toReturn;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	toReturn = ptr->ptr;
	if (toReturn == (struct _lwc_linkedNode *)0) {
		return (void *)0;
	}
	else {
		ptr->ptr = toReturn->next;
		return toReturn->obj;
	}
}


/*
 * Delete the item at the current position in the list (see lwc_rewindLL()
 * and lwc_nextLL()) and return the associated object (or NULL if the position
 * is not set. ie lwc_nextLL() has not been called yet)
 *
 * Return:
 *  The object associated with the deleted item or
 *  NULL if the position is not set
 */
void *
lwc_delCurrentLL (lwc_LL *ptr)
{
	struct _lwc_linkedNode *toDelete;
	void *item;

	if (ptr == (lwc_LL *)0) {
		return (void *)0;
	}

	if (ptr->ptr == (struct _lwc_linkedNode *)0) {
		toDelete = ptr->end;
	}
	else {
		toDelete = ptr->ptr->prev;
	}
	if (toDelete == (struct _lwc_linkedNode *)0) {
		return (void *)0;
	}

	item = toDelete->obj;
	
	if (toDelete->prev == (struct _lwc_linkedNode *)0) {
		ptr->start = toDelete->next;
	}
	else {
		toDelete->prev->next = toDelete->next;
	}
	if (toDelete->next == (struct _lwc_linkedNode *)0) {
		ptr->end = toDelete->prev;
	}
	else {
		toDelete->next->prev = toDelete->prev;
	}
	free (toDelete);
	ptr->num--;
	return item;
}


/*
 * Concatenate two unsorted lists (add src at the end of dst and empty src)
 */
void
lwc_concatUnsortedLL (lwc_LL *src, lwc_LL *dst)
{
	if (src == (lwc_LL *)0 || lwc_getNumNode (src) <= 0) {
		return;
	}

#if HAVE_ASSERT_H
	assert (dst != (lwc_LL *)0);
#endif

	dst->num += src->num;
	if (	   dst->start == (struct _lwc_linkedNode *)0
		|| dst->end == (struct _lwc_linkedNode *)0)
	{
		dst->start = src->start;
		dst->end   = src->end;
		dst->ptr   = (struct _lwc_linkedNode *)0;
	}
	else {
		dst->end->next = src->start;
		src->start->prev = dst->end;
		dst->end = src->end;
	}

	/* Empty the src list */
	src->num = 0;
	src->start = (struct _lwc_linkedNode *)0;
	src->end = (struct _lwc_linkedNode *)0;
	src->ptr = (struct _lwc_linkedNode *)0;
}

/******************************** End Of File ********************************/
