/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_env.c -- environments and environment_var table management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_env.h>

#define SQL_ENV_LIST "SELECT id,name FROM environments_s WHERE workload_date=%d ORDER BY name"
#define SQL_ENV_DETAIL "SELECT id,name,description FROM environments_s WHERE workload_date=%d AND id=\"%s\""
#define SQL_ENV_UPDATE "UPDATE environments SET name=\"%s\",description=\"%s\" WHERE id=\"%s\""
#define SQL_ENV_NEW "INSERT INTO environments (name,description) VALUES (\"%s\",\"%s\")"
#define SQL_ENV_GET_VARIABLES "SELECT position,env_key,env_value FROM environment_var_s WHERE workload_date=%d AND env_id=\"%s\" ORDER BY position"
#define SQL_ENV_ADD_VARIABLE "REPLACE INTO environment_var (env_id,position,env_key,env_value) VALUES "
#define SQL_ENV_ADD_VARIABLE_VALUES "(\"%s\",%d,\"%s\",\"%s\")"
#define SQL_ENV_DELETE_VARIABLES "DELETE FROM environment_var WHERE env_id=\"%s\""
#define SQL_ENV_DELETE "DELETE FROM environments WHERE id=\"%s\""


/*
 * Retrieve from the database the environments list
 *
 * Only the environments not in the provided current_envs list are taken into
 * account.
 * For each environment, the provided add_row() function is called with the
 * following parameters:
 *     1) the provided user_data_add_row parameter
 *     2) the environment id
 *     3) the environment name
 * add_row() must return 0 if successful or an other code in case of error.
 *
 * Return:
 *  >0 --> No error.  The returned value is the number of environments (minus
 *         the one in current_envs)
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_env_list (	int workload_date,
		lwc_LL *current_envs,
		int (*add_row)(void *, const char *, const char *),
		void *user_data_add_row,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	int nb;
	unsigned int ret;
	char **row;

#if HAVE_ASSERT_H
	assert (add_row != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_ENV_LIST,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	nb = 0;
	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		if (lwc_searchLL (current_envs, row[0],
			(int (*)(const void *, const void *))strcmp) == NULL)
		{
			if (add_row (user_data_add_row, row[0], row[1]) != 0) {
				sql_free_row (row);
				lwc_delLL (rows, (void (*)(const void *))
								sql_free_row);
				return -1;
			}
			nb++;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);
	return nb;
}


/*
 * Get the environment details from the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_env_get_main (	int workload_date, const char *env_id,
			void (*set_env_id)(const char *, void *),
			void (*set_env_name)(const char *, void *),
			void (*set_env_descr)(const char *, void *),
			void *user_data_set_env,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows;
	char **row;

#if HAVE_ASSERT_H
	assert (env_id != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_ENV_DETAIL,
				SQL_INT, (long int)workload_date,
				SQL_STRING, env_id,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (char **) lwc_delStartLL (rows);
	if (row == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Unknown host id in the database"),
					ret);
		}
		return 2;
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	set_env_id (row[0], user_data_set_env);
	set_env_name (row[1], user_data_set_env);
	set_env_descr (row[2], user_data_set_env);

	sql_free_row (row);

	return 0;
}


/*
 * Insert a new environment in the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_env_new (	const char *name,
		const char *description,
		unsigned long int *new_env_id,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (name != NULL && description != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, new_env_id, NULL,
				SQL_ENV_NEW,
				SQL_STRING, name,
				SQL_STRING, description,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Update an enviroment in the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_env_update (	const char *env_id,
			const char *name,
			const char *description,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (env_id != NULL && name != NULL && description != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_ENV_UPDATE,
				SQL_STRING, name,
				SQL_STRING, description,
				SQL_STRING, env_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Retrieve from the database the variables associated with an environment
 *
 * For each variable, the provided add_row() function is called with the
 * following parameters:
 *     1) the provided user_data_add_row parameter
 *     2) the position of the variable in the environment
 *     3) the variable key
 *     4) the variable value
 * add_row() must return 0 if successful or an other code in case of error.
 *
 * Return:
 *   0 --> No error
 *  -1 --> The add_row() function returned with a code different from 0
 *  -2 --> Other error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_env_get_variables (
		int workload_date,
		const char *env_id,
		int (*add_row)(void *, int, const char *, const char *),
		void *user_data_add_row,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	char **row;
	unsigned int ret;
	int pos;

#if HAVE_ASSERT_H
	assert (env_id != NULL && add_row != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_ENV_GET_VARIABLES,
				SQL_INT, (long int)workload_date,
				SQL_STRING, env_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	pos = 1;
	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		if (add_row (user_data_add_row, pos, row[1], row[2]) != 0) {
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return -1;
		}
		sql_free_row (row);
		pos++;
	}
	lwc_delLL (rows, NULL);
	return 0;
}


/*
 * Set the variables of an environment
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 *  -3 --> Internal error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_env_set_variables (	const char *env_id,
			lwc_LL *list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *req;
	unsigned int req_len, idx;
	char *err_msg = NULL;
	char *esc_envid, *key, *value;
	int pos, ret;
	char **var;
	unsigned int sql_ret;

#if HAVE_ASSERT_H
	assert (env_id != NULL);
#endif

	if (list == NULL || lwc_getNumNode (list) <= 0) {
		return 0;
	}

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Escape the environment id string */
	esc_envid = sql_escape (sql, env_id);
	if (esc_envid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	req_len =	  schedwi_strlen (SQL_ENV_ADD_VARIABLE) 
			+ (	  schedwi_strlen (SQL_ENV_ADD_VARIABLE_VALUES)
				+ schedwi_strlen (esc_envid)
				+ 10)
					* lwc_getNumNode (list);
	lwc_rewindLL (list);
	while ((var = (char **) lwc_nextLL (list)) != NULL) {
		req_len +=	  schedwi_strlen (var[0]) * 2
				+ schedwi_strlen (var[1]) * 2;
	}

	req = (char *) malloc (req_len);
	if (req == NULL) {
		end_mysql ();
		free (esc_envid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	strcpy (req, SQL_ENV_ADD_VARIABLE);
	idx = schedwi_strlen (req);

	pos = 1;
	lwc_rewindLL (list);
	while ((var = (char **) lwc_nextLL (list)) != NULL) {

		key = sql_escape (sql, var[0]);
		value = sql_escape (sql, var[1]);
		if (key == NULL || value == NULL) {
			end_mysql ();
			free (esc_envid);
			free (req);
			if (key != NULL) {
				free (key);
			}
			if (value != NULL) {
				free (value);
			}
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return -1;
		}

		ret = snprintf (req + idx, req_len - idx,
				SQL_ENV_ADD_VARIABLE_VALUES,
				esc_envid, pos, key, value);
		if (ret >= req_len - idx || ret < 0) {
			end_mysql ();
			free (esc_envid);
			free (req);
			free (key);
			free (value);
			if (error_func != NULL) {
				error_func (user_data_error_func,
					_("Internal error: buffer too small"),
					0);
			}
			return -3;
		}
		free (key);
		free (value);

		idx = schedwi_strlen (req);
		req[idx++] = ',';
		pos++;
	}
	req[idx -1] = '\0';
	free (esc_envid);

	/* Insert the environment variables */
	sql_ret = mysql_query (sql, req);
	if (sql_ret != 0) {
		free (req);
		compose_error_message (_("REPLACE"), &err_msg);
		end_mysql ();
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, sql_ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}
	end_mysql ();
	free (req);
	return 0;
}


/*
 * Remove all the variables associated with the provided environment
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_env_del_variables (	const char *env_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (env_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_ENV_DELETE_VARIABLES,
				SQL_STRING, env_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the environment from the database
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_env_delete (	const char *env_id,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (env_id != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 2;
	}

	/* Delete the variables associated with this environment */
	ret = sql_env_del_variables (env_id, error_func, user_data_error_func);
	if (ret != 0) {
		end_mysql ();
		return ret;
	}

	/* Delete the environment */
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_ENV_DELETE,
				SQL_STRING, env_id,
				SQL_END);
	if (ret != 0) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	end_mysql ();
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
