/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sql_where_used.c -- Retrieve references of objects associated with others */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_where_used.h>

#define SQL_HOST_JOB_LIST "SELECT id,name,type FROM job_host_s,job_main_s WHERE job_host_s.workload_date=%d AND job_main_s.workload_date=%d AND job_id=id AND host_id=\"%s\"" 
#define SQL_HOST_FILE_LIST "SELECT id,name,type,filename FROM constraint_file_s,job_main_s WHERE constraint_file_s.workload_date=%d AND job_main_s.workload_date=%d AND job_id=id AND host_id=\"%s\"" 
#define SQL_ENV_JOB_LIST "SELECT id,name,type FROM job_environment_s,job_main_s WHERE job_environment_s.workload_date=%d AND job_main_s.workload_date=%d AND job_id=id AND env_id=\"%s\"" 
#define SQL_ENV_HOST_LIST "SELECT host_id,CONCAT(hostname,' (',portnum,')'),2 FROM host_environment_s,hosts WHERE workload_date=%d AND host_id=id AND env_id=\"%s\" ORDER BY hostname" 
#define SQL_CAL_JOB_LIST "SELECT id,name,type FROM job_main_s WHERE workload_date=%d AND cal_id IN "


/*
 * Create and return a new referenced_object_t structure
 *
 * Return:
 *   The new object (to be freed using destroy_referenced_object()) or
 *   NULL in case of memory allocation error
 */
static referenced_object_t *
new_referenced_object ()
{
	referenced_object_t *ptr;

	ptr = (referenced_object_t *) malloc (sizeof (referenced_object_t));
	if (ptr != NULL) {
		ptr->name = NULL;
		ptr->id = NULL;
		ptr->obj_type = JOBSET;
		ptr->more_data = NULL;
	}
	return ptr;
}


/*
 * Free a referenced_object_t object
 */
void
destroy_referenced_object (referenced_object_t *ptr)
{
	if (ptr != NULL) {
		if (ptr->name != NULL) {
			free (ptr->name);
		}
		if (ptr->id != NULL) {
			free (ptr->id);
		}
		if (ptr->more_data != NULL) {
			free (ptr->more_data);
		}
		free (ptr);
	}
}


/*
 * Retrieve from the database the object associated with a host or environment
 *
 * Return:
 *   0 --> No error (list contains the list of objects - this list must be
 *         freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
static int
parse_sql_result (	MYSQL *sql, const char *req, lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	MYSQL_RES *result;
	MYSQL_ROW row;
	lwc_LL *obj_list;
	referenced_object_t *obj;
	unsigned int num_fields, ret;

#if HAVE_ASSERT_H
	assert (sql != NULL && req != NULL && list != NULL);
#endif

	/* Get the object list */
	ret = mysql_query (sql, req);
	if (ret != 0) {
		compose_error_message (_("SELECT"), &err_msg);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Build the objects list */
	result = mysql_store_result (sql);
	if (result == NULL) {
		compose_error_message (_("SELECT"), &err_msg);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	num_fields = mysql_num_fields (result);

	obj_list = lwc_newLL ();
	if (obj_list == NULL) {
		mysql_free_result (result);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), ret);
		}
		return -1;
	}

	/* For each row (row[0]=id row[1]=name row[2]=type) */
	while ((row = mysql_fetch_row (result)) != NULL) {

		obj = new_referenced_object ();
		if (obj == NULL || lwc_addEndLL (obj_list, obj) != 0) {
			mysql_free_result (result);
			destroy_referenced_object (obj);
			lwc_delLL (obj_list, (void (*)(const void *))
						destroy_referenced_object);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						ret);
			}
			return -1;
		}

		obj->id = (char *) malloc (schedwi_strlen (row[0]) + 1);
		if (obj->id == NULL) {
			mysql_free_result (result);
			lwc_delLL (obj_list, (void (*)(const void *))
						destroy_referenced_object);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						ret);
			}
			return -1;
		}
		strcpy (obj->id, row[0]);

		obj->name = (char *) malloc (schedwi_strlen (row[1]) + 1);
		if (obj->name == NULL) {
			mysql_free_result (result);
			lwc_delLL (obj_list, (void (*)(const void *))
						destroy_referenced_object);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						ret);
			}
			return -1;
		}
		strcpy (obj->name, row[1]);
		switch (row[2][0]) {
			case '0':
				obj->obj_type = JOBSET;
				break;
			case '1':
				obj->obj_type = JOB;
				break;
			case '2':
				obj->obj_type = HOST;
				break;
			case '3':
				obj->obj_type = FILENAME;
				break;
		}
		if (num_fields > 3) {
			obj->more_data = (char *) malloc (
						schedwi_strlen (row[3]) + 1);
			if (obj->more_data == NULL) {
				mysql_free_result (result);
				lwc_delLL (obj_list, (void (*)(const void *))
						destroy_referenced_object);
				if (error_func != NULL) {
					error_func (	user_data_error_func,
						_("Memory allocation error"),
						ret);
				}
				return -1;
			}
			strcpy (obj->more_data, row[3]);
		}
	}
	mysql_free_result (result);
	*list = obj_list;
	return 0;
}


/*
 * Retrieve from the database the objects associated with a host, environment
 * or calendar
 *
 * Return:
 *   0 --> No error (list contains the list of objects - this list must be
 *         freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 *  -3 --> Internal error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
static int
parse_sql_result_job (	int workload_date, MYSQL *sql,
			const char *req, lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	int ret;
	unsigned int ret_get_path;
	lwc_LL *jobs;
	referenced_object_t *job;
	char *err_msg = NULL, *path;

#if HAVE_ASSERT_H
	assert (sql != NULL && req != NULL && list != NULL);
#endif

	ret = parse_sql_result (sql, req, &jobs, error_func,
				user_data_error_func);
	if (ret != 0) {
		return ret;
	}

	/* Add the hierarchy path to the job/jobset name */
	lwc_rewindLL (jobs);
	while ((job = (referenced_object_t *)lwc_nextLL (jobs)) != NULL) {

		/* Get the path */
		path = NULL;
		ret_get_path = get_job_path (	workload_date, job->id,
						&path, &err_msg);
		if (ret != 0) {
			lwc_delLL (jobs, (void (*)(const void *))
						destroy_referenced_object);
			if (error_func != NULL) {
				error_func (user_data_error_func, err_msg, 0);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			return (ret_get_path == 1) ? -1 : -2;
		}

		if (path != NULL) {
			if (job->name != NULL) {
				free (job->name);
			}
			job->name = path;
		}
	}
	*list = jobs;
	return 0;
}


/*
 * Retrieve from the database the job/jobsets associated with a host
 *
 * Return:
 *   0 --> No error (list contains the list of jobs/jobsets that use the
 *         provided host - this list must be freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 *  -3 --> Internal error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_host_job_list (	int workload_date, const char *host_id, lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	char *esc_hostid, *req;
	int ret;

#if HAVE_ASSERT_H
	assert (host_id != NULL && list != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Build the request */
	esc_hostid = sql_escape (sql, host_id);
	if (esc_hostid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	req = (char *) malloc (	  schedwi_strlen (SQL_HOST_JOB_LIST) +
				+ schedwi_strlen (esc_hostid) + 2 * 25);
	if (req == NULL) {
		end_mysql ();
		free (esc_hostid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	sprintf (	req, SQL_HOST_JOB_LIST,
			workload_date, workload_date, esc_hostid);
	free (esc_hostid);

	/* Built the list */
	ret = parse_sql_result_job (	workload_date, sql, req, list,
					error_func, user_data_error_func);
	free (req);
	end_mysql ();
	return ret;
}


/*
 * Retrieve from the database the job/jobsets associated with an environment
 *
 * Return:
 *   0 --> No error (list contains the list of jobs/jobsets that use the
 *         provided environment - this list must be freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 *  -3 --> Internal error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_env_job_list (	int workload_date, const char *env_id, lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	char *esc_envid, *req;
	int ret;

#if HAVE_ASSERT_H
	assert (env_id != NULL && list != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Build the request */
	esc_envid = sql_escape (sql, env_id);
	if (esc_envid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	req = (char *) malloc (	  schedwi_strlen (SQL_ENV_JOB_LIST)
				+ schedwi_strlen (esc_envid) + 2 * 25);
	if (req == NULL) {
		end_mysql ();
		free (esc_envid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	sprintf (	req, SQL_ENV_JOB_LIST,
			workload_date, workload_date, esc_envid);
	free (esc_envid);

	/* Built the list */
	ret = parse_sql_result_job (	workload_date, sql, req, list,
					error_func, user_data_error_func);
	free (req);
	end_mysql ();
	return ret;
}


/*
 * Retrieve from the database the hosts associated with an environment
 *
 * Return:
 *   0 --> No error (list contains the list of hosts that use the
 *         provided environment - this list must be freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_env_host_list (	int workload_date, const char *env_id, lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	char *esc_envid, *req;
	int ret;

#if HAVE_ASSERT_H
	assert (env_id != NULL && list != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Build the request */
	esc_envid = sql_escape (sql, env_id);
	if (esc_envid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	req = (char *) malloc (	  schedwi_strlen (SQL_ENV_HOST_LIST)
				+ schedwi_strlen (esc_envid) + 25);
	if (req == NULL) {
		end_mysql ();
		free (esc_envid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	sprintf (req, SQL_ENV_HOST_LIST, workload_date, esc_envid);
	free (esc_envid);

	/* Built the list */
	ret = parse_sql_result (sql, req, list,
				error_func, user_data_error_func);
	free (req);
	end_mysql ();
	return ret;
}


/*
 * Retrieve from the database the file names associated with a host
 *
 * Return:
 *   0 --> No error (list contains the list of jobs/jobsets that use the
 *         provided host - this list must be freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_host_file_list (	int workload_date, const char *host_id, lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	char *esc_hostid, *req, *str, *fmt_job, *fmt_jobset;
	int ret;
	lwc_LL *obj_list;
	referenced_object_t *obj;
	unsigned int str_len;

#if HAVE_ASSERT_H
	assert (host_id != NULL && list != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Build the request */
	esc_hostid = sql_escape (sql, host_id);
	if (esc_hostid == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	req = (char *) malloc (	  schedwi_strlen (SQL_HOST_FILE_LIST)
				+ schedwi_strlen (esc_hostid) + 2 * 25);
	if (req == NULL) {
		end_mysql ();
		free (esc_hostid);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	sprintf (	req, SQL_HOST_FILE_LIST,
			workload_date, workload_date, esc_hostid);
	free (esc_hostid);

	/* 
	 * Built the list of jobs containing the files associated
	 * with host_id
	 */
	ret = parse_sql_result_job (	workload_date, sql, req, &obj_list,
					error_func, user_data_error_func);
	free (req);
	end_mysql ();

	/* Convert the jobs to meaningful names */
	fmt_job = _("File %s defined in job %s");
	fmt_jobset = _("File %s defined in jobset %s");
	lwc_rewindLL (obj_list);
	while ((obj = (referenced_object_t *)lwc_nextLL (obj_list)) != NULL) {

		if (obj->obj_type == JOBSET) {
			str_len =	  schedwi_strlen (fmt_jobset)
					+ schedwi_strlen (obj->more_data)
					+ schedwi_strlen (obj->name);
		}
		else {
			str_len =	  schedwi_strlen (fmt_job)
					+ schedwi_strlen (obj->more_data)
					+ schedwi_strlen (obj->name);
		}
		str = (char *) malloc (str_len);
		if (str == NULL) {
			lwc_delLL (obj_list, (void (*)(const void *))
						destroy_referenced_object);
			return -1;
		}
		if (obj->obj_type == JOBSET) {
			sprintf (str, fmt_jobset, obj->more_data, obj->name);
		}
		else {
			sprintf (str, fmt_job, obj->more_data, obj->name);
		}
		free (obj->name);
		free (obj->more_data);
		obj->name = str;
		obj->more_data = NULL;
		obj->obj_type = FILENAME;
	}
	*list = obj_list;
	return ret;
}


/*
 * Retrieve from the database the job/jobsets associated with a calendar
 *
 * Return:
 *   0 --> No error (list contains the list of jobs/jobsets that use the
 *         provided calendars - this list must be freed by the caller using
 *          lwc_delLL (list, (void (*)(const void *))destroy_referenced_object);
 *         )
 *  -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *         with user_data_error_func as its first parameter and the error
 *         message as the second parameter)
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 *  -3 --> Internal error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_cal_job_list (	int workload_date,
			char **cal_id_array, unsigned int len_array,
			lwc_LL **list,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	char *req, *value, *temp_req;
	int ret;
	unsigned int req_len, i, idx;

#if HAVE_ASSERT_H
	assert (cal_id_array != NULL && list != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}

	/* Insert the workload date to the SQL request */
	temp_req = (char *) malloc (schedwi_strlen (SQL_CAL_JOB_LIST) + 25);
	if (temp_req == NULL) {
		end_mysql ();
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}
	sprintf (temp_req, SQL_CAL_JOB_LIST, workload_date);

	/* Build the request */
	req_len = schedwi_strlen (temp_req) + 3;
	for (i = 0; i < len_array && cal_id_array[i] != NULL; i++) {
		req_len += schedwi_strlen (cal_id_array[i]) * 2 + 3; 
	}

	req = (char *) malloc (req_len);
	if (req == NULL) {
		end_mysql ();
		free (temp_req);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	strcpy (req, temp_req);
	free (temp_req);
	idx = schedwi_strlen (req);
	req[idx++] = '(';
	for (i = 0; i < len_array && cal_id_array[i] != NULL; i++) {
		value = sql_escape (sql, cal_id_array[i]);
		if (value == NULL) {
			end_mysql ();
			free (req);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return -1;
		}
		req[idx++] = '"';
		strcpy (req + idx, value);
		idx += schedwi_strlen (value);
		req[idx++] = '"';
		req[idx++] = ',';
		free (value);
	}
	req[idx - 1] = ')';
	req[idx] = '\0';

	/* Built the list */
	ret = parse_sql_result_job (	workload_date, sql, req, list,
					error_func, user_data_error_func);
	free (req);
	end_mysql ();
	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
