/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* net_module_killjob.c -- Module to stop a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lwc_log.h>
#include <child_mgnt.h>
#include <lib_functions.h>
#include <net_module_killjob.h>


/*
 * Parse the server request (in buffer) and stop the specified jobs
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an error
 *         message)
 *  -2 --> Error kill a job (lwc_writeLog() is used to display an error
 *         message)
 */
int
kill_jobs (net_id *sock, char *buffer, size_t size)
{
	char *err_msg;
	char *buff_cpy, *str;
	int ret, nb_errors;

#if HAVE_ASSERT_H
	assert (buffer != NULL);
#endif

	/* Copy the input buffer to add a '\0' at the end (for strtok below) */
	buff_cpy = (char *) malloc (size + 1);
	if (buff_cpy == NULL) {
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}
	strncpy (buff_cpy, buffer, size);
	buff_cpy [size] = '\0';

	/* Parse the request */
	nb_errors = 0;
	for (	str = strtok (buff_cpy, "\n");
		str != NULL;
		str = strtok (NULL, "\n"))
	{
		/* Skip the spaces at the begining of the line */
		while (*str != '\0' && isspace (*str) != 0) {
			str++;
		}
		if (*str != '\0') {
			ret = terminate_job (str);
			if (ret != 0) {
				nb_errors++;
				if (ret == 1) {
					lwc_writeLog (	LOG_WARNING,
					_("Job %s: cannot stop (not found)"),
							str);
				}
				else {
					lwc_writeLog (	LOG_WARNING,
					_("Job %s: cannot stop: %s"),
							str, strerror (errno));
				}
			}
		}
	}
	free (buff_cpy);

	if (nb_errors > 0) {
		err_msg = _("Failed to stop the job(s)");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		return -2;
	}
	else {
		net_write (sock, "0", 1);
		return 0;
	}
}

/*-----------------============== End Of File ==============-----------------*/
