/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU Library General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * srvtest_run.c -- Test program for the Schedwi agent
 *
 * Parameters:
 *     1) The absolute path of the folder containing the `foo' test command
 *        to run
 *     2) The current username
 *
 * Output:
 *     stdout) The result of the command execution send by schedwiclnt
 *     stderr) The result of the submission of the command to schedwiclnt
 *
 * Return:
 *   0 on success or
 *   an other return code for error
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifndef HAVE_RECV
#define recv(a,b,c,d) read(a,b,c)
#endif

#ifndef HAVE_SEND
#define send(a,b,c,d) write(a,b,c)
#endif

#ifndef PF_INET
#define PF_INET AF_INET
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif


/*
 * memset - fill memory with a constant byte
 */
void *
schedwi_memset (void *s, int c, unsigned int n)
{
        char *tmp = s;

        if (s != 0) {
                while (n/10 != 0) {
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        *(tmp++) = c;
                        n -= 10;
                }
                while (n > 0) {
                        *(tmp++) = c;
                        n--;
                }
        }
        return s;
}


/*
 * strlen - calculate the length of a string
 */
size_t
schedwi_strlen (const char *s)
{
	size_t i;

	if (s == NULL) {
		return 0;
	}
	for (i = 0; s[i] != '\0'; i++);
	return i;
}


/*
 * strncmp -- compare two strings
 */
int
schedwi_strncmp (const char *s1, const char *s2, unsigned int n)
{
	unsigned int i;

	if (s1 == 0) {
		if (s2 == 0) {
			return 0;
		}
		return -1;
	}
	if (s2 == 0) {
		return 1;
	}

	for (	i = 0;
		   i < n
		&& s1[i] != '\0' && s2[i] != '\0'
		&& s1[i] == s2[i];
		i++);
	return (i == n) ? 0 : s1[i] - s2[i];
}


/*
 * strnstr - locate a substring in the first n characters
 */
char *
schedwi_strnstr (char *haystack, const char *needle, unsigned int n)
{
	unsigned int len;

	if (needle == 0 || *needle == '\0' || haystack == 0) {
		return 0;
	}

	len = schedwi_strlen (needle);

	while (len <= n) {
		if (schedwi_strncmp (needle, haystack, len) == 0) {
			return haystack;
		}
		n--;
		haystack++;
	}

	return 0;
}


/*
 * atoi - convert a string to an integer (very simplified version)
 */
int
schedwi_atoi (const char *s)
{
	size_t i;
	int val;

	if (s == NULL) {
		return 0;
	}

	while (*s != '\0' && (*s == ' ' || *s == '\t')) {
		s++;
	}

	val = 0;
	for (i = 0; s[i] != '\0' && s[i] >= '0' && s[i] <= '9'; i++) {
		val *= 10;
		val += s[i] - '0';
	}
	return val;
}


/*
 * Send a job request to the client
 */
int
start_job (const char *path_to_cmd, const char *user_name)
{
	int sock;
	struct sockaddr_in sad;
	char buff[4096];
	char result;
	ssize_t nb_read;
	struct hostent *he;
#if HAVE_GETENV
	char *env_str;
#endif

	/* Get the address of the local host */
	he = gethostbyname ("127.0.0.1");
	if (he == NULL) {
		return 8;
	}

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		return 9;
	}

	/* Compose the remote address */
	schedwi_memset (&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = *((unsigned long *)(he->h_addr_list[0]));
	sad.sin_port = htons (schedwi_atoi (SCHEDWI_DEFAULT_AGTPORT));

	/* Connect to the address */
	if (connect (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		closesocket (sock);
		return 10;
	}

	/* Build and send the header */
	schedwi_memset (buff, 0, 25);
	buff[0] = 'r';
	buff[1] = 'u';
	buff[2] = 'n';
	buff[3] = 'j';
	buff[4] = 'o';
	buff[5] = 'b';
	buff[6] = '-';
	buff[7] = '0';
	buff[8] = '.';
	buff[9] = '1';
	send (sock, buff, 25, 0);

	/* Write the request */
	send (sock, "JID:20070324_23\n", 16, 0);

	send (sock, "CMD:", 4, 0);
	send (sock, path_to_cmd, schedwi_strlen (path_to_cmd), 0);
	send (sock, "/", 1, 0);
	send (sock, "${CMD}\n", 7, 0);

	send (sock, "USR:", 4, 0);
	send (sock, user_name, schedwi_strlen (user_name), 0);
	send (sock, "\n", 1, 0);

	send (sock, "OUT:", 4, 0);
	send (sock, path_to_cmd, schedwi_strlen (path_to_cmd), 0);
	send (sock, "/foo.out\n", 9, 0);

	send (sock, "ERR:", 4, 0);
	send (sock, path_to_cmd, schedwi_strlen (path_to_cmd), 0);
	send (sock, "/foo.err\n", 9, 0);

	send (sock, "LDE:0\n", 6, 0);

	send (sock, "ARG:2:arg2\n", 11, 0);
	send (sock, "ARG:1:arg1\n", 11, 0);

	send (sock, "ENV:1:A:1\n", 10, 0); 
	send (sock, "ENV:4:CMD:$C\n", 13, 0); 
	send (sock, "ENV:2:B:2\n", 10, 0); 
	send (sock, "ENV:3:C:foo\n", 12, 0); 

	/*
	 * Load and set the library environment variables that may be required
	 * by the command to run by the client (as schedwiclnt clear the
	 * environment before running a command)
	 */
#if HAVE_GETENV
	/* Linux, Solaris, HP-UX, Tru64, SGI */
	env_str = getenv ("LD_LIBRARY_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:10:LD_LIBRARY_PATH:", 23, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* Solaris */
	env_str = getenv ("LD_LIBRARY_PATH_64");
	if (env_str != NULL) {
		send (sock, "ENV:11:LD_LIBRARY_PATH_64:", 26, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* HP-UX */
	env_str = getenv ("SHLIB_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:12:SHLIB_PATH:", 18, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* AIX, OS/390 */
	env_str = getenv ("LIBPATH");
	if (env_str != NULL) {
		send (sock, "ENV:13:LIBPATH:", 15, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* SGI */
	env_str = getenv ("LD_LIBRARYN32_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:14:LD_LIBRARYN32_PATH:", 26, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	env_str = getenv ("LD_LIBRARY64_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:15:LD_LIBRARY64_PATH:", 25, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* Windows */
	env_str = getenv ("PATH");
	if (env_str != NULL) {
		send (sock, "ENV:16:PATH:", 12, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* MacOS X */
	env_str = getenv ("DYLD_LIBRARY_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:17:DYLD_LIBRARY_PATH:", 25, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	env_str = getenv ("DYLD_FALLBACK_LIBRARY_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:18:DYLD_FALLBACK_LIBRARY_PATH:", 34, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	/* BeOS */
	env_str = getenv ("LIBRARY_PATH");
	if (env_str != NULL) {
		send (sock, "ENV:19:LIBRARY_PATH:", 20, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
	env_str = getenv ("BELIBRARIES");
	if (env_str != NULL) {
		send (sock, "ENV:20:BELIBRARIES:", 19, 0); 
		send (sock, env_str, schedwi_strlen (env_str), 0); 
		send (sock, "\n", 1, 0);
	}
#endif

	send (sock, "\nbYe", 4, 0);

	/* Read the reply and dump it to stderr */
	result = 0;
	do{
		nb_read = recv (sock, buff, 4096, 0);
		if (nb_read == 0) {
			break;
		}
		if (nb_read < 0) {
			closesocket (sock);
			return 11;
		}
		if (result == 0) {
			result = buff[0];
		}
		write (2, buff, nb_read);
	} while (1);

	closesocket (sock);

	if (result != '0') {
		return 12;
	}
	return 0;
}


/*
 * Main function
 */
int
main (int argc, char **argv)
{
	int sock, s, opt, ret;
	struct sockaddr_in sad;
	ssize_t nb_read;
	char buff[4096], *path_to_cmd, *user_name;

	if (argc != 3) {
		return 1;
	}
	path_to_cmd = argv[1];
	user_name = argv[2];

	/*
	 * Create the server socket (for the reply)
	 */

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		return 2;
	}

	/* Set the REUSEADDR flag */
	opt = -1;
	setsockopt (sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof (opt));

	/* Compose the address */
	schedwi_memset (&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = htonl (INADDR_ANY);
	sad.sin_port = htons (schedwi_atoi (SCHEDWI_DEFAULT_SRVPORT)); 

	/* Bind to the address */
	if (bind (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		closesocket (sock);
		return 3;
	}

	/* Prepare to accept requests */
	if (listen (sock, 5) != 0) {
		closesocket (sock);
		return 4;
	}


	/*
	 * Start the job
	 */
	ret = start_job (path_to_cmd, user_name);
	if (ret != 0) {
		closesocket (sock);
		return ret;
	}


	/*
	 * Wait for the reply 
	 */
	s = accept (sock, NULL, NULL);
	if (s < 0) {
		closesocket (sock);
		return 6;
	}

	do {
		nb_read = recv (s, buff, 4096, 0);
		if (nb_read == 0) {
			break;
		}
		if (nb_read < 0) {
			closesocket (s);
			closesocket (sock);
			return 7;
		}

		/*
	 	 * Write the reply to stdout
		 */
		write (1, buff, nb_read);
		if (schedwi_strnstr (buff, "\nbYe", nb_read) != NULL) {
			break;
		}
	} while (1);

	send (s, "0", 1, 0);
	closesocket (s);
	closesocket (sock);

	return 0;
}

/*------------------------======= End Of File =======------------------------*/
