/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* env_list_cb.c - Environment list window */

#include <schedwi.h>

#include <schedwi_interface.h>

#include <sql_env.h>
#include <references_cb.h>
#include <sql_where_used.h>
#include <message_windows.h>
#include <cursor.h>
#include <env_window_cb.h>
#include <main_cb.h>
#include <workload.h>
#include <env_list_cb.h>


/*
 * TREE VIEW WIDGET MANAGEMENT
 */

/*
 * Callback when the selection changes
 * The buttons are made sensitive if an item is selected
 */
static void
tree_selection_changed_cb (GtkTreeSelection *selection, gpointer data)
{
	GtkWidget *widget = data;
	GtkWidget *edit, *where;
	GtkTreeModel *model;

	edit = lookup_widget (widget, "button_env_edit");
	where = lookup_widget (widget, "button_env_where_used");

	if (gtk_tree_selection_get_selected (selection, &model, NULL) == TRUE)
	{
		gtk_widget_set_sensitive (edit, TRUE);
		gtk_widget_set_sensitive (where, TRUE);
	}
	else {
		gtk_widget_set_sensitive (edit, FALSE);
		gtk_widget_set_sensitive (where, FALSE);
	}
}


/*
 * Double-click on a row
 */
static void
view_onRowActivated (	GtkTreeView *treeview,
			GtkTreePath *path,
			GtkTreeViewColumn *col,
			gpointer userdata)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *env_id;
	int workload_date;

	model = gtk_tree_view_get_model (treeview);
	if (gtk_tree_model_get_iter (model, &iter, path) == TRUE) {
		/* Retrieve the workload date */
		workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (lookup_widget (GTK_WIDGET (treeview),
							"dialog_env_list")),
				"workload_date"));
		cursor_busy (GTK_WIDGET (treeview));
		gtk_tree_model_get (model, &iter, 0, &env_id, -1);
		new_dialog_env (treeview, env_id, workload_date);
		g_free (env_id);
		cursor_normal (GTK_WIDGET (treeview));
	}
}


/*
 * Add the column and cell to a GtkTreeView
 */
static void
build_view (GtkTreeView *view)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkTreeSelection *select;

	/* Environment name column */
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new_with_attributes (_("Environment"),
							renderer,
							"text", 1,
							NULL);
	gtk_tree_view_append_column (view, column);

	/* Selection callbacks */
	select = gtk_tree_view_get_selection (view);
	gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);
	g_signal_connect (	G_OBJECT (select), "changed",
				G_CALLBACK (tree_selection_changed_cb), view);
	g_signal_connect(	G_OBJECT (view), "row-activated",
				G_CALLBACK (view_onRowActivated), NULL);
}


/* Structure used with sql_env_list() to exchange data between functions */ 
struct refresh_cb_user_data {
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	gchar *selected_id;
};

/*
 * Append an environment row to a GtkListStore model
 * Always return 0
 */
static int
list_env_add_row (void *ptr, const char *id, const char *name)
{
	struct refresh_cb_user_data *data = ptr;
	GtkTreeIter iter;

	gtk_list_store_append (GTK_LIST_STORE (data->model), &iter);
	gtk_list_store_set (	GTK_LIST_STORE (data->model), &iter,
				0, id, 1, name, -1);
	if (data->selected_id != NULL && strcmp (data->selected_id, id) == 0) {
		gtk_tree_selection_select_iter  (data->selection, &iter);
		g_free (data->selected_id);
		data->selected_id = NULL;
	}
	return 0;
}


/*
 * Initialize the list
 *
 * Return:
 *       0 --> No error
 *   other --> SQL error (a message has been displayed to the user)
 */
static int
dialog_env_list_init (GtkWidget *dialog_envs, int workload_date)
{
	GtkWidget *view;
	GtkListStore *store;
	struct refresh_cb_user_data data;
	int ret;

	gtk_window_set_type_hint (	GTK_WINDOW (dialog_envs),
					GDK_WINDOW_TYPE_HINT_NORMAL);

	/* Save the workload date in the widget */
	g_object_set_data (     G_OBJECT (dialog_envs),
				"workload_date",
				GINT_TO_POINTER (workload_date));

	view = lookup_widget (dialog_envs, "treeview_environments");

	/* Create and fill the GtkListStore model */
	store = gtk_list_store_new (2,
				G_TYPE_STRING,	/* Env id (not displayed) */
				G_TYPE_STRING);	/* Environment name */
	data.selected_id = NULL;
	data.model = GTK_TREE_MODEL (store);

	/* Fill the GtkListStore model */
	ret = sql_env_list (	workload_date, NULL, list_env_add_row, &data,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
				_("Database error"));
	if (ret < 0) {
		g_object_unref (store);
		return ret;
	}

	/* Associate the model with the view */
	gtk_tree_view_set_model (GTK_TREE_VIEW (view), GTK_TREE_MODEL (store));
	g_object_unref (store);
	gtk_tree_view_set_search_column (GTK_TREE_VIEW (view), 1);

	/* Add the columns, cells and selection callbacks */
	build_view (GTK_TREE_VIEW (view));

	return 0;
}


/*
 * Create a dialog environment list window
 *
 * Return:
 *   The new GtkWidget (`show'ed by this function) or
 *   NULL in case of error (an error message has been displayed for the user)
 */
GtkWidget *
new_dialog_env_list (	void (*hide_cb)(gpointer), gpointer user_data,
			int workload_date)
{
	GtkWidget *widget;

	widget = create_dialog_env_list ();
	if (dialog_env_list_init (widget, workload_date) != 0) {
		gtk_widget_destroy (widget);
		return NULL;
	}
	else {
		g_object_set_data (G_OBJECT (widget), "hide_cb",
						hide_cb);
		g_object_set_data (G_OBJECT (widget), "hide_cb_data",
						user_data);

		gtk_widget_show (widget);
		return widget;
	}
}


/*
 * Set or change the workload date
 */
void
dialog_env_list_set_workload_date (	GtkWidget *dialog_env_list,
					int workload_date)
{
	g_object_set_data (     G_OBJECT (dialog_env_list),
				"workload_date",
				GINT_TO_POINTER (workload_date));
	env_list_refresh_widget (dialog_env_list);
}


/*
 * delete-event callback (when the user closes the window)
 */
gboolean
env_list_delete_event (GtkWidget *widget)
{
	GtkWidget *window;
	void (*hide_cb)(gpointer);
	gpointer user_data;

	/*
	 * Call the hide_cb function provided when the
	 * new_dialog_calendar_list() was called
	 */
	window = lookup_widget (widget, "dialog_env_list");
	hide_cb = (void (*)(gpointer))g_object_get_data (
							G_OBJECT(window),
							"hide_cb");
	user_data = g_object_get_data (G_OBJECT (window), "hide_cb_data");
	if (hide_cb != NULL) {
		hide_cb (user_data);
	}
	else {
		/* Hide the Environment list window */
		gtk_widget_hide (window);
	}
	/*
	 * Return TRUE to stop other handlers from being invoked for
	 * the event. Otherwise, the window will be destroyed
	 */
	return TRUE;
}


/*
 * CLOSE button callback
 */
void
env_list_close_clicked (GtkButton *button)
{
	GtkWidget *window;
	void (*hide_cb)(gpointer);
	gpointer user_data;

	/*
	 * Call the hide_cb function provided when the
	 * new_dialog_calendar_list() was called
	 */
	window = lookup_widget (GTK_WIDGET (button), "dialog_env_list");
	hide_cb = (void (*)(gpointer))g_object_get_data (
							G_OBJECT(window),
							"hide_cb");
	user_data = g_object_get_data (G_OBJECT (window), "hide_cb_data");
	if (hide_cb != NULL) {
		hide_cb (user_data);
	}
	else {
		/* Hide the Calendar list window */
		gtk_widget_hide (window);
	}
}


/*
 * Refresh the environment list
 */
void
env_list_refresh (GtkTreeView *view)
{
	struct refresh_cb_user_data data;
	GtkTreeIter iter;
	int workload_date;

	cursor_busy (GTK_WIDGET (view));

	/* Retrieve the current selected object */
	data.selection = gtk_tree_view_get_selection (view);
	if (gtk_tree_selection_get_selected (	data.selection,
						&(data.model), &iter) == TRUE)
	{
		gtk_tree_model_get (	data.model, &iter,
					0, &(data.selected_id), -1);
	}
	else {
		data.selected_id = NULL;
	}

	/* Clear the model */
	gtk_list_store_clear (GTK_LIST_STORE (data.model));

	/* Retrieve the workload date */
	workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (lookup_widget (GTK_WIDGET (view),
							"dialog_env_list")),
				"workload_date"));

	/* Re-read the database and fill the model */
	sql_env_list (	workload_date, NULL, list_env_add_row, &data,
			(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
			_("Database error"));
	g_free (data.selected_id);
	cursor_normal (GTK_WIDGET (view));
}


/*
 * Callback of the `Refresh' button
 */
void
env_list_refresh_clicked (GtkButton *button)
{
	GtkWidget *view;

	view = lookup_widget (GTK_WIDGET (button), "treeview_environments");
	env_list_refresh (GTK_TREE_VIEW (view));
}


/*
 * Refresh the list
 */
void
env_list_refresh_widget (GtkWidget *widget)
{
	GtkWidget *view;

	view = lookup_widget (widget, "treeview_environments");
	env_list_refresh (GTK_TREE_VIEW (view));
}


/*
 * Callback for the `Edit' button
 */
void
env_edit_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gchar *id;
	int workload_date;

	view = lookup_widget (GTK_WIDGET (button), "treeview_environments");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	if (gtk_tree_selection_get_selected (selection, &model, &iter) == TRUE)
	{
		/* Retrieve the workload date */
		workload_date = (int) GPOINTER_TO_INT (g_object_get_data (
				G_OBJECT (lookup_widget (view,
							"dialog_env_list")),
				"workload_date"));
		cursor_busy (view);
		gtk_tree_model_get (model, &iter, 0, &id, -1);
		new_dialog_env (GTK_TREE_VIEW (view), id, workload_date);
		g_free (id);
		cursor_normal (view);
	}
}


/*
 * Callback for the `Edit' item in the popup menu
 */
void
env_edit_menu (GtkMenuItem *item)
{
	GtkWidget *menu, *view;
	GtkButton *bt;

	menu = gtk_widget_get_ancestor (GTK_WIDGET (item), GTK_TYPE_MENU);
	view = (GtkWidget *)g_object_get_data (G_OBJECT (menu), "tree_view");
	bt = (GtkButton *)lookup_widget (view, "button_env_edit");
	env_edit_clicked (bt);
}


/*
 * Callback for the `References' button
 */
void
env_where_used_clicked (GtkButton *button)
{
	GtkWidget *view;
	GtkTreeModel *model;
	GtkTreeSelection *selection;
	GtkTreeIter iter;
	gchar *id, *name;
	int workload_date;

	view = lookup_widget (GTK_WIDGET (button), "treeview_environments");
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (view));

	if (gtk_tree_selection_get_selected (selection, &model, &iter) == TRUE)
	{
		workload_date = (int) workload_get_workload (application_main);
		cursor_busy (GTK_WIDGET (button));
		gtk_tree_model_get (model, &iter, 0, &id, 1, &name, -1);
		new_dialog_env_references (id, name, workload_date);
		g_free (id);
		g_free (name);
		cursor_normal (GTK_WIDGET (button));
	}
}


/*
 * Callback for the `References' item in the popup menu
 */
void
env_where_used_menu (GtkMenuItem *item)
{
	GtkWidget *menu, *view;
	GtkButton *bt;

	menu = gtk_widget_get_ancestor (GTK_WIDGET (item), GTK_TYPE_MENU);
	view = (GtkWidget *)g_object_get_data (G_OBJECT (menu), "tree_view");
	bt = (GtkButton *)lookup_widget (view, "button_env_where_used");
	env_where_used_clicked (bt);
}


/*
 * POPUP MENU
 */

struct view_and_path {
	GtkTreeView *view;
	GtkTreePath *path;
};


/*
 * Menu positioning function used when the popup menu is displayed by
 * Shift+F10
 */
static void
menu_position (	GtkMenu *menu, gint *x, gint *y,
		gboolean *push_in, gpointer user_data)
{
	struct view_and_path *ptr = user_data;
	GdkRectangle rect;

	gtk_tree_view_get_cell_area (ptr->view, ptr->path, NULL, &rect);
	gdk_window_get_origin ((GTK_WIDGET (ptr->view))->window, x, y);
	*x += 10;
	*y += rect.y + rect.height + 1;
	*push_in = TRUE;
}


/*
 * Popup the menu
 */
static void
view_popup_menu_env (	GdkEventButton *event,
			GtkTreeView *view, GtkTreePath *path)
{
	GtkWidget *menu;
	struct view_and_path ptr;

	menu = create_menu_env_list ();
	gtk_widget_show_all (menu);

	/*
	 * Store the GtkTreeView widget in the menu.  This will be
	 * used by the callbacks to access the view
	 */
	g_object_set_data (G_OBJECT (menu), "tree_view", view);

	if (view != NULL && path != NULL) {
		ptr.view = view;
		ptr.path = path;
	}
	/* Note: event can be NULL here when called from view_onPopupMenu;
	 *  gdk_event_get_time() accepts a NULL argument
	 */
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
			(view != NULL && path != NULL)? menu_position: NULL,
			&ptr,
			(event != NULL) ? event->button : 0,
			gdk_event_get_time((GdkEvent*)event));
}


/*
 * Callback for the "button-press-event" event.  The clicked item
 * is retrieved and selected
 *
 * Return:
 *    TRUE --> The event has been taken into account
 *   FALSE --> Event no handled (to propagate the event further)
 */
gboolean
env_on_treeview_cal_button_press_event (	GtkWidget *treeview,
						GdkEventButton *event)
{
        GtkTreeSelection *selection;
	GtkTreePath *path;
	

	/* Single click with the right mouse button? */
	if (event->type != GDK_BUTTON_PRESS  ||  event->button != 3) {
		return FALSE;
	}

	/* Get the path of the item that was clicked */
	if (gtk_tree_view_get_path_at_pos (	GTK_TREE_VIEW (treeview),
						(gint) event->x,
						(gint) event->y,
						&path,
						NULL, NULL, NULL) == FALSE)
	{
		return FALSE;
	}

	/* Select this item */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	gtk_tree_selection_select_path (selection, path);
	gtk_tree_path_free (path);

	/* Popup the menu */
	view_popup_menu_env (event, GTK_TREE_VIEW (treeview), NULL);
	return TRUE;
}


/*
 * Callback for the "popup-menu" event (Shift-F10).
 *
 * Return:
 *    TRUE --> A popup menu has been displayed
 *   FALSE --> No popup menu displayed because no item is selected
 */
gboolean
env_on_treeview_cal_popup_menu (GtkWidget *treeview)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreePath *path;

	/* Retrieve the selected item */
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
	if (gtk_tree_selection_get_selected (selection, &model, &iter) == FALSE)
	{
		return FALSE;
	}

	/* Popup the menu */
	path = gtk_tree_model_get_path (model, &iter);
	view_popup_menu_env (NULL, GTK_TREE_VIEW (treeview), path);
	gtk_tree_path_free (path);
	return TRUE;
}

/*------------------------======= End Of File =======------------------------*/
