/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwi_jobtree.c -- Function to manage the job/jobset tree */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#include <message_windows.h>
#include <sql_job.h>
#include <sql_status.h>
#include <sql_stat.h>
#include <gcalendar_list.h>
#include <gschedwi_jobtree.h>


/*
 * Compute the currently percentage duration of the provided job/jobset
 */
static void
schedwi_jobtree_percentage (schedwi_jobtree_node_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->average_duration == 0) {
			ptr->average_duration = 1;
		}
		switch (ptr->status) {
			case JOB_STATUS_STATE_WAITING:
				ptr->run_pct = 0;
				break;
			case JOB_STATUS_STATE_COMPLETED:
				ptr->run_pct = 100;
				break;
			case JOB_STATUS_STATE_FAILED:
				ptr->run_pct =	  100 * ptr->last_run_duration
						/ ptr->average_duration;
				break;
			case JOB_STATUS_STATE_RUNNING:
				ptr->run_pct =	  100 * (schedwi_time_now ()
							- ptr->start_time)
						/ ptr->average_duration;
				break;
			default:
				ptr->run_pct = 0;
				break;
		}
		if (ptr->run_pct > 100) {
			ptr->run_pct = 100;
		}
	}
}

/*
 * Create a new node
 *
 * Return:
 *   The new node (to be freed by the caller by g_free()) or
 *   NULL in case of error (an error message has been displayed)
 */
static schedwi_jobtree_node_ptr
new_schedwi_jobtree_node (	const char *id,
				int workload_date,
				const char *node_type,
				schedwi_date job_date,
				const char *job_time,
				lwc_LL *hierarchy_list)
{
	schedwi_jobtree_node_ptr ptr;
	short int time_limit;
	int current_status;
	long int start_time;
	char *status_message = NULL;

	g_assert (	   id != NULL && node_type != NULL
			&& job_time != NULL && hierarchy_list != NULL);

	/* Allocate memory for the new node */
	ptr = g_new (schedwi_jobtree_node, 1);

	/* Retrieve parameters from the database */
	if (sql_job_get_start_limit (	workload_date, hierarchy_list,
					&time_limit,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error")) != 0)
	{
		g_free (ptr);
		return NULL;
	}
	ptr->start_limit = time_limit * 60; /* The limit in DB is in min */

	if (sql_job_get_max_duration (	workload_date, hierarchy_list,
					&time_limit,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error")) != 0)
	{
		g_free (ptr);
		return NULL;
	}
	ptr->max_duration = time_limit * 60; /* The limit in DB is in min */

	if (sql_job_get_retries (	workload_date, hierarchy_list,
					&(ptr->retries),
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error")) != 0)
	{
		free (ptr);
		return NULL;
	}

	if (sql_job_get_retries_interval (workload_date, hierarchy_list,
					&time_limit,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error")) != 0)
	{
		free (ptr);
		return NULL;
	}
	ptr->retries_interval = time_limit * 60; /* The value in DB is in min*/

 	if (sql_status_get (	schedwi_date_to_int (job_date),
				id, &current_status, &start_time,
				&(ptr->retry_num),
				&status_message,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error")) != 0)
	{
		g_free (ptr);
		return NULL;
	}

	if (sql_stat_get_duration (id, &(ptr->average_duration),
				&(ptr->last_run_duration),
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error")) != 0)
	{
		if (status_message != NULL) {
			free (status_message);
		}
		g_free (ptr);
		return NULL;
	}

	/* Fill the structure */
	ptr->status = (current_status == 0)
				? JOB_STATUS_STATE_WAITING
				: job_status_state_int2status (current_status);
	ptr->parent     = NULL;
	ptr->children   = NULL;
	ptr->next       = NULL;
	ptr->prev       = NULL;
	ptr->start_time = (schedwi_time) start_time;
	ptr->status_message = status_message;
	ptr->node_type  = (*node_type == '0') ? 0 : 1;
	ptr->id	        = strtoull (id, NULL, 0);
	ptr->run_time  = schedwi_time_convert_from_string (job_date, job_time);
	schedwi_jobtree_percentage (ptr);
	return ptr;
}


/*
 * Recursively free a node tree
 */
static void
free_jobtree_all_nodes (schedwi_jobtree_node_ptr ptr)
{
	if (ptr != NULL) {
		free_jobtree_all_nodes (ptr->children);
		free_jobtree_all_nodes (ptr->next);
		if (ptr->status_message != NULL) {
			free (ptr->status_message);
		}
		g_free (ptr);
	}
}


/*
 * Comparison function used by qsort() to sort the job/jobset list by ID
 */
static int
compar_job_id (const void *a, const void *b)
{
	jobtree_jobs_ptr ptr1 = (jobtree_jobs_ptr) a;
	jobtree_jobs_ptr ptr2 = (jobtree_jobs_ptr) b;

	return ptr1->id - ptr2->id;
}


/*
 * Comparison function used by bsearch() to search a job/jobset list by ID
 */
static int
search_job_id (const void *key, const void *str)
{
	unsigned long long int *id = (unsigned long long int *)key;
	jobtree_jobs_ptr ptr = (jobtree_jobs_ptr) str;

	return *id - ptr->id;
}


/*
 * Retrieve a job/jobset node from the tree given its ID
 *
 * Return:
 *   The node or
 *   NULL if not found
 */
schedwi_jobtree_node_ptr
jobtree_find (jobtree_ptr tree, unsigned long long int id)
{
	jobtree_jobs_ptr ptr;

	if (tree == NULL) {
		return NULL;
	}
	ptr =  bsearch (&id, tree->jobs, tree->nb_jobs,
			sizeof (jobtree_jobs), search_job_id);
	if (ptr == NULL) {
		return NULL;
	}
	return ptr->node;
}


/*
 * Recursively walk through the tree and fill the provided index array
 */
static void
build_index_recur (   schedwi_jobtree_node_ptr top,
			unsigned int *idx_list,
			jobtree_jobs_ptr list)
{
	while (top != NULL) {
		list[*idx_list].id = top->id;
		list[*idx_list].node = top;
		(*idx_list)++;
		build_index_recur (top->children, idx_list, list);
		top = top->next;
	}
}


/*
 * Build the index (ordered array) of all the jobs/jobsets in the tree
 * The last element of the array has an ID of 0 and node is NULL.
 * The array has been allocated by this function and must be freed by the
 * caller by g_free()
 */
static void
build_index (	schedwi_jobtree_node_ptr top, unsigned int nb,
		jobtree_jobs_ptr *list)
{
	jobtree_jobs_ptr ptr_list;
	unsigned int idx_list;

	/*
	 * Allocate memory for the array
	 */
	ptr_list = g_new (jobtree_jobs, nb + 1);

	/*
	 * Fill the array
	 */
	idx_list = 0;
	build_index_recur (top, &idx_list, ptr_list);
	ptr_list[idx_list].id = 0;
	ptr_list[idx_list].node = NULL;

	/*
	 * Sort the array by ID
	 */
	qsort (ptr_list, idx_list, sizeof (jobtree_jobs), compar_job_id);

	*list = ptr_list;
}


/*
 * Recursively build the job/jobset tree
 *
 * Return:
 *   0 --> No error.  children contains the tree and must be freed by 
 *         free_jobtree_all_nodes(). nb_jobs contains the number of
 *         jobs/jobsets in the returned tree.
 *  -1 --> Memory allocation error.  An error message has been displayed.
 *  -2 --> Database error.  An error message has been displayed.
 */
static int
build_jobtree_recur (	schedwi_jobtree_node_ptr jobset,
			const char *jobset_id,
			schedwi_date jobset_date,
			int workload_date,
			const char *jobset_time,
			calendar_list_t_ptr calendars,
			lwc_LL *hierarchy_list,
			schedwi_jobtree_node_ptr *children,
			unsigned int *nb_jobs)
{
	lwc_LL *job_list;
	char **row;
	unsigned int sql_ret;
	int ret;
	char *calendars_for_today, *calendars_for_tomorrow;
	schedwi_date date_tomorrow;
	schedwi_jobtree_node_ptr ptr, nxt;

	g_assert (	   jobset_id != NULL && calendars != NULL
			&& hierarchy_list != NULL && jobset_time != NULL
			&& children != NULL && nb_jobs != NULL);

	ptr = nxt = NULL;



	/*
	 * Build the tree with the jobs and jobsets that must run today
	 */

	/* Get the calendar ID list for today */
	get_calendar_list_for_the_day (	calendars, jobset_date,
					&calendars_for_today);

	/* Get the job/jobset list */
	sql_ret = sql_job_today (	workload_date,
					&job_list,
					jobset_id, calendars_for_today,
					jobset_time,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error"));
	g_free (calendars_for_today);
	if (sql_ret != 0) {
		return (sql_ret == 1) ? -1 : -2;
	}

	/* Build the tree from the list */
	while ((row = (char **) lwc_delStartLL (job_list)) != NULL) {
		/*
		 * row[0] --> ID
		 * row[1] --> Type (0: jobset and 1: job)
		 * row[2] --> Start time
		 */

		/* Add the new job/jobset ID to the hierarchy list */
		if (lwc_addStartLL (hierarchy_list, row[0]) != 0) {
			sql_free_row (row);
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			free_jobtree_all_nodes (nxt);
			error_window (_("Memory allocation error"), NULL);
			return -1;
		}

		/* Create the new job object */
		ptr = new_schedwi_jobtree_node (	row[0], workload_date,
							row[1],
							jobset_date, row[2],
							hierarchy_list);
		if (ptr == NULL) {
			lwc_delStartLL (hierarchy_list);
			sql_free_row (row);
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			free_jobtree_all_nodes (nxt);
			return -1;
		}

		/* Link the new job/jobset */
		(*nb_jobs)++;
		ptr->parent = jobset;
		ptr->next = nxt;
		if (nxt != NULL) {
			nxt->prev = ptr;
		}
		nxt = ptr;

		/* If it's a jobset, recursively build the job tree */
		if (ptr->node_type == 0) {
			ret = build_jobtree_recur (	ptr,
							row[0],
							jobset_date,
							workload_date,
							row[2],
							calendars,
							hierarchy_list,
							&(ptr->children),
							nb_jobs);
			if (ret != 0) {
				lwc_delStartLL (hierarchy_list);
				sql_free_row (row);
				lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
				free_jobtree_all_nodes (nxt);
				return ret;
			}
		}

		/* Remove the job/jobset ID from the hierarchy list */
		lwc_delStartLL (hierarchy_list);

		sql_free_row (row);
	}
	lwc_delLL (job_list, NULL);


	/*
	 * Complete the tree with the jobs and jobsets that must run tomorrow 
	 */

	/* Get the calendar ID list for tomorrow */
	schedwi_date_add_days (jobset_date, &date_tomorrow, 1);
	get_calendar_list_for_the_day (	calendars, date_tomorrow,
					&calendars_for_tomorrow);

	/* Get the job/jobset list */
	sql_ret = sql_job_tomorrow (	workload_date,
					&job_list,
					jobset_id, calendars_for_tomorrow,
					jobset_time,
				(void (*)(void *, const char*, unsigned int))
						error_window_ignore_errno,
					_("Database error"));
	g_free (calendars_for_tomorrow);
	if (sql_ret != 0) {
		free_jobtree_all_nodes (nxt);
		return (sql_ret == 1) ? -1 : -2;
	}

	/* Build the tree from the list */
	while ((row = (char **) lwc_delStartLL (job_list)) != NULL) {
		/*
		 * row[0] --> ID
		 * row[1] --> Type (0: jobset and 1: job)
		 * row[2] --> Start time
		 */

		/* Add the new job/jobset ID to the hierarchy list */
		if (lwc_addStartLL (hierarchy_list, row[0]) != 0) {
			sql_free_row (row);
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			free_jobtree_all_nodes (nxt);
			error_window (_("Memory allocation error"), NULL);
			return -1;
		}

		/* Create the new job object */
		ptr = new_schedwi_jobtree_node (	row[0], workload_date,
							row[1],
							date_tomorrow, row[2],
							hierarchy_list);
		if (ptr == NULL) {
			lwc_delStartLL (hierarchy_list);
			sql_free_row (row);
			lwc_delLL (	job_list,
					(void (*)(const void *)) sql_free_row);
			free_jobtree_all_nodes (nxt);
			return -1;
		}

		/* Link the new job/jobset */
		(*nb_jobs)++;
		ptr->parent = jobset;
		ptr->next = nxt;
		if (nxt != NULL) {
			nxt->prev = ptr;
		}
		nxt = ptr;

		/* If it's a jobset, recursively build the job tree */
		if (ptr->node_type == 0) {
			ret = build_jobtree_recur (	ptr,
							row[0],
							date_tomorrow,
							workload_date,
							row[2],
							calendars,
							hierarchy_list,
							&(ptr->children),
							nb_jobs);
			if (ret != 0) {
				lwc_delStartLL (hierarchy_list);
				sql_free_row (row);
				lwc_delLL (job_list,
					(void (*)(const void *)) sql_free_row);
				free_jobtree_all_nodes (nxt);
				return ret;
			}
		}

		/* Remove the job/jobset ID from the hierarchy list */
		lwc_delStartLL (hierarchy_list);

		sql_free_row (row);
	}
	lwc_delLL (job_list, NULL);

	*children = ptr;
	return 0;
}


/*
 * Build the job/jobset tree
 *
 * Return:
 *   0 --> No error.  top contains the tree and must be freed by 
 *         free_jobtree_all_nodes(). nb_jobs contains the number of
 *         jobs/jobsets in the returned tree.
 *  -1 --> Memory allocation error.  An error message has been displayed.
 *  -2 --> Database error.  An error message has been displayed.
 */
static int
build_jobtree (	schedwi_date jobset_date,
		calendar_list_t_ptr calendars,
		schedwi_jobtree_node_ptr *top,
		unsigned int *nb_jobs)
{
	lwc_LL *hierarchy_list;
	gchar *id;
	int ret, workload_date;

	/*
	 * Initialize the hierarchy working list with the root job (ID=1).
	 * This list will be used to retrieve the job parameters from the
	 * database.
	 */ 
	hierarchy_list = lwc_newLL ();
	if (hierarchy_list == NULL) {
		error_window (_("Memory allocation error"), NULL);
		return -1;
	}

	id = g_strdup ("1");
	if (lwc_addEndLL (hierarchy_list, id) != 0) {
		g_free (id);
		lwc_delLL (hierarchy_list, (void (*)(const void *))g_free);
		error_window (_("Memory allocation error"), NULL);
		return -1;
	}

	/* Build the job/jobset tree */
	*nb_jobs = 0;
	workload_date = schedwi_date_to_int (jobset_date);
	ret = build_jobtree_recur (	NULL, "0", jobset_date,
					workload_date, "0", calendars,
					hierarchy_list,
					top, nb_jobs);
	lwc_delLL (hierarchy_list, (void (*)(const void *))g_free);
	return ret;
}


/*
 * Create a new jobtree object
 *
 * Return:
 *   The new object (to be freed by the caller by free_jobtree()) or
 *   NULL in case of error (an error message has been displayed)
 */
jobtree_ptr
new_jobtree (schedwi_date date)
{
	jobtree_ptr ptr;
	unsigned int nb_jobs;
	calendar_list_t_ptr calendars;

	/* Load the calendar list for the provided date */
	calendars = new_calendar_list (date);
	if (calendars == NULL) {
		return NULL;
	}

	/* Allocate memory for the new object */
	ptr = g_new (jobtree, 1);

	/* Build the job/jobset tree */
	if (build_jobtree (date, calendars, &(ptr->top), &nb_jobs) != 0) {
		g_free (ptr);
		destroy_calendar_list (calendars);
		return NULL;
	}
	destroy_calendar_list (calendars);

	/* Walk in the job/jobset tree and fill the provided array */
	build_index (ptr->top, nb_jobs, &(ptr->jobs));

	ptr->nb_jobs = nb_jobs;
	ptr->date = date;

	return ptr;
}


/*
 * Free a job/jobset tree
 */
void
free_jobtree (jobtree_ptr ptr)
{
	if (ptr != NULL) {
		g_free (ptr->jobs);
		free_jobtree_all_nodes (ptr->top);
		g_free (ptr);
	}
}


/*
 * Check if the workload is completed (ie. the root node is completed)
 *
 * Return:
 *   0 --> Not completed
 *   1 --> Completed
 */
int
schedwi_jobtree_is_completed (jobtree_ptr ptr)
{
	int ret;

	if (ptr == NULL) {
		return 1;
	}

	if (ptr->top == NULL || ptr->top->status == JOB_STATUS_STATE_COMPLETED)
	{
		ret = 1;
	}
	else {
		ret = 0;
	}

	return ret;
}

/*-----------------============== End Of File ==============-----------------*/
