/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_stat.c -- Update and retrieve statistics (table job_stat) */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_stat.h>


#define SQL_STAT_GET "SELECT num_success,num_failed,total_duration,last_duration FROM job_stat WHERE job_id=\"%s\""
#define SQL_STAT_GET_DURATION "SELECT total_duration/num_success,last_duration FROM job_stat WHERE job_id=\"%s\""
#define SQL_STAT_SUCCESS_UPDATE "UPDATE job_stat SET num_success=num_success+1,total_duration=total_duration+%ld,last_duration=%ld WHERE job_id=\"%s\""
#define SQL_STAT_FAIL_UPDATE "UPDATE job_stat SET num_failed=num_failed+1,last_duration=%ld WHERE job_id=\"%s\""
#define SQL_STAT_SUCCESS_REPLACE "REPLACE INTO job_stat (job_id,num_success,num_failed,total_duration,last_duration) VALUES (\"%s\",1,0,%ld,%ld)"
#define SQL_STAT_FAIL_REPLACE "REPLACE INTO job_stat (job_id,num_success,num_failed,total_duration,last_duration) VALUES (\"%s\",0,1,0,%ld)"


/*
 * Get the usual duration of the specified job/jobset
 *
 * Return:
 *     0 --> No error.  duration contains the retrieved average duration.
 *           last_run_duration contains the retrieved last run duration.
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_stat_get_duration (	const char *job_id,
			long int *duration,
			long int *last_run_duration,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows;
	char **row;

#if HAVE_ASSERT_H
	assert (   job_id != NULL && duration != NULL
		&& last_run_duration != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_STAT_GET_DURATION,
				SQL_STRING, job_id,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (char **) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	if (row == NULL) {
		*duration = 1;
		*last_run_duration = 0;
	}
	else {
		*duration = strtol (row[0], NULL, 0);
		*last_run_duration = strtol (row[1], NULL, 0);
		sql_free_row (row);
	}
	return 0;
}


/*
 * Get the statistics details from the database for the specified job/jobset
 *
 * Return:
 *     0 --> No error.  row contains the retrieve data (or NULL if no
 *           statistics are available for the provided job).  row must be
 *           freed by the caller by sql_free_row()
 *               row[0] --> Number of successful runs
 *               row[1] --> Number of failed runs
 *               row[2] --> Total duration of the job/jobset
 *               row[3] --> Last run duration
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_stat_get (	const char *job_id,
		char ***row,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	lwc_LL *rows;

#if HAVE_ASSERT_H
	assert (job_id != NULL && row != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_STAT_GET, SQL_STRING, job_id, SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	*row = (char **) lwc_delStartLL (rows);
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
	return 0;
}


/*
 * Update the statistics for the provided successful job
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_stat_success (	const char *job_id,
			long int duration,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	unsigned long long int nb = 0;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, &nb,
				SQL_STAT_SUCCESS_UPDATE,
				SQL_INT, duration,
				SQL_INT, duration,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* There is not yet a statistic row for this job.  Add one */
	err_msg = NULL;
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_STAT_SUCCESS_REPLACE,
				SQL_STRING, job_id,
				SQL_INT, duration,
				SQL_INT, duration,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}


/*
 * Update the statistics for the provided failed job
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_stat_fail (	const char *job_id,
		long int duration,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	unsigned long long int nb = 0;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, &nb,
				SQL_STAT_FAIL_UPDATE,
				SQL_INT, duration,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Update successful.  There was already a row for this job */
	if (nb != 0) {
		return 0;
	}

	/* There is not yet a statistic row for this job.  Add one */
	err_msg = NULL;
	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL,
				SQL_STAT_FAIL_REPLACE,
				SQL_STRING, job_id,
				SQL_INT, duration,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
