/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_test.c -- Test the connection to a Schedwi database */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#include <mysql/mysql.h>
#include <mysql/mysqld_error.h>

#include <lib_functions.h>
#include <sql_test.h>


#define SQL_TEST_REQ "SELECT version_number,installed_date FROM schedwi_version ORDER BY installed_date DESC"


/*
 * Test the connection to the Schedwi database.  The provided step_status()
 * function will be called for each step of the connection with the
 * following parameters:
 *    1/ Step (SQL_TEST_CONNECT, SQL_TEST_DATABASE or SQL_TEST_VERSION)
 *    2/ Status (0 --> Success, 1 --> Error)
 *    3/ Associated message (may be NULL)
 *    4/ The provided user_data pointer
 *
 * Return:
 *   0 --> Success
 *  -1 --> Error
 */
int
sql_test (const char *host, const char *user, const char *password,
	const char *database, long int port, const char *socket_file,
	void (*step_status)(sql_test_step_t, char, const char *, void *),
	void *user_data)
{
	MYSQL sql;
	MYSQL_RES *result;
	MYSQL_ROW row;
	char *str, *version_str;

	/*
	 * Initialization
	 */
	mysql_init (&sql);
	mysql_options (&sql, MYSQL_READ_DEFAULT_GROUP, SOFTWARE_NAME);


	/*
	 * Connection
	 */
	if (mysql_real_connect (&sql, host, user, password, database,
				(unsigned int)port, socket_file, 0) == NULL)
	{
		if (step_status != NULL) {
			step_status (	SQL_TEST_CONNECT, 1,
					mysql_error (&sql), user_data);
		}
		mysql_close (&sql);
		return -1;
	}
	if (step_status != NULL) {
		step_status (SQL_TEST_CONNECT, 0, NULL, user_data);
	}


	/*
	 * Read the schedwi_version table
	 */
	if (mysql_query (&sql, SQL_TEST_REQ) != 0) {
		/* The table probably doesn't exist */
		if (step_status != NULL) {
			step_status (	SQL_TEST_DATABASE, 1,
					mysql_error (&sql), user_data);
		}
		mysql_close (&sql);
		return -1;
	}

	result = mysql_store_result (&sql);
	/* Error */
	if (result == NULL && mysql_errno(&sql) != 0) {
		if (step_status != NULL) {
			step_status (	SQL_TEST_DATABASE, 1,
					mysql_error (&sql), user_data);
		}
		mysql_close (&sql);
		return -1;
	}

	/* Nothing returned */
	if (	   result == NULL
		|| mysql_num_rows(result) == 0
		|| (row = mysql_fetch_row (result)) == NULL)
	{
		if (result != NULL){
			mysql_free_result (result);
		}
		mysql_close (&sql);
		if (step_status != NULL) {
			step_status (	SQL_TEST_DATABASE, 1,
					_("Database not initialized"),
					user_data);
		}
		return -1;
	}
	if (step_status != NULL) {
		step_status (SQL_TEST_DATABASE, 0, NULL, user_data);
	}

	/* Built the version string */
	str = _("Schedwi Database Version: %s");
	version_str = (char *) malloc (	  schedwi_strlen (str)
					+ schedwi_strlen (row[0])); 
	if (version_str != NULL) {
		sprintf (version_str, str, row[0]);
	}

	if (step_status != NULL) {
		step_status (SQL_TEST_VERSION, 0, version_str, user_data);
	}

	if (version_str != NULL) {
		free (version_str);
	}
	mysql_free_result (result);
	mysql_close (&sql);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
