/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_module_runjob.c -- Module to start a job */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_linkedlist.h>
#include <job_parameters.h>
#include <lwc_log.h>
#include <child_mgnt.h>
#include <job_launcher.h>
#include <parse_env_and_arg.h>
#include <net_module_runjob.h>


#if CYGWIN_IN_USE
/*
 * For Cygwin, some environment variables must be set to fork/exec the
 * command job.  This function get the required variables from the current
 * environment and copy them to the environment of the new job
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error
 */
static int
cygwin_required_env (job_parameters_t *job)
{
	char *e;

	if (job == NULL) {
		return 0;
	}

	e = getenv ("PATH");
	if (e != NULL && add_env_job_parameters (job, "PATH", e) != 0) {
		return -1;
	}
	return 0;
}
#endif


/*
 * Parse the server request (in buffer) and run the specified job
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (lwc_writeLog() is used to display an error
 *         message)
 *  -2 --> Syntax error in the request from the server (lwc_writeLog() is used
 *         to display an error message)
 *  -3 --> Error starting the job (lwc_writeLog() is used to display an error
 *         message)
 */
int
run_job (net_id *sock, char *buffer, size_t size)
{
	lwc_LL *list_arguments, *list_environment;
	char *err_msg;
	char *buff_cpy, *str;
	job_parameters_t job;
	char *job_id;
	int ret;
#if HAVE_PID_T
	pid_t child;
#else
	int child;
#endif

#if HAVE_ASSERT_H
	assert (sock != NULL && buffer != NULL);
#endif

	/* Arguments and environment lists */
	list_arguments = lwc_newLL ();
	if (list_arguments == NULL) {
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}
	list_environment = lwc_newLL ();
	if (list_environment == NULL) {
		lwc_delLL (list_arguments, NULL);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	/* Copy the input buffer to add a '\0' at the end (for strtok below) */
	buff_cpy = (char *) malloc (size + 1);
	if (buff_cpy == NULL) {
		lwc_delLL (list_environment, NULL);
		lwc_delLL (list_arguments, NULL);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}
	strncpy (buff_cpy, buffer, size);
	buff_cpy [size] = '\0';

	init_job_parameters (&job);

	/*
	 * For Cygwin, some environment variables are required to fork/exec
	 * the job command
	 */
#if CYGWIN_IN_USE
	if (cygwin_required_env (&job) != 0) {
		destroy_job_parameters (&job);
		free (buff_cpy);
		lwc_delLL (list_environment, NULL);
		lwc_delLL (list_arguments, NULL);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}
#endif

	job_id = NULL;

	for (	str = strtok (buff_cpy, "\n");
		str != NULL;
		str = strtok (NULL, "\n"))
	{
		/* Skip the spaces at the begining of the line */
		while (*str != '\0' && isspace (*str) != 0) {
			str++;
		}

		/* Environment variable */
		if (schedwi_strncasecmp (str, "ENV:", 4) == 0) {

			ret = parse_environment (sock, str + 4,
							list_environment);
			if (ret != 0) {
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				return ret;
			}
			continue;
		}

		/* An argument */
		if (schedwi_strncasecmp (str, "ARG:", 4) == 0) {
			ret = parse_arguments (sock, str + 4, list_arguments);
			if (ret != 0) {
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				return ret;
			}
			continue;
		}

		/* Job ID */
		if (schedwi_strncasecmp (str, "JID:", 4) == 0) {
			if (add_env_job_parameters (	&job,
							"SCHEDWI_JOBID",
							str + 4) != 0)
			{
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				err_msg = _("Memory allocation error");
				net_write (sock, "1", 1);
				net_write (sock, err_msg,
						schedwi_strlen (err_msg));
				lwc_writeLog (LOG_CRIT, err_msg);
				return -1;
			}
			job_id = str + 4;
			continue;
		}

		/* Command */
		if (schedwi_strncasecmp (str, "CMD:", 4) == 0) {
			if (add_command_job_parameters (&job, str + 4) != 0) {
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				err_msg = _("Memory allocation error");
				net_write (sock, "1", 1);
				net_write (sock, err_msg,
						schedwi_strlen (err_msg));
				lwc_writeLog (LOG_CRIT, err_msg);
				return -1;
			}
			continue;
		}

		/* Username */
		if (schedwi_strncasecmp (str, "USR:", 4) == 0) {
			if (add_username_job_parameters (&job, str + 4) != 0) {
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				err_msg = _("Memory allocation error");
				net_write (sock, "1", 1);
				net_write (sock, err_msg,
						schedwi_strlen (err_msg));
				lwc_writeLog (LOG_CRIT, err_msg);
				return -1;
			}
			continue;
		}

		/* File out */
		if (schedwi_strncasecmp (str, "OUT:", 4) == 0) {
			if (add_stdout_job_parameters (&job, str + 4) != 0) {
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				err_msg = _("Memory allocation error");
				net_write (sock, "1", 1);
				net_write (sock, err_msg,
						schedwi_strlen (err_msg));
				lwc_writeLog (LOG_CRIT, err_msg);
				return -1;
			}
			continue;
		}

		/* File err */
		if (schedwi_strncasecmp (str, "ERR:", 4) == 0) {
			if (add_stderr_job_parameters (&job, str + 4) != 0) {
				destroy_job_parameters (&job);
				destroy_argument_list (list_arguments);
				destroy_environment_list (list_environment);
				free (buff_cpy);
				err_msg = _("Memory allocation error");
				net_write (sock, "1", 1);
				net_write (sock, err_msg,
						schedwi_strlen (err_msg));
				lwc_writeLog (LOG_CRIT, err_msg);
				return -1;
			}
			continue;
		}

		/* Load environment */
		if (schedwi_strncasecmp (str, "LDE:", 4) == 0) {
			add_loadenv_job_parameters (&job, atoi (str + 4));
			continue;
		}
	}

	/* Check that the job_id (JID) has been provided */
	if (job_id == NULL) {
		destroy_job_parameters (&job);
		destroy_argument_list (list_arguments);
		destroy_environment_list (list_environment);
		free (buff_cpy);
		err_msg = _("Syntax error: job ID not provided");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_ERR, err_msg);
		return -2;
	}

	/* Add the argument list to the job structure */
	if (argument_list_to_job_parameter (	list_arguments,
						&(job.arguments)) != 0)
	{
		destroy_job_parameters (&job);
		destroy_argument_list (list_arguments);
		destroy_environment_list (list_environment);
		free (buff_cpy);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	/* Add the environment variables to the job structure */
	if (environment_list_to_job_env (	list_environment,
						&(job.environ)) != 0)
	{
		destroy_job_parameters (&job);
		destroy_environment_list (list_environment);
		free (buff_cpy);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	/* Start the job */
	ret = submit_job (&job, job_id, &child, &err_msg);
	destroy_job_parameters (&job);
	if (ret != 0) {
		free (buff_cpy);
		net_write (sock, "1", 1);
		if (err_msg != NULL) {
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_ERR, err_msg);
		}
		return (ret == -2) ? -3: ret;
	}
	else {
		lwc_writeLog (	LOG_INFO, _("Job %s: Started (PID %d)"),
				job_id, child);
	}

	/* Add the job to the list of the running jobs */
	if (add_child (child, job_id) != 0) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
	}
	free (buff_cpy);

	net_write (sock, "0", 1);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
