/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_module_testfile.c -- Module to test the presence of a file */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#include <lib_functions.h>
#include <lwc_linkedlist.h>
#include <lwc_log.h>
#include <parse_env_and_arg.h>
#include <parsevar.h>
#include <net_module_testfile.h>


/*
 * Test the presence of a file/directory/socket/... and send the result to
 * the network client
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (a message is displayed by using
 *         lwc_writeLog())
 *  -2 --> Syntax error (a message is displayed by using lwc_writeLog())
 */
int
test_file (net_id *sock, char *buffer, size_t size)
{
	lwc_LL *list_environment;
	char *err_msg;
	char *buff_cpy, *str, *file, *out;
	size_t out_len;
	int ret, idx;
	struct stat file_info;
	environment_t env, parsed_env;

#if HAVE_ASSERT_H
	assert (sock != NULL && buffer != NULL);
#endif

	/* Environment list */
	list_environment = lwc_newLL ();
	if (list_environment == NULL) {
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	/* Copy the input buffer to add a '\0' at the end (for strtok below) */
	buff_cpy = (char *) malloc (size + 1);
	if (buff_cpy == NULL) {
		lwc_delLL (list_environment, NULL);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}
	strncpy (buff_cpy, buffer, size);
	buff_cpy [size] = '\0';

	init_environment (&env);
	file = NULL;

	for (	str = strtok (buff_cpy, "\n");
		str != NULL;
		str = strtok (NULL, "\n"))
	{
		/* Skip the spaces at the begining of the line */
		while (*str != '\0' && isspace (*str) != 0) {
			str++;
		}

		/* Environment variable */
		if (schedwi_strncasecmp (str, "ENV:", 4) == 0) {

			ret = parse_environment (sock, str + 4,
							list_environment);
			if (ret != 0) {
				destroy_environment_list (list_environment);
				destroy_environment (&env);
				free (buff_cpy);
				return ret;
			}
			continue;
		}

		/* File name */
		if (schedwi_strncasecmp (str, "FIL:", 4) == 0) {
			file = str + 4;
			continue;
		}

		/* Job ID */
		if (schedwi_strncasecmp (str, "JID:", 4) == 0) {
			if (add_env (&env, "SCHEDWI_JOBID", str + 4) != 0) {
				destroy_environment_list (list_environment);
				destroy_environment (&env);
				free (buff_cpy);
				err_msg = _("Memory allocation error");
				net_write (sock, "1", 1);
				net_write (sock, err_msg,
						schedwi_strlen (err_msg));
				lwc_writeLog (LOG_CRIT, err_msg);
				return -1;
			}
			continue;
		}
	}

	/* Check that the file name has been provided */
	if (file == NULL) {
		destroy_environment_list (list_environment);
		destroy_environment (&env);
		free (buff_cpy);
		err_msg = _("Syntax error: file name not provided");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_ERR, err_msg);
		return -2;
	}

	/* Convert the list to an environment_t object */
	if (environment_list_to_job_env (list_environment, &env) != 0) {
		destroy_environment_list (list_environment);
		destroy_environment (&env);
		free (buff_cpy);
		err_msg = _("Memory allocation error");
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		lwc_writeLog (LOG_CRIT, err_msg);
		return -1;
	}

	/* Parse the provided environment and create a new one */
	init_environment (&parsed_env);
	switch (parse_param (&env, &parsed_env, &idx)) {
		case -1:
			destroy_environment (&env);
			destroy_environment (&parsed_env);
			free (buff_cpy);
			err_msg = _("Memory allocation error");
			net_write (sock, "1", 1);
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_CRIT, err_msg);
			return -1;
		case -2: case -3:
			destroy_environment (&parsed_env);
			free (buff_cpy);
			err_msg = _("Syntax error in the environment");
			net_write (sock, "1", 1);
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_ERR,
				_("Syntax error in the environment: %s"),
				env.env[idx]);
			destroy_environment (&env);
			return -2;
	}
	destroy_environment (&env);

	/* Parse the file name */
	out = NULL;
	out_len = 0;
	switch (parse_string (file, schedwi_strlen (file), &parsed_env,
			&out, &out_len))
	{
		case -1:
			destroy_environment (&parsed_env);
			free (buff_cpy);
			err_msg = _("Memory allocation error");
			net_write (sock, "1", 1);
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_CRIT, err_msg);
			return -1;
		case -2:
			destroy_environment (&parsed_env);
			err_msg = _("Syntax error in the file name");
			net_write (sock, "1", 1);
			net_write (sock, err_msg, schedwi_strlen (err_msg));
			lwc_writeLog (LOG_ERR,
				_("Syntax error in the file name: %s"),
				file);
			free (buff_cpy);
			return -2;
	}
	destroy_environment (&parsed_env);
	free (buff_cpy);

	/* Check the file */
	if (out == NULL || out_len == 0 || stat (out, &file_info) != 0) {
		err_msg = strerror (errno);
		net_write (sock, "1", 1);
		net_write (sock, err_msg, schedwi_strlen (err_msg));
	}
	else {
		net_write (sock, "0", 1);
	}
	if (out != NULL) {
		free (out);
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
