/* Schedwi
   Copyright (C) 2007 Herve Quatremain
     
   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * net_utils_sock.c -- Useful network functions (basic socket connection
 * without SSL - which is not available on this host)
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif

#if HAVE_NETDB_H
#include <netdb.h>
#endif

#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#else
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_RECV
#define recv(a,b,c,d) read(a,b,c)
#endif

#ifndef HAVE_SEND
#define send(a,b,c,d) write(a,b,c)
#endif

#ifndef PF_INET
#define PF_INET AF_INET
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <lib_functions.h>
#include <net_utils_sock.h>


#define BUFFER_LEN 1024
#define ENF_OF_REQUEST "\nbYe"

/* Protocol number for TCP */
static int protocol = -1;


/*
 *  Convert a port number/name in an unsigned short int suitable for the
 *  member sin_port of the sockaddr_in structure
 *
 *  Return :
 *    The port number or 0 in case of error
 */
static unsigned short int
get_port_number (const char *port)
{
	int i;
#if HAVE_GETSERVBYNAME
	struct servent *entry;
#endif

#if HAVE_ASSERT_H
	assert (port != NULL);
#endif

	/* Skip the spaces */
	while (*port != '\0' && isspace (*port) != 0) {
		port++;
	}

	/* Empty string ! */
	if (*port == '\0') {
		return 0;
	}

	/* Check if the string contains only digits */
	for (i = 0; port[i] != '\0' && isdigit (port[i]) != 0; i++);

	/* Only digits */
	if (port[i] == '\0') {
		return htons ((unsigned short int) atoi (port));
	}

#if HAVE_GETSERVBYNAME
	/*
	 * The string is a port name. This name will be searched in
	 * /etc/services (or equivalent)
	 */
	entry = getservbyname (port, "tcp");
	if (entry != NULL) {
		return (unsigned short int) entry->s_port;
	}
#endif
	return  0;
}


/*
 * Check that the network client is allowed
 *
 * Return:
 *   0 --> Client allowed
 *   1 --> Client not allowed
 *  -1 --> Error (a message is displayed using lwc_writeLog())
 */
static int
net_check_client (struct sockaddr_in *in, const char *schedwi_server_name)
{
	char *addr;
	int i, try_no;
	struct hostent *he;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

	if (in == NULL) {
		return 1;
	}

	addr = inet_ntoa (in->sin_addr);
	if (schedwi_strcasecmp (addr, schedwi_server_name) == 0) {
		return 0;
	}

	try_no = 0;
	do {
		he = gethostbyaddr (	&(in->sin_addr),
					sizeof (in->sin_addr),
					AF_INET);
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES) {
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				lwc_writeLog (	LOG_ERR,
						_("Name server error"));
				return -1;
			}
			else {
				return 1;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);
	if (he == NULL) {
		lwc_writeLog (LOG_WARNING, _("Name server unavailable"));
		return -1;
	}

	if (schedwi_strcasecmp (he->h_name, schedwi_server_name) == 0) {
		return 0;
	}

	for (i = 0; (he->h_aliases)[i] != NULL; i++) {
		if (schedwi_strcasecmp ((he->h_aliases)[i],
					schedwi_server_name) == 0)
		{
			return 0;
		}
	}
	return 1;
}


/*
 * Wait for a network request and return the socket associated with the
 * accepted connection.  The client host is check to be sure that it's the
 * Schedwi server
 *
 * Return:
 *   The new socket or
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 */
int
net_accept_sock (int sock, const char *schedwi_server_name)
{
	struct sockaddr_in sockname;
#if HAVE_SOCKLEN_T
	socklen_t len;
#else
	int len;
#endif
	int s, ret, save_errno;

	while (1) {
		len = sizeof(struct sockaddr_in);
		s = accept (sock, (struct sockaddr *)&sockname, &len);
		if (s < 0) {
#if CYGWIN_IN_USE
			/*
			 * Sometimes, using Cygwin, the errno value is
			 * not set to EINTR nor EAGAIN when accept has
			 * been interrupted by a signal.  When accept()
			 * returns -1, the error is then ignored and accept()
			 * is run again
			 */
			continue;
#else
			if (	   errno == EINTR
				|| errno == EAGAIN)
			{
				continue;
			}
			save_errno = errno;
			lwc_writeLog (	LOG_CRIT,
					_("Network error: accept: %s"),
					strerror (errno));
			errno = save_errno;
			return -1;
#endif
		}

		/*
		 * Only the server (defined by the configuration option
		 * SERVER) is allowed
		 */
		if (	   schedwi_server_name != NULL
			&& schedwi_server_name[0] != '\0')
		{
			ret = net_check_client (&sockname, schedwi_server_name);
			if (ret == 1) {
				lwc_writeLog (	LOG_NOTICE,
						_("Connection refused for %s"),
						inet_ntoa(sockname.sin_addr));
			}
			if (ret != 0) {
				closesocket (s);
				continue;
			}
		}
		return s;
	}
	return -1;
}


/*
 * Create a socket ready to accept requests
 *
 * port_number is the port number to listen to (a number or a service name
 *             in /etc/services)
 *       iface is the network interface to listen to (NULL or empty means all
 *             the interfaces)
 *
 * Return:
 *   The socket or
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_server_sock (const char *port_number, const char *iface)
{
	struct protoent *prot;
	int sock, opt, try_no;
	struct  sockaddr_in sad;
	unsigned long int iface_addr;
	unsigned short int port;
	struct hostent *he;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL);
#endif

	/* Get the port number */
	port = get_port_number (port_number);
	if (port == 0) {
		lwc_writeLog (LOG_ERR,
		 	_("Cannot get the port number for `%s'"),
			port_number);
		return -1;
	}

	/* Get the IP address of the network interface to listen to */
	if (iface == NULL || iface[0] == '\0') {
		iface_addr = htonl (INADDR_ANY);
	}
	else {
		try_no = 0;
		do {
			he = gethostbyname (iface);
			if (he == NULL) {
				if (h_errno == TRY_AGAIN) {
					try_no++;
					if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES)
					{
#if HAVE_NANOSLEEP
						nanosleep (&req, NULL);
#elif HAVE_SLEEP
						sleep (
						    SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
						usleep (
						    SCHEDWI_HOSTLOOKUP_SLEEP
						  * 1000000);
#endif
					}
				}
				else {
				if (h_errno == NO_RECOVERY) {
					lwc_writeLog (	LOG_ERR,
							_("Name server error"));
					return -1;
				}
				else {
					lwc_writeLog (	LOG_ERR,
						_("Cannot get address for %s"),
						iface);
					return -1;
				}}
			}
		} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

		if (he == NULL) {
			lwc_writeLog (	LOG_WARNING,
					_("Name server unavailable"));
			return -1;
		}

		iface_addr = *((unsigned long *)(he->h_addr_list[0]));
	}

	/* Get the protocol number for tcp */
	if (protocol == -1) {
#if HAVE_GETPROTOBYNAME
		prot = getprotobyname ("tcp");
		if (prot == NULL) {
			lwc_writeLog (LOG_ERR,
		    _("getprotobyname: cannot get protocol number for tcp"));
			return -1;
		}
		protocol = prot->p_proto;
#else
		protocol = 0;
#endif
	}

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, protocol);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Set the REUSEADDR flag */
	opt = -1;
	if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof (opt)) != 0)
	{
		lwc_writeLog (LOG_ERR, _("setsockopt: %s"), strerror (errno));
	}

	/* Compose the address */
	schedwi_memset(&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = iface_addr;
	sad.sin_port = port;

	/* Bind to the address */
	if (bind (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		lwc_writeLog (LOG_ERR, _("bind: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	/* Prepare to accept requests */
	if (listen (sock, SCHEDWI_LISTEN_BACKLOG) != 0) {
		lwc_writeLog (LOG_ERR, _("listen: %s"), strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
}


/*
 * Create a socket connected to a server
 * port_number is the port number to connect to (a number or a service name
 *             in /etc/services)
 *
 * Return:
 *   The socket or
 *   -1 in case of error (an error message is displayed using lwc_writeLog())
 */
int
net_client_sock (const char *port_number, const char *hostname)
{
	struct protoent *prot;
	int sock, try_no;
	struct  sockaddr_in sad;
	unsigned long int iface_addr;
	unsigned short int port;
	struct hostent *he;
#if HAVE_NANOSLEEP
	struct timespec req;

	req.tv_sec = SCHEDWI_HOSTLOOKUP_SLEEP;
	req.tv_nsec = 0;
#endif

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL);
#endif

	/* Get the port number */
	port = get_port_number (port_number);
	if (port == 0) {
		lwc_writeLog (LOG_ERR,
		 	_("Cannot get the port number for `%s'"),
			port_number);
		return -1;
	}

	/* Get the IP address of the server */
	try_no = 0;
	do {
		he = gethostbyname (hostname);
		if (he == NULL) {
			if (h_errno == TRY_AGAIN) {
				try_no++;
				if (try_no < SCHEDWI_HOSTLOOKUP_RETRIES)
				{
#if HAVE_NANOSLEEP
					nanosleep (&req, NULL);
#elif HAVE_SLEEP
					sleep (SCHEDWI_HOSTLOOKUP_SLEEP);
#elif HAVE_USLEEP
					usleep (  SCHEDWI_HOSTLOOKUP_SLEEP
						* 1000000);
#endif
				}
			}
			else {
			if (h_errno == NO_RECOVERY) {
				lwc_writeLog (	LOG_ERR,
						_("Name server error"));
				return -1;
			}
			else {
				lwc_writeLog (	LOG_ERR,
					_("Cannot get address for %s"),
					hostname);
				return -1;
			}}
		}
	} while (he == NULL && try_no < SCHEDWI_HOSTLOOKUP_RETRIES);

	if (he == NULL) {
		lwc_writeLog (	LOG_WARNING,
				_("Name server unavailable"));
		return -1;
	}

	iface_addr = *((unsigned long *)(he->h_addr_list[0]));

	/* Get the protocol number for tcp */
	if (protocol == -1) {
#if HAVE_GETPROTOBYNAME
		prot = getprotobyname ("tcp");
		if (prot == NULL) {
			lwc_writeLog (LOG_ERR,
		    _("getprotobyname: cannot get protocol number for tcp"));
			return -1;
		}
		protocol = prot->p_proto;
#else
		protocol = 0;
#endif
	}

	/* Create the socket */
	sock = socket (PF_INET, SOCK_STREAM, protocol);
	if (sock < 0) {
		lwc_writeLog (LOG_ERR, _("socket: %s"), strerror (errno));
		return -1;
	}

	/* Compose the remote address */
	schedwi_memset(&sad, 0, sizeof(sad));
        sad.sin_family = AF_INET;
        sad.sin_addr.s_addr = iface_addr;
	sad.sin_port = port;

	/* Connect to the address */
	if (connect (sock, (struct sockaddr *)&sad, sizeof(sad)) != 0) {
		lwc_writeLog (LOG_ERR, _("Failed to connect to %s: %s"),
				hostname, strerror (errno));
		closesocket (sock);
		return -1;
	}

	return sock;
}


/*
 * Read data from the provided socket handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message is displayed using lwc_writeLog())
 *    -2 if too many data are received (a message is displayed using
 *       lwc_writeLog())
 */
ssize_t
net_read_sock (int sock, char **buff, size_t *len)
{
	ssize_t nb_read, total;
	size_t to_read;
	char *b, *tmp;
	fd_set rfds;
	struct timeval tv;
	int ret;

#if HAVE_ASSERT_H
	assert (buff != NULL && len != NULL);
#endif

	if (*buff == NULL || *len == 0) {
		*buff = (char *) malloc (BUFFER_LEN);
		if (*buff == NULL) {
			*len = 0;
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
		*len = BUFFER_LEN;
	}

	total = 0;
	to_read = *len;
	b = *buff;
	do {
		do {
			FD_ZERO (&rfds);
			FD_SET (sock, &rfds);
			tv.tv_sec = 5;
			tv.tv_usec = 0;
			ret = select (sock + 1, &rfds, NULL, NULL, &tv);
		} while (ret < 0 && (errno == EINTR || errno == EAGAIN));

		if (ret < 0) {
			lwc_writeLog (LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			return -1;
		}

		if (ret == 0) {
			lwc_writeLog (LOG_NOTICE,
			_("Timeout while waiting to read network data"));
			return -1;
		}

		do {
			nb_read = recv (sock, b, to_read, 0);
		} while (nb_read < 0 && (errno == EINTR || errno == EAGAIN));

		if (nb_read == 0) {
			return total;
		}

		if (nb_read < 0) {
			lwc_writeLog (	LOG_ERR,
					_("Error while reading data: %s"),
					strerror (errno));
			return -1;
		}

		total += nb_read;

		/* Find the EOF tag */
		tmp = schedwi_strnstr (*buff, ENF_OF_REQUEST, total);
		if (tmp != NULL) {
			return (ssize_t)(tmp - *buff);
		}

		if (nb_read < to_read) {
			to_read -= nb_read;
			b += nb_read;
		}
		else {
			if (*len + BUFFER_LEN > SCHEDWI_MAX_BUFFER_LEN) {
				/* Too many data */
				lwc_writeLog (LOG_NOTICE,
		_("Failed to read the network request: buffer overflow"));
				return -2;
			}

			tmp = (char *) realloc (*buff, *len + BUFFER_LEN);
			if (tmp == NULL) {
				lwc_writeLog (	LOG_CRIT,
						_("Memory allocation error"));
				return -1;
			}
			*buff = tmp;
			*len += BUFFER_LEN;
			b = *buff + total;
			to_read = BUFFER_LEN;
		}
	} while (1);

	return total;
}


/*
 * Read the reply
 *
 * The format of the reply must be as follow:
 *    The first character: '0' means success, an other character means failure
 *    The next characters: An optional explanation message (which is returned
 *                         in result_msg)
 * No reply (nothing to read) means failure.
 *
 * Return:
 *   0 --> The reply means the remote action was successfull. If result_msg is
 *         not NULL, it is set with the message sent by the remote socket. It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed. If result_msg is not NULL,
 *         it is set with the message sent by the remote socket. It may be NULL
 *         if no message was sent. It must be freed by the caller.
 *  -1 --> Error (a message is diplayed using lwc_writeLog())
 */
int
net_read_result_sock (int sock, char **result_msg)
{
	char *buff, ret_code;
	size_t len;
	ssize_t nb_read;
	int i;

	if (result_msg != NULL) {
		*result_msg = NULL;
	}

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_sock (sock, &buff, &len);
	if (nb_read < 0 ) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	if (nb_read > 0) {
		/* Return code sent by the remote process */
		ret_code = buff[0];

		if (result_msg != NULL) {
			/* Shift the message 1 character to the left */
			for (i = 1; i < nb_read; i++) {
				buff[i - 1] = buff[i];
			}
			buff [nb_read - 1] = '\0';
			*result_msg = buff;
		}
		else {
			free (buff);
		}

		/* Return code '0' means success */
		if (ret_code == '0') {
			return 0;
		}
	}
	else {
		if (buff != NULL) {
			free (buff);
		}
	}
	return 1;
}


/*
 * Write data to the provided socket handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_sock (int sock, const char *buff, size_t len)
{
	ssize_t nb_write;

	if (buff == NULL || len == 0) {
		return 0;
	}

	do {
		nb_write = send (sock, buff, len, 0);
		if (nb_write == len) {
			return 0;
		}

		if (nb_write < 0) {
			if (errno != EINTR && errno != EAGAIN) {
				lwc_writeLog (LOG_ERR,
					_("Error while sending data: %s"),
					strerror (errno));
				return -1;
			}
		}
		else {
			buff += nb_write;
			len -= nb_write;
		}

	} while (1);

	return 0;
}


/*
 * Write the End of Request tag
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_eof_sock (int sock)
{
	return net_write_sock (sock, ENF_OF_REQUEST,
				(size_t)schedwi_strlen (ENF_OF_REQUEST));
}


/*
 * Close the socket
 */
int
net_close_sock (int sock)
{
	if (sock >= 0) {
#if HAVE_SHUTDOWN
		shutdown (sock, SHUT_RDWR);
#endif
		return closesocket (sock);
	}
	return 0;
}

/*------------------------======= End Of File =======------------------------*/
