/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_utils.c -- Useful network functions */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif

#if HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#ifndef HAVE_CLOSESOCKET
#define closesocket(x) close(x)
#endif

#include <pthread.h>

#include <thread_init.h>
#include <lib_functions.h>
#include <lwc_log.h>
#include <sql_hosts.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <net_utils.h>

#define ENF_OF_REQUEST "\nbYe"


/*
 * Structure used to exchange parameters between the net_accept() function and
 * the new thread which manages the client connection
 */ 
struct _net_utils_data {
	struct in_addr in;
	int (*run)(schedwi_BIO *, void *);
	void *user_data;
	int s;
};
typedef struct _net_utils_data net_utils_data;
typedef struct _net_utils_data *net_utils_data_ptr;


/*
 * Create a new net_utils_data_ptr object
 *
 * Return:
 *   The new object (to be freed by the caller by net_utils_data_destroy()) or
 *   NULL in case of memory allocation error.
 */
static net_utils_data_ptr
net_utils_data_new (	int s, struct sockaddr_in *in,
			int (*run)(schedwi_BIO *, void *), void *user_data)
{
	net_utils_data_ptr ptr;

#if HAVE_ASSERT_H
	assert (in != NULL && run != NULL);
#endif

	ptr = (net_utils_data_ptr) malloc (sizeof (net_utils_data));
	if (ptr == NULL) {
		return NULL;
	}
	ptr->in        = in->sin_addr;
	ptr->s         = s;
	ptr->run       = run;
	ptr->user_data = user_data;
	return ptr;
}


/*
 * Free the provided net_utils_data_ptr object.  The associated socket is
 * also closed.
 */
static void
net_utils_data_destroy (net_utils_data_ptr ptr)
{
	if (ptr != NULL) {
		if (ptr->s >= 0) {
			net_close_sock (ptr->s);
		}
		free (ptr);
	}
}


/*
 * Error callback function for the sql_host_certificate() function
 */
static void
sql_host_certificate_error_logger (	void *data, const char *msg,
					unsigned int err_code)
{
	if (msg != NULL) {
		lwc_writeLog (LOG_ERR, msg);
	}
	else {
		lwc_writeLog (LOG_ERR,
	_("Database error while trying to retrieve the client parameters"));
	}
}


/*
 * Thread function used to manage the new client connection
 */
static void *
net_utils_server_thread (void *data)
{
	net_utils_data_ptr ptr = (net_utils_data_ptr) data;
	char **names;
	char use_ssl, *client_certificate;
	unsigned long int len_client_certificate;
	schedwi_BIO in;

	if (ptr == NULL) {
		pthread_exit (NULL);
	}

	/* Initialize the thread */
	thread_init ();

	/* Install the cleanup callback */
	pthread_cleanup_push (thread_clean, NULL);

	/* Retrieve the names associated with the in_addr structure */
	if (build_client_names_from_addr (ptr->in, &names) != 0) {
		net_utils_data_destroy (ptr);
		pthread_exit (NULL);
	}

	/*
	 * Retrieve whether SSL must be used for the connexion and retrieve
	 * the associated certificate
	 */
	client_certificate = NULL;
	if (sql_host_certificate (	names, &use_ssl, &client_certificate,
					&len_client_certificate,
					sql_host_certificate_error_logger,
					NULL) != 0)
	{
		net_utils_data_destroy (ptr);
		destroy_client_names (names);
		pthread_exit (NULL);
	}

	/*
	 * No SSL
	 */
	if (use_ssl == 0) {
		in.use_ssl = 0;
		in.trusted_cert_ptr = NULL;
		in.fd = ptr->s;
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		(ptr->run)(&in, ptr->user_data);
		net_utils_data_destroy (ptr);
		pthread_exit (NULL);
	}


	/*
	 * SSL
	 */

	in.trusted_cert_ptr = (gnutls_certificate_credentials_t *) malloc (
				sizeof (gnutls_certificate_credentials));
	if (in.trusted_cert_ptr == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		net_utils_data_destroy (ptr);
		pthread_exit (NULL);
	}
	in.use_ssl = 1;
	in.fd = ptr->s;

	if (net_server_ssl (&(in.session), in.fd, in.trusted_cert_ptr, names,
			client_certificate, len_client_certificate) != 0)
	{
		if (client_certificate != NULL) {
			free (client_certificate);
		}
		destroy_client_names (names);
		net_utils_data_destroy (ptr);
		free (in.trusted_cert_ptr);
		pthread_exit (NULL);
	}

	if (client_certificate != NULL) {
		free (client_certificate);
	}
	destroy_client_names (names);
	(ptr->run)(&in, ptr->user_data);
	net_close_ssl (in.session, in.fd, in.trusted_cert_ptr);
	ptr->s = -1; /* Socket already closed */
	net_utils_data_destroy (ptr);

	pthread_cleanup_pop (1);
	pthread_exit (NULL);
}


/*
 * Manage all the network request on the provided socket.  For each request
 * a new thread is started.
 * The run function is called to interract with the client (its second
 * parameter is the provided user_data parameter).
 *
 * Return:
 *   Never or
 *   -1 in case of error (errno is set and a message is displayed using
 *      lwc_writeLog())
 */
int
net_accept (int sock, int (*run)(schedwi_BIO *, void *), void *user_data)
{
	pthread_attr_t attr;
	pthread_t thread;
	int s, save_errno;
	net_utils_data_ptr ptr;
	struct sockaddr_in sockname;
#if HAVE_SOCKLEN_T
	socklen_t len;
#else
	int len;
#endif

#if HAVE_ASSERT_H
	assert (run != NULL);
#endif

	/* Initialize the thread attribute object */
	pthread_attr_init (&attr);
	pthread_attr_setdetachstate (&attr, PTHREAD_CREATE_DETACHED);

	/*
	 * In case this function is in a thread, make sure that if the thread
	 * is cancelled the attr object if freed
	 */
	pthread_cleanup_push (	(void (*) (void *))pthread_attr_destroy,
				(void *)&attr);

	/*
	 * For each received connection, a new detached thread is started
	 * to manage the new connection
	 */
	while (1) {
		pthread_testcancel ();
		len = sizeof(struct sockaddr_in);
		s = accept (sock, (struct sockaddr *)&sockname, &len);
		if (s < 0) {
			if (errno == EINTR || errno == EAGAIN) {
				/* Retry */
				continue;
			}
			save_errno = errno;
			lwc_writeLog (	LOG_CRIT,
					_("Network error: accept: %s"),
					strerror (errno));
			pthread_attr_destroy (&attr);
			errno = save_errno;
			return -1;
		}

		ptr = net_utils_data_new (s, &sockname, run, user_data);
		if (ptr == NULL) {
			closesocket (s);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			continue;
		}
		if (pthread_create (	&thread, &attr,
					net_utils_server_thread, ptr) != 0)
		{
			lwc_writeLog (	LOG_CRIT,
_("Cannot create a new thread to manage the network connection: %s"),
					strerror (errno));
			net_utils_data_destroy (ptr); /* Also close `s' */
			continue;
		}
	}
	/* Never reach this point */
	pthread_attr_destroy (&attr);
	pthread_cleanup_pop (0);
	return -1;
}


/*
 * Initialize the network layer (SSL algorithms) for a server
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
net_init_server ()
{
	return net_init_server_ssl ();
}


/*
 * Initialize the network layer.  Warning: this function is not thread safe
 * and must then be called only once in the main thread before the creation
 * of the other one.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error.  An error message has been logged by lwc_writeLog()
 */
int
net_init (	const char *schedwisrv_crt_file,
		const char *schedwisrv_key_file,
		char quick_random)
{
	return net_init_ssl (	schedwisrv_crt_file, schedwisrv_key_file,
				quick_random);
}


/*
 * Free the network layer.  This function may be called before exiting the
 * schedwisrv program
 */
void
net_destroy ()
{
	net_destroy_ssl ();
}


/*
 * Create a socket connected to a server.
 * port_number is the port number to connect to (a number or a service name
 * in /etc/services)
 *
 * Return:
 *   The schedwi_BIO handler to be freed by the caller by net_close() or
 *   NULL in case of error (an error message has been logged by lwc_writeLog())
 */
schedwi_BIO *
net_client (	const char *port_number, const char *hostname, char use_ssl,
		char *remote_crt, unsigned int remote_crt_len)
{
	schedwi_BIO *out;
	int ret;

#if HAVE_ASSERT_H
	assert (port_number != NULL && hostname != NULL);
#endif

	/* Allocate space for the returned schedwi_BIO object */
	out = (schedwi_BIO *) malloc (sizeof (schedwi_BIO));
	if (out == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	/* 
	 * No SSL, only socket
	 */
	if (use_ssl == 0) {
		out->use_ssl = 0;
		out->trusted_cert_ptr = NULL;
		out->fd = net_client_sock (port_number, hostname);
		if (out->fd < 0) {
			free (out);
			return NULL;
		}
		return out;
	}

	/*
	 * SSL
	 */
	out->use_ssl = 1;
	out->trusted_cert_ptr = (gnutls_certificate_credentials_t *) malloc (
				sizeof (gnutls_certificate_credentials));
	if (out->trusted_cert_ptr == NULL) {
		free (out);
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return NULL;
	}

	ret = net_client_ssl (	port_number, hostname,
				&(out->session), &(out->fd),
				out->trusted_cert_ptr,
				remote_crt, remote_crt_len);
	if (ret != 0) {
		free (out->trusted_cert_ptr);
		free (out);
		return NULL;
	}

	return out;
}


/*
 * Read data from the provided schedwi_BIO handler.
 * The data are store in *buff (of size *len). If buffer is NULL or not big
 * enough, it is reallocated to a bigger size. It must be freed by the caller
 *
 * Return:
 *    The total number of characters read and stored in *buff or
 *    -1 on error (a message has been displayed using lwc_writeLog())
 *    -2 if too many data are received (a message has been displayed using
 *       lwc_writeLog())
 */
int
net_read (schedwi_BIO *in, char **buff, unsigned int *len)
{
#if HAVE_ASSERT_H
	assert (in != NULL && buff != NULL && len != NULL);
#endif

	if (in->use_ssl == 1) {
		return net_read_ssl (in->session, buff, len);
	}
	else {
		return net_read_sock (in->fd, buff, len);
	}
}


/*
 * Read the reply
 *
 * The format of the reply must be as follow:
 *    The first character: '0' means success, an other character means failure
 *    The next characters: An optional explanation message (which is returned
 *                         in result_msg)
 * No reply (nothing to read) means failure.
 *
 * Return:
 *   0 --> The reply means the remote action was successfull.  If result_msg is
 *         not NULL, it is set with the message sent by the remote socket.  It
 *         must be freed by the caller.
 *   1 --> The reply means the remote action failed.  If result_msg is not
 *         NULL, it is set with the message sent by the remote socket.  It may
 *         be NULL if no message was sent.  It must be freed by the caller.
 *  -1 --> Error (a message has been logged by lwc_writeLog())
 */
int
net_read_result (schedwi_BIO *in, char **result_msg)
{
	char *buff, ret_code;
	unsigned int len;
	int nb_read;
	int i;

	if (result_msg != NULL) {
		*result_msg = NULL;
	}

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read (in, &buff, &len);
	if (nb_read < 0 ) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	if (nb_read > 0) {
		/* Return code sent by the remote process */
		ret_code = buff[0];

		if (result_msg != NULL) {
			/* Shift the message 1 character to the left */
			for (i = 1; i < nb_read; i++) {
				buff[i - 1] = buff[i];
			}
			buff [nb_read - 1] = '\0';
			*result_msg = buff;
		}
		else {
			free (buff);
		}

		/* Return code '0' means success */
		if (ret_code == '0') {
			return 0;
		}
	}
	else {
		if (buff != NULL) {
			free (buff);
		}
	}
	return 1;
}


/*
 * Write data to the provided handler.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message has been logged by lwc_writeLog())
 */
int
net_write (schedwi_BIO *out, const char *buff, int len)
{
	if (buff == NULL || len == 0) {
		return 0;
	}

#if HAVE_ASSERT_H
	assert (out != NULL);
#endif

	if (out->use_ssl == 1) {
		return net_write_ssl (out->session, buff, len);
	}
	else {
		return net_write_sock (out->fd, buff, len);
	}
}


/*
 * Write the End of Request tag
 *
 * Return:
 *   0 --> No error
 *  -1 --> Write error (a message is displayed using lwc_writeLog())
 */
int
net_write_eof (schedwi_BIO *out)
{
	return net_write (out, ENF_OF_REQUEST,
				(int)schedwi_strlen (ENF_OF_REQUEST));  
}


/*
 * Close the handler 
 */
void
net_close (schedwi_BIO *out)
{
	if (out != NULL) {
		if (out->use_ssl == 1) {
			net_close_ssl (	out->session, out->fd,
					out->trusted_cert_ptr);
		}
		else {
			net_close_sock (out->fd);
		}
		free (out);
	}
}

/*------------------------======= End Of File =======------------------------*/
