/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_job.c -- jobs management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_job.h>


#define SQL_JOB_TODAY "SELECT id,type,start_time,name FROM job_main_s WHERE workload_date=%d AND parent=\"%s\" AND enabled = 1 AND cal_id IN (0%s%s) AND (start_time = -1 OR start_time >= \"%s\")"
#define SQL_JOB_TOMORROW "SELECT id,type,start_time,name FROM job_main_s WHERE workload_date=%d AND parent=\"%s\" AND enabled = 1 AND cal_id IN (0%s%s) AND start_time >= 0 AND start_time < \"%s\""


/*
 * Get all the jobs and jobsets under the provided jobset, matching the
 * provided calendar list and the provided start time.
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved jobs/jobsets and must be
 *           freed by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_today ( int workload_date,
		lwc_LL **rows,
		const char *parent_id,
		const char *cal_list,
		const char *start_time,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL && parent_id != NULL && start_time != NULL);
#endif

	ret = sql_select (NULL, NULL, &err_msg, NULL, rows, NULL,
			SQL_JOB_TODAY,
			SQL_INT, (long int)workload_date,
			SQL_STRING, parent_id,
			SQL_STRING_NON_ESCAPE,
				(cal_list == NULL || cal_list[0] == '\0')
							? "": ",", 
			SQL_STRING_NON_ESCAPE,
				(cal_list == NULL || cal_list[0] == '\0')
							? "": cal_list, 
			SQL_STRING, start_time,
			SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Get all the jobs and jobsets under the provided jobset, matching the
 * provided calendar list and the provided start time.
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved jobs/jobsets and must be
 *           freed by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_tomorrow (	int workload_date,
			lwc_LL **rows,
			const char *parent_id,
			const char *cal_list,
			const char *start_time,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL && parent_id != NULL && start_time != NULL);
#endif

	ret = sql_select (NULL, NULL, &err_msg, NULL, rows, NULL,
			SQL_JOB_TOMORROW,
			SQL_INT, (long int)workload_date,
			SQL_STRING, parent_id,
			SQL_STRING_NON_ESCAPE,
				(cal_list == NULL || cal_list[0] == '\0')
							? "": ",", 
			SQL_STRING_NON_ESCAPE,
				(cal_list == NULL || cal_list[0] == '\0')
							? "": cal_list,
			SQL_STRING, start_time,
			SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Retrieve the `start limit' parameter for the provided job.
 *
 * Return:
 *     0 --> No error.  start_limit contains the retrieved parameter (number
 *           of minutes for the job to start)
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *     2 --> Database connection error (if error_func() is not NULL, it is
 *           called with user_data_error_func as its first parameter and the
 *           error message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_get_start_limit (int workload_date,
			lwc_LL *hierarchy_list,
			short int *start_limit,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	char *value = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (hierarchy_list != NULL);
#endif

	if (start_limit == NULL) {
		return 0;
	}
	ret = get_job_parameter (	workload_date, hierarchy_list,
					"job_start_limit_s", "start_limit",
					&value, NULL, &err_msg);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	if (value == NULL) {
		*start_limit = 0;
	}
	else  {
		*start_limit = atoi (value);
		free (value);
	}
	return 0;
}


/*
 * Retrieve the `max duration' parameter for the provided job.
 *
 * Return:
 *     0 --> No error.  max_duration contains the retrieved parameter (max
 *           number of minutes for the job to finish)
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *     2 --> Database connection error (if error_func() is not NULL, it is
 *           called with user_data_error_func as its first parameter and the
 *           error message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_get_max_duration (int workload_date,
			lwc_LL *hierarchy_list,
			short int *max_duration,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	char *value = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (hierarchy_list != NULL);
#endif

	if (max_duration == NULL) {
		return 0;
	}
	ret = get_job_parameter (	workload_date, hierarchy_list,
					"job_max_duration_s", "max_duration",
					&value, NULL, &err_msg);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	if (value == NULL) {
		*max_duration = 0;
	}
	else  {
		*max_duration = atoi (value);
		free (value);
	}
	return 0;
}


/*
 * Retrieve the `retries' parameter for the provided job.
 *
 * Return:
 *     0 --> No error.  retries contains the retrieved parameter (number of
 *           retries)
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *     2 --> Database connection error (if error_func() is not NULL, it is
 *           called with user_data_error_func as its first parameter and the
 *           error message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_get_retries (int workload_date,
			lwc_LL *hierarchy_list,
			int *retries,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	char *value = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (hierarchy_list != NULL);
#endif

	if (retries == NULL) {
		return 0;
	}
	ret = get_job_parameter (	workload_date, hierarchy_list,
					"job_retries_s", "retries",
					&value, NULL, &err_msg);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	if (value == NULL) {
		*retries = 0;
	}
	else  {
		*retries = atoi (value);
		free (value);
	}
	return 0;
}


/*
 * Retrieve the `retries_interval' parameter for the provided job.
 *
 * Return:
 *     0 --> No error.  retries_interval contains the retrieved parameter
 *           (number of minutes to wait between retries)
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *     2 --> Database connection error (if error_func() is not NULL, it is
 *           called with user_data_error_func as its first parameter and the
 *           error message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_job_get_retries_interval (int workload_date,
			lwc_LL *hierarchy_list,
			short int *retries_interval,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	char *value = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (hierarchy_list != NULL);
#endif

	if (retries_interval == NULL) {
		return 0;
	}
	ret = get_job_parameter (	workload_date, hierarchy_list,
					"job_retries_interval_s",
					"retries_interval",
					&value, NULL, &err_msg);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	if (value == NULL) {
		*retries_interval = 0;
	}
	else  {
		*retries_interval = atoi (value);
		free (value);
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
