/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * calendar_canvas.c - GNOME Canvas displaying a calendar
 *
 * Usage:
 *
 *   calendar_canvas_t *cal;
 *   cal = new_calendar_canvas (canvas, READWRITE);
 *
 *       new_calendar_canvas() creates and returns a new calendar_canvas_t
 *       object.
 *       The canvas parameter in a GnomeCanvas which must exists
 *       prior the call to new_calendar_canvas().  This canvas will be used
 *       to draw the calendar.
 *       The READWRITE parameter tells that the user can interact with the
 *       calendar (ie. selecting/deselecting days, month, ...)  The READONLY
 *       option disables this possibility.
 *       After the call to new_calendar_canvas() the canvas is initialized and
 *       a white background is displayed.
 *
 *
 *   draw_year_canvas (cal, 2005);
 *       
 *       draw_year_canvas() draws the calendar to the canvas for the provided
 *       year (2005 in this example).  This function can be called several
 *       times to change the year.  In this case the calendar is redraw for
 *       the new year.
 *
 *
 *   cal_errcode_t ret;
 *   int idx_err;
 *   ret = set_calendar (cal, "Jan-Dec/Fri/-1\nJan-Jun/Mon/1", &idx_err);
 *
 *       set_calendar() accepts a calendar formula
 *       ("Jan-Dec/Fri/-1\nJan-Jun/Mon/1" in this example) and marks/selects
 *       all the corresponding days in the calendar view.  The formula
 *       is stored internally so there is no need to call this function after
 *       each call to draw_year_canvas(): the selected days are automatically
 *       redrawn.
 *       The underline functions to manage the formula are the ones from
 *       calendar.c (see calendar.c for more details then).
 *       idx_err is an output parameter which is set in case of syntax error
 *       in the formula.  It specifies the position (index) of the error in the
 *       formula string.
 *       The return value of set_calendar() is the return value of str2cal()
 *       (in calendar.c).  In particular, CAL_NOERROR means no error.
 *
 *
 *   clear_calendar (cal);
 *
 *       clear_calendar() remove all the marks/selections from the calendar
 *       view.  The internal formula stored by set_calendar() is removed.
 *
 *
 *   gboolean my_set_day_cb (calendar_canvas_t *cal, GDateDay day,
 *                           GDateMonth month, gpointer user_data)
 *   {
 *       return TRUE;
 *   }
 *   set_callbacks_set_day (cal, my_set_day_cb, NULL); 
 *   set_callbacks_unset_day (cal, ..., NULL); 
 *   set_callbacks_set_wday (cal, ..., NULL); 
 *   set_callbacks_unset_wday (cal, ..., NULL); 
 *   set_callbacks_set_month (cal, ..., NULL); 
 *   set_callbacks_unset_month (cal, ..., NULL); 
 *
 *       Only if new_calendar_canvas() has been called with READWRITE. 
 *       The set_callbacks_set_*() functions are used to install callback
 *       functions which will be called when the user clicks on a day (or a
 *       month name or a week day name).  The last parameter (NULL in this
 *       case) is a user provided data which will be passed as the last
 *       parameter in the callback function.
 *       The set_callbacks_unset_*() functions are used to install callback
 *       functions which will be called when the user unselect a day (or a
 *       month name or a week day name).
 *       The return value (TRUE or FALSE) of the callback functions is used
 *       as follow:
 *       If TRUE is returned, the day (or month or week days) is selected (a
 *       mark is drawn for *_set_* functions) or removed (the mark is removed
 *       for the *_unset_* functions) on the canvas.
 *       If FALSE is returned, no mark is drawn nor removed from the canvas.
 *       WARNING:  If the days are selected/deselected using set_calendar() or
 *       clear_calendar() these callback functions are not called.  Moreover,
 *       a call to clear_calendar() will removed all the days selected by the
 *       user.  This is the same for set_calendar() which will deselect all the
 *       days not matching the provided formula.
 *       The same apply for draw_year_canvas() to view an other year.  All the
 *       marks set by the user are removed and only the marks corresponding to
 *       a formula set by set_calendar() are redrawn.
 *       The GConf notify function called to redisplay the calendar with new
 *       parameters also removes the user marks.
 *       The solution is to build a calendar formula inside the callback
 *       functions, to append it to the already existing formula and to call
 *       the set_calendar() function with the new formula.
 *
 *
 *   destroy_calendar_canvas (cal);
 *
 *       destroy_calendar_canvas() destroys the provided calendar_canvas_t
 *       object which is then invalid and cannot be used anymore.  The
 *       GnomeCanvas provided with new_calendar_canvas() is cleared and can
 *       be reused (it is not destroyed) 
 *
 * A GConf notify function is internally installed to reply to changes in
 * parameters (first week day, selection color, number of months displayed
 * on each rows, font)
 */

#include <schedwi.h>

#include <langinfo.h>

#include <schedwi_gconf.h>
#include <schedwi_style.h>
#include <calendar_canvas.h>


#define TEXT_BORDER_WIDTH 1
#define TEXT_BORDER_HEIGHT 1
#define DEFAULT_FIRST_DAY_OF_WEEK G_DATE_MONDAY
#define DEFAULT_MONTHS_BY_ROW 4
#define CALENDAR_SELECT_WIDTH 10


/*
 * Compute the height and width of the space required by a day.  For this
 * the abreviation of the day names, and the 2 digits (99) are drawn
 * (temporarly) on the canvas and their size computed.  The heighest and
 * widest is used to get the high and width of the space reserved to
 * draw a day.
 *
 * w, h and day_border are set by this function.
 * day_border is the space to leave around the day numbers for a proper
 * positioning
 */
static void
get_day_size (	GnomeCanvas *canvas, const gchar *font,
		gdouble *w, gdouble *h, gdouble *day_border)
{
	GnomeCanvasItem *item;
	gdouble tw, th, max_tw, max_th;

	max_tw = max_th = 0.0;


	/*
	 * First, the abreviation of all the day names (Mon, Tue, ...)
	 * are drawn
	 */
	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_1),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_2),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_3),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_4),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_5),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_6),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", nl_langinfo (ABDAY_7),
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}


	/*
	 * Now, draw two digits (99) to compute the size of a day number
	 * (01 to 31)
	 */
	item = gnome_canvas_item_new (
			gnome_canvas_root (canvas),
			gnome_canvas_text_get_type (),
			"x", (gdouble)0,
			"y", (gdouble)0,
			"anchor", GTK_ANCHOR_NORTH_WEST,
			"font", font,
			"text", "99",
			NULL);
	g_object_get(item, "text-width", &tw, "text-height", &th, NULL);
	gtk_object_destroy (GTK_OBJECT (item));
	if (tw > max_tw) {
		max_tw = tw;
	}
	if (th > max_th) {
		max_th = th;
	}

	/* Store the result in the return parameters */
	*w = max_tw + 2 * TEXT_BORDER_WIDTH;
	*h = max_th + 2 * TEXT_BORDER_HEIGHT;
	*day_border = (*w - tw) / 2;
}


/*
 * Select the provided day of the provided month
 */
static void
mark_day (calendar_canvas_t *ptr, GDateDay day, guchar month)
{
	day_t *d;

	d = ptr->month_obj[month - 1]->days[day - 1];
	if (d == NULL) {
		return;
	}
	gnome_canvas_item_set (	d->background,
				"fill-color-gdk",
				&(ptr->base_color_selected),
				NULL);
	gnome_canvas_item_set (	d->text,
				"fill-color-gdk",
				&(ptr->text_color_selected),
				NULL);
	d->selected = TRUE;
}


/*
 * Deselect the provided day of the provided month
 */
static void
unmark_day (calendar_canvas_t *ptr, GDateDay day, guchar month)
{
	day_t *d;

	d = ptr->month_obj[month - 1]->days[day - 1];
	if (d == NULL) {
		return;
	}
	gnome_canvas_item_set (	d->background,
				"fill-color-gdk",
				&(ptr->base_color_normal),
				NULL);
	gnome_canvas_item_set (	d->text,
				"fill-color-gdk",
				&(ptr->text_color_normal),
				NULL);
	d->selected = FALSE;
}


/*
 * Select all the days of the provided month
 */
static void
mark_month (calendar_canvas_t *ptr, guchar month)
{
	GDateDay d;

	for (d = 1; d <= 31; d++) {
		mark_day (ptr, d, month);
	}
}


/*
 * Deselect all the days of the provided month
 */
static void
unmark_month (calendar_canvas_t *ptr, guchar month)
{
	GDateDay d;

	for (d = 1; d <= 31; d++) {
		unmark_day (ptr, d, month);
	}
}


/*
 * Select the provided week day in every month
 */
static void
mark_wday (calendar_canvas_t *ptr, GDateWeekday day)
{
	GDate *dt;
	gshort offset;

	dt = g_date_new_dmy (1, 1, ptr->year);
	offset = day - g_date_get_weekday (dt);
	if (offset < 0) {
		offset = 7 + offset;
	}
	g_date_add_days (dt, offset);

	do {
		mark_day (ptr, g_date_get_day (dt), g_date_get_month (dt));
		g_date_add_days (dt, 7);
	} while (g_date_get_year (dt) == ptr->year);
	g_date_free (dt);
}


/*
 * Select the provided week day in every month
 */
static void
unmark_wday (calendar_canvas_t *ptr, GDateWeekday day)
{
	GDate *dt;
	gshort offset;

	dt = g_date_new_dmy (1, 1, ptr->year);
	offset = day - g_date_get_weekday (dt);
	if (offset < 0) {
		offset = 7 + offset;
	}
	g_date_add_days (dt, offset);

	do {
		unmark_day (ptr, g_date_get_day (dt), g_date_get_month (dt));
		g_date_add_days (dt, 7);
	} while (g_date_get_year (dt) == ptr->year);
	g_date_free (dt);
}


/*
 * Callback function called when the user clicks on a day
 */
static gboolean
day_event (	GnomeCanvasItem *day_item,
		GdkEvent *event,
		gpointer user_data)
{
	day_t *ptr = (day_t *)user_data;
	calendar_canvas_t *cal;

	cal = ptr->self_calendar;

	switch (event->type) {
		case GDK_ENTER_NOTIFY:
			if (ptr->selected == FALSE) {
				gnome_canvas_item_set (ptr->background,
						"fill-color-gdk",
						&(cal->base_color_prelight),
						NULL);
				gnome_canvas_item_set (ptr->text,
						"fill-color-gdk",
						&(cal->text_color_prelight),
						NULL);
			}
			return TRUE;

		case GDK_LEAVE_NOTIFY:
			if (ptr->selected == FALSE) {
				gnome_canvas_item_set (ptr->background,
						"fill-color-gdk",
						&(cal->base_color_normal),
						NULL);
				gnome_canvas_item_set (ptr->text,
						"fill-color-gdk",
						&(cal->text_color_normal),
						NULL);
			}
			return TRUE;

		case GDK_BUTTON_PRESS:
			if (ptr->selected == TRUE) {
				if (	   cal->cb_unset_day == NULL
					|| cal->cb_unset_day (
						cal,
						g_date_get_day (ptr->d),
						g_date_get_month (ptr->d),
						cal->cb_unset_day_user_data)
					   == TRUE)
				{
					gnome_canvas_item_set (ptr->background,
						"fill-color-gdk",
						&(cal->base_color_normal),
						NULL);
					gnome_canvas_item_set (ptr->text,
						"fill-color-gdk",
						&(cal->text_color_normal),
						NULL);
					ptr->selected = FALSE;
				}
			}
			else {

				if (	   cal->cb_set_day == NULL
					|| cal->cb_set_day (	cal,
						g_date_get_day (ptr->d),
						g_date_get_month (ptr->d),
						cal->cb_set_day_user_data)
					   == TRUE)
				{
					gnome_canvas_item_set (ptr->background,
						"fill-color-gdk",
						&(cal->base_color_selected),
						NULL);
					gnome_canvas_item_set (ptr->text,
						"fill-color-gdk",
						&(cal->text_color_selected),
						NULL);
					ptr->selected = TRUE;
				}
			}
			return TRUE;

		default:
			return FALSE;
	}
}


/*
 * Callback function called when the user clicks on a month header
 */
static gboolean
month_event (	GnomeCanvasItem *month_item,
		GdkEvent *event,
		gpointer user_data)
{
	month_t *ptr = user_data;
	calendar_canvas_t *cal;

	cal = ptr->self_calendar;

	switch (event->type) {
		case GDK_ENTER_NOTIFY:
			gnome_canvas_item_set (	ptr->background,
						"fill-color-gdk",
						&(cal->base_color_prelight),
						NULL);
			gnome_canvas_item_set (	ptr->text,
						"fill-color-gdk",
						&(cal->text_color_prelight),
						NULL);
			return TRUE;

		case GDK_LEAVE_NOTIFY:
			gnome_canvas_item_set (	ptr->background,
						"fill-color-gdk",
						&(cal->base_color_normal),
						NULL);
			gnome_canvas_item_set (	ptr->text,
						"fill-color-gdk",
						&(cal->text_color_normal),
						NULL);
			return TRUE;

		case GDK_BUTTON_PRESS:

			if ((cal->bit_selected_month & (1 << ptr->month)) != 0)
			{
				if (	   cal->cb_unset_month == NULL
					|| cal->cb_unset_month (
						cal,
						ptr->month,
						cal->cb_unset_month_user_data)
					   == TRUE)
				{
					unmark_month (cal, ptr->month);
				}
				cal->bit_selected_month &= ~(1 << ptr->month);
			}
			else {

				if (	   cal->cb_set_month == NULL
					|| cal->cb_set_month (
						cal,
						ptr->month,
						cal->cb_set_month_user_data)
					   == TRUE)
				{
					mark_month (cal, ptr->month);
				}
				cal->bit_selected_month |= 1 << ptr->month;
			}
			return TRUE;

		default:
			return FALSE;
	}
}


/*
 * Callback function called when the user clicks on a day name
 */
static gboolean
wday_event (	GnomeCanvasItem *wday_item,
		GdkEvent *event,
		gpointer user_data)
{
	calendar_canvas_t *cal = user_data;
	GnomeCanvasItem *fg, *bg;
	guchar wday;

	fg = (GnomeCanvasItem *)g_object_get_data (	G_OBJECT (wday_item),
							"item_fg");
	bg = (GnomeCanvasItem *)g_object_get_data (	G_OBJECT (wday_item),
							"item_bg");
	switch (event->type) {
		case GDK_ENTER_NOTIFY:
			gnome_canvas_item_set (	bg,
						"fill-color-gdk",
						&(cal->base_color_prelight),
						NULL);
			gnome_canvas_item_set ( fg,
						"fill-color-gdk",
						&(cal->text_color_prelight),
						NULL);
			return TRUE;

		case GDK_LEAVE_NOTIFY:
			gnome_canvas_item_set (	bg,
						"fill-color-gdk",
						&(cal->base_color_normal),
						NULL);
			gnome_canvas_item_set ( fg,
						"fill-color-gdk",
						&(cal->text_color_normal),
						NULL);
			return TRUE;

		case GDK_BUTTON_PRESS:
			wday = (guchar)GPOINTER_TO_INT (g_object_get_data (
						G_OBJECT (wday_item), "wday"));

			if ((cal->bit_selected_wday & (1 << wday)) != 0) {
				if (	   cal->cb_unset_wday == NULL
					|| cal->cb_unset_wday (
						cal,
						(wday == 0)	? G_DATE_SUNDAY
								: wday,
						cal->cb_unset_wday_user_data)
					   == TRUE)
				{
					unmark_wday (cal, (wday == 0)
								? G_DATE_SUNDAY
								: wday);
				}
				cal->bit_selected_wday &= ~(1 << wday);
			}
			else {

				if (	   cal->cb_set_wday == NULL
					|| cal->cb_set_wday (
						cal,
						(wday == 0)	? G_DATE_SUNDAY
								: wday,
						cal->cb_set_wday_user_data)
					   == TRUE)
				{
					mark_wday (cal, (wday == 0)
								? G_DATE_SUNDAY
								: wday);
				}
				cal->bit_selected_wday |= 1 << wday;
			}
			return TRUE;

		default:
			return FALSE;
	}
}


/*
 * Initialize the canvas used to display the calendar
 */
static void
init_canvas (calendar_canvas_t *ptr)
{
	gdouble w, h;
	gint num_rows;

	/*
	 * width = width_day * (7 days * MONTHS_BY_ROW months in a row
	 *                        + MONTHS_BY_ROW-1 spaces between months)
	 */
	w =  ptr->w * (ptr->num_month_by_row * 7 + ptr->num_month_by_row - 1);

	/*
	 * height = height_day * (
	 *  (6 week by month + month name header + day names header) * num_rows
	 *  + num_rows-1 spaces between months)
	 */
	num_rows =	  12 / ptr->num_month_by_row
			+ ((12 % ptr->num_month_by_row == 0) ? 0: 1);
	h = ptr->h * (num_rows * 8 + num_rows - 1);

	/* Set the scrollable area */
	gnome_canvas_set_scroll_region (ptr->canvas, 0, 0, w, h);
	gnome_canvas_set_center_scroll_region (ptr->canvas, FALSE);

	/* Draw the background rectangle */
	if (ptr->background != NULL) {
		gtk_object_destroy (GTK_OBJECT (ptr->background));
	}
	ptr->background = gnome_canvas_item_new (
					gnome_canvas_root (ptr->canvas),
					gnome_canvas_rect_get_type (),
					"x1", (gdouble)0,
					"y1", (gdouble)0,
					"x2", w,
					"y2", h,
					"fill-color-gdk",
						&(ptr->base_color_normal),
					NULL);
	gnome_canvas_item_lower_to_bottom (ptr->background);
}


/*
 * Convert a string to a GDateWeekday
 *
 * Return:
 *   The GDateWeekday corresponding to DAY_NAME
 */
static GDateWeekday
str_to_weekday (const gchar *day_name)
{
	if (day_name == NULL) {
		return DEFAULT_FIRST_DAY_OF_WEEK;
	}
	
	if (g_ascii_strncasecmp (day_name, "MON", 3) == 0) {
		return G_DATE_MONDAY;
	}
	if (g_ascii_strncasecmp (day_name, "SUN", 3) == 0) {
		return G_DATE_SUNDAY;
	}
	if (g_ascii_strncasecmp (day_name, "TUE", 3) == 0) {
		return G_DATE_TUESDAY;
	}
	if (g_ascii_strncasecmp (day_name, "WED", 3) == 0) {
		return G_DATE_WEDNESDAY;
	}
	if (g_ascii_strncasecmp (day_name, "THU", 3) == 0) {
		return G_DATE_THURSDAY;
	}
	if (g_ascii_strncasecmp (day_name, "FRI", 3) == 0) {
		return G_DATE_FRIDAY;
	}
	if (g_ascii_strncasecmp (day_name, "SAT", 3) == 0) {
		return G_DATE_SATURDAY;
	}
	return DEFAULT_FIRST_DAY_OF_WEEK;
}


/*
 * Get the day name of the first day of the week from GConf
 *
 * Return:
 *   The first day name of the week
 */
static GDateWeekday
gconf_first_day_of_week ()
{
	gchar *day_name;
	GDateWeekday d;

	day_name = gconf_client_get_string (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/first_day_of_week",
				NULL);

	d = str_to_weekday (day_name);
	g_free (day_name);
	return d;
}


/*
 * Free a day_t object
 */
static void
destroy_day_t (day_t *ptr)
{
	if (ptr != NULL) {
		if (ptr->background != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->background));
		}
		if (ptr->text != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->text));
		}
		if (ptr->d != NULL) {
			g_date_free (ptr->d);
		}
		g_free (ptr);
	}
}
		

/*
 * Free a month_t object.
 */
static void
destroy_month_t (month_t *ptr)
{
	guchar i;

	if (ptr != NULL) {
		for (i = 0; i < 31; i++) {
			destroy_day_t (ptr->days[i]);
		}
		if (ptr->month_group != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->month_group));
		}
		g_free (ptr);
	}
}


/*
 * Destroy the provided calendar_canvas_t object.
 */
void
destroy_calendar_canvas (calendar_canvas_t *ptr)
{
	guchar month;

	if (ptr != NULL) {
		gconf_client_notify_remove (schedwi_gconf, ptr->gconf_notify);
		for (month = 0; month < 12; month++) {
			destroy_month_t (ptr->month_obj[month]);
		}
		g_free (ptr->current_calendar);
		g_free (ptr->font);
		if (ptr->background != NULL) {
			gtk_object_destroy (GTK_OBJECT (ptr->background));
		}
		g_free (ptr);
	}
}


/*
 * Draw the provided day in the provided GnomeCanvasGroup group and return
 * a new day_t object (to be freed by the caller using destroy_day_t())
 */
static day_t *
draw_day (	calendar_canvas_t *ptr, GnomeCanvasGroup *group,
		GDateDay day, guchar month, GDateYear year,
		guint col, guint row)
{
	gchar *str;
	day_t *new_day;
	gdouble x, y;

	/* New day_t object */
	new_day = (day_t *) g_malloc (sizeof (day_t));
	new_day->d = g_date_new_dmy (day, month, year);
	new_day->selected = FALSE;
	new_day->self_calendar = ptr;
	x = ptr->w * col;
	y = ptr->h * (row + 2);

	/* Rectangle, background for the day number */
	new_day->background = gnome_canvas_item_new (
			group,
			gnome_canvas_rect_get_type (),
			"x1", x,
			"y1", y,
			"x2", x + ptr->w,
			"y2", y + ptr->h,
			"fill-color-gdk", &(ptr->base_color_normal),
			"join-style", GDK_JOIN_ROUND,
			"width-units", (gdouble)CALENDAR_SELECT_WIDTH,
			NULL);
	if (ptr->mode != READONLY) {
		g_signal_connect (	(gpointer) new_day->background,
					"event",
					G_CALLBACK (day_event),
					new_day);
	}
	str = g_strdup_printf ("%d", day);
	new_day->text = gnome_canvas_item_new(
				group,
				gnome_canvas_text_get_type (),
				"x", x + ptr->w - ptr->day_border,
				"y", y + row / 2,
				"anchor", GTK_ANCHOR_NORTH_EAST,
				"font", ptr->font,
				"fill-color-gdk", &(ptr->text_color_normal),
				"text", str,
				NULL);
	g_free (str);
	if (ptr->mode != READONLY) {
		g_signal_connect (	(gpointer) new_day->text,
					"event",
					G_CALLBACK (day_event),
					new_day);
	}
	return new_day;
}


/*
 * Draw a month name and the day names
 */
static void
draw_month_headers (month_t *month_ptr)
{
	gdouble x, y;
	char *str;
	guchar day_num;
	GnomeCanvasPoints *points;
	GnomeCanvasItem *item_bg, *item_fg;
	calendar_canvas_t *cal_ptr;

	cal_ptr = month_ptr->self_calendar;

	/* Get the month name */
	switch (month_ptr->month) {
		case 1:
			str = nl_langinfo (MON_1);
			break;
		case 2:
			str = nl_langinfo (MON_2);
			break;
		case 3:
			str = nl_langinfo (MON_3);
			break;
		case 4:
			str = nl_langinfo (MON_4);
			break;
		case 5:
			str = nl_langinfo (MON_5);
			break;
		case 6:
			str = nl_langinfo (MON_6);
			break;
		case 7:
			str = nl_langinfo (MON_7);
			break;
		case 8:
			str = nl_langinfo (MON_8);
			break;
		case 9:
			str = nl_langinfo (MON_9);
			break;
		case 10:
			str = nl_langinfo (MON_10);
			break;
		case 11:
			str = nl_langinfo (MON_11);
			break;
		case 12: default:
			str = nl_langinfo (MON_12);
			break;
	}

	/* Draw the month name background */
	month_ptr->background = gnome_canvas_item_new (
			month_ptr->month_group,
			gnome_canvas_rect_get_type (),
			"x1", (gdouble)0,
			"y1", (gdouble)0,
			"x2", cal_ptr->w * 7,
			"y2", cal_ptr->h,
			"fill-color-gdk", &(cal_ptr->base_color_normal),
			NULL);
	if (cal_ptr->mode != READONLY) {
		g_signal_connect (	(gpointer) month_ptr->background,
					"event",
					G_CALLBACK (month_event),
					month_ptr);
	}

	/* Draw the month name */
	month_ptr->text = gnome_canvas_item_new(
			month_ptr->month_group,
			gnome_canvas_text_get_type (),
			"x", (cal_ptr->w * 7) / 2,
			"y", cal_ptr->h / 2,
			"anchor", GTK_ANCHOR_CENTER,
			"font", cal_ptr->font,
			"fill-color-gdk", &(cal_ptr->text_color_normal),
			"text", str,
			NULL);
	if (cal_ptr->mode != READONLY) {
		g_signal_connect (	(gpointer) month_ptr->text,
					"event",
					G_CALLBACK (month_event),
					month_ptr);
	}

	/*
	 * Draw the day names
	 */
	x = 0;
	y = cal_ptr->h;
	day_num = cal_ptr->first_day_of_week % 7;
	do {
		/* Get the abbreviated day name */
		switch (day_num) {
			case 0:	/* Sunday */
				str = nl_langinfo (ABDAY_1);
				break;
			case 1:	/* Monday */
				str = nl_langinfo (ABDAY_2);
				break;
			case 2:	/* Tuesday */
				str = nl_langinfo (ABDAY_3);
				break;
			case 3:	/* Wednesday */
				str = nl_langinfo (ABDAY_4);
				break;
			case 4:	/* Thursday */
				str = nl_langinfo (ABDAY_5);
				break;
			case 5:	/* Friday */
				str = nl_langinfo (ABDAY_6);
				break;
			case 6:	/* Saturday */
				str = nl_langinfo (ABDAY_7);
				break;
		}

		/* Background for the day name */
		item_bg = gnome_canvas_item_new (
			month_ptr->month_group,
			gnome_canvas_rect_get_type (),
			"x1", x,
			"y1", y,
			"x2", x + cal_ptr->w,
			"y2", y + cal_ptr->h,
			"fill-color-gdk", &(cal_ptr->base_color_normal),
			NULL);

		/* Draw the day name */
		item_fg = gnome_canvas_item_new (
			month_ptr->month_group,
			gnome_canvas_text_get_type (),
			"x", x + cal_ptr->w / 2,
			"y", y + cal_ptr->h / 2,
			"anchor", GTK_ANCHOR_CENTER,
			"font", cal_ptr->font,
			"fill-color-gdk", &(cal_ptr->text_color_normal),
			"text", str,
			NULL);
		if (cal_ptr->mode != READONLY) {
			g_object_set_data (G_OBJECT (item_bg), "wday",
					GINT_TO_POINTER ((gint)day_num));
			g_object_set_data (G_OBJECT (item_bg), "item_fg",
					item_fg);
			g_object_set_data (G_OBJECT (item_bg), "item_bg",
					item_bg);

			g_object_set_data (G_OBJECT (item_fg), "wday",
					GINT_TO_POINTER ((gint)day_num));
			g_object_set_data (G_OBJECT (item_fg), "item_fg",
					item_fg);
			g_object_set_data (G_OBJECT (item_fg), "item_bg",
					item_bg);

			g_signal_connect (	(gpointer) item_fg, "event",
						G_CALLBACK (wday_event),
						cal_ptr);
			g_signal_connect (	(gpointer) item_bg, "event",
						G_CALLBACK (wday_event),
						cal_ptr);
		}

		x += cal_ptr->w;
		day_num = (day_num + 1) % 7;
	} while (day_num != cal_ptr->first_day_of_week % 7);

	/*
	 * Draw the separator line
	 */
	points = gnome_canvas_points_new (2);
	points->coords[0] = (gdouble)0;
	points->coords[1] = points->coords[3] = cal_ptr->h * 2 - 1;
	points->coords[2] = cal_ptr->w * 7;
	gnome_canvas_item_new(
			month_ptr->month_group,
			gnome_canvas_line_get_type (),
			"points", points,
			"fill-color-gdk", &(cal_ptr->text_color_normal),
			"width_units", (double)1,
			NULL);
	gnome_canvas_points_free(points);
}


/*
 * Draw the provided month in the provided year and return the associated
 * month_t object (to be freed by destroy_month_t())
 */
static month_t *
draw_month (calendar_canvas_t *ptr, guchar month, GDateYear year)
{
	month_t *new_month;
	gdouble x, y;
	GDate *d;
	guint col, row;
	GDateDay day_month;
	gshort offset;

	/*
	 * Create the new object
	 */
	new_month = (month_t *) g_malloc (sizeof (month_t));
	new_month->month = month;
	new_month->self_calendar = ptr;

	/* Position of the month group */
	x = ((month - 1) % ptr->num_month_by_row) * ptr->w * 8;
	y = ((month - 1) / ptr->num_month_by_row) * ptr->h * 9;

	/*
	 * Month group creation
	 */
	new_month->month_group = (GnomeCanvasGroup *)gnome_canvas_item_new (
					gnome_canvas_root (ptr->canvas),
					gnome_canvas_group_get_type (),
					"x", x,
					"y", y,
					NULL);

	/*
	 * Draw the headers (month name and day names)
	 */
	draw_month_headers (new_month);

	
	d = g_date_new_dmy (1, month, year);
	offset = g_date_get_weekday (d) - ptr->first_day_of_week;
	if (offset < 0) {
		offset = 7 + offset;
	}

	/*
	 * Draw days
	 */
	while (1) {
		day_month = g_date_get_day (d);
		col = (offset + day_month - 1) % 7;
		row = (offset + day_month - 1) / 7;
		new_month->days[day_month - 1] = draw_day (
							ptr,
							new_month->month_group,
							day_month, month, year,
							col, row);
		if (g_date_is_last_of_month (d) == TRUE) {
			break;
		}
		g_date_add_days (d, 1);
	}
	g_date_free (d);

	/* Set the days from the last day of the month to the day 31 to NULL */
	while (day_month < 31) {
		new_month->days[day_month++] = NULL;
	}
	return new_month;
}


/*
 * Draw the calendar for the provided year
 */
void
draw_year_canvas (calendar_canvas_t *ptr, GDateYear year)
{
	guchar month;

	for (month = 0; month < 12; month++) {
		destroy_month_t (ptr->month_obj[month]);
		ptr->month_obj[month] = draw_month (ptr, month + 1, year);
	}
	ptr->year = year;
	set_calendar (ptr, ptr->current_calendar, NULL);
}


/*
 * GConf notify function for the calendar keys
 */
static void
calendar_notify (	GConfClient *client, guint cnxn_id,
			GConfEntry *entry, gpointer user_data)
{
	calendar_canvas_t *ptr = user_data;

	if (strcmp(	entry->key,
			GCONF_PATH "/ui/calendar/first_day_of_week") == 0)
	{
		ptr->first_day_of_week = str_to_weekday (
					gconf_value_get_string (entry->value));
		draw_year_canvas (ptr, ptr->year);
		return;
	}

	if (strcmp(entry->key, GCONF_PATH "/ui/calendar/months_by_row") == 0) {
		ptr->num_month_by_row = gconf_value_get_int (entry->value);
		if (ptr->num_month_by_row == 0) {
			ptr->num_month_by_row = DEFAULT_MONTHS_BY_ROW;
		}
		init_canvas (ptr);
		draw_year_canvas (ptr, ptr->year);
		return;
	}

	if (strcmp(entry->key, GCONF_PATH "/ui/calendar/use_system_font") == 0)
	{
		g_free (ptr->font);
		if (gconf_value_get_bool (entry->value) == TRUE) {
			ptr->font = NULL;
		}
		else {
			ptr->font = gconf_client_get_string (
						client,
						GCONF_PATH "/ui/calendar/font",
						NULL);
		}
		get_day_size (	ptr->canvas, ptr->font,
				&(ptr->w), &(ptr->h), &(ptr->day_border));
		init_canvas (ptr);
		draw_year_canvas (ptr, ptr->year);
		return;
	}
	if (	   strcmp(entry->key, GCONF_PATH "/ui/calendar/font") == 0
		&& gconf_client_get_bool (
				client,
				GCONF_PATH "/ui/calendar/use_system_font",
				NULL) == FALSE)
	{
		g_free (ptr->font);
		ptr->font = g_strdup (gconf_value_get_string (entry->value));
		get_day_size (	ptr->canvas, ptr->font,
				&(ptr->w), &(ptr->h), &(ptr->day_border));
		init_canvas (ptr);
		draw_year_canvas (ptr, ptr->year);
		return;
	}
}


/*
 * Create and return a new calendar_canvas_t object.
 * This object must be freed by destroy_calendar_canvas()
 */
calendar_canvas_t *
new_calendar_canvas (GnomeCanvas *canvas, calendar_canvas_mode_t mode)
{
	calendar_canvas_t *ptr;
	guchar month;

	ptr = (calendar_canvas_t *) g_malloc (sizeof (calendar_canvas_t));
	ptr->canvas = canvas;
	ptr->mode = mode;
	ptr->current_calendar = NULL;
	ptr->background = NULL;
	ptr->bit_selected_wday = 0;
	ptr->bit_selected_month = 0;
	ptr->cb_set_day = NULL;
	ptr->cb_set_day_user_data = NULL;
	ptr->cb_unset_day = NULL;
	ptr->cb_unset_day_user_data = NULL;
	ptr->cb_set_wday = NULL;
	ptr->cb_set_wday_user_data = NULL;
	ptr->cb_unset_wday = NULL;
	ptr->cb_unset_wday_user_data = NULL;
	ptr->cb_set_month = NULL;
	ptr->cb_set_month_user_data = NULL;
	ptr->cb_unset_month = NULL;
	ptr->cb_unset_month_user_data = NULL;

	schedwi_style_get_text_color (	GTK_STATE_NORMAL,
					&(ptr->text_color_normal));
	schedwi_style_get_text_color (	GTK_STATE_PRELIGHT,
					&(ptr->text_color_prelight));
	schedwi_style_get_text_color (	GTK_STATE_SELECTED,
					&(ptr->text_color_selected));
	schedwi_style_get_base_color (	GTK_STATE_NORMAL,
					&(ptr->base_color_normal));
	schedwi_style_get_base_color (	GTK_STATE_PRELIGHT,
					&(ptr->base_color_prelight));
	schedwi_style_get_base_color (	GTK_STATE_SELECTED,
					&(ptr->base_color_selected));

	ptr->first_day_of_week = gconf_first_day_of_week ();

	if (gconf_client_get_bool (schedwi_gconf,
				GCONF_PATH "/ui/calendar/use_system_font",
				NULL) == TRUE)
	{
		ptr->font = NULL;
	}
	else {
		ptr->font = gconf_client_get_string (
						schedwi_gconf,
						GCONF_PATH "/ui/calendar/font",
						NULL);
	}

	ptr->num_month_by_row = gconf_client_get_int (
				schedwi_gconf,
				GCONF_PATH "/ui/calendar/months_by_row",
				NULL);
	if (ptr->num_month_by_row == 0) {
		ptr->num_month_by_row = DEFAULT_MONTHS_BY_ROW;
	}

	get_day_size (	canvas, ptr->font,
			&(ptr->w), &(ptr->h), &(ptr->day_border));
	init_canvas (ptr);

	for (month = 0; month < 12; month++) {
		ptr->month_obj[month] = NULL;
	}

	ptr->gconf_notify = gconf_client_notify_add (schedwi_gconf,
						GCONF_PATH "/ui/calendar",
						calendar_notify,
						ptr,
						NULL,
						NULL);
	return ptr;
}


/*
 * Define the callback function to call when a day is selected by the user.
 * This callback function must return TRUE if the day must be selected and
 * FALSE if not
 */
void
set_callbacks_set_day (calendar_canvas_t *ptr,
	gboolean (*f)(calendar_canvas_t *, GDateDay, GDateMonth, gpointer),
	gpointer user_data)
{
	if (ptr != NULL) {
		ptr->cb_set_day = f;
		ptr->cb_set_day_user_data = user_data;
	}
}


/*
 * Define the callback function to call when a day is deselected by the user.
 * This callback function must return TRUE if the day must be deselected and
 * FALSE if not
 */
void
set_callbacks_unset_day (calendar_canvas_t *ptr,
	gboolean (*f)(calendar_canvas_t *, GDateDay, GDateMonth, gpointer),
	gpointer user_data)
{
	if (ptr != NULL) {
		ptr->cb_unset_day = f;
		ptr->cb_unset_day_user_data = user_data;
	}
}


/*
 * Define the callback function to call when a week day is selected
 * by the user.
 * This callback function must return TRUE if the week day must be selected and
 * FALSE if not
 */
void
set_callbacks_set_wday (calendar_canvas_t *ptr,
		gboolean (*f)(calendar_canvas_t *, GDateWeekday, gpointer),
		gpointer user_data)
{
	if (ptr != NULL) {
		ptr->cb_set_wday = f;
		ptr->cb_set_wday_user_data = user_data;
	}
}


/*
 * Define the callback function to call when a week day is deselected
 * by the user.
 * This callback function must return TRUE if the week day must be deselected
 * and FALSE if not
 */
void
set_callbacks_unset_wday (calendar_canvas_t *ptr,
		gboolean (*f)(calendar_canvas_t *, GDateWeekday, gpointer),
		gpointer user_data)
{
	if (ptr != NULL) {
		ptr->cb_unset_wday = f;
		ptr->cb_unset_wday_user_data = user_data;
	}
}


/*
 * Define the callback function to call when a month is selected
 * by the user.
 * This callback function must return TRUE if the month must be selected and
 * FALSE if not
 */
void
set_callbacks_set_month (calendar_canvas_t *ptr,
		gboolean (*f)(calendar_canvas_t *, GDateMonth, gpointer),
		gpointer user_data)
{
	if (ptr != NULL) {
		ptr->cb_set_month = f;
		ptr->cb_set_month_user_data = user_data;
	}
}


/*
 * Define the callback function to call when a month is deselected
 * by the user.
 * This callback function must return TRUE if the month must be deselected and
 * FALSE if not
 */
void
set_callbacks_unset_month (calendar_canvas_t *ptr,
		gboolean (*f)(calendar_canvas_t *, GDateMonth, gpointer),
		gpointer user_data)
{
	if (ptr != NULL) {
		ptr->cb_unset_month = f;
		ptr->cb_unset_month_user_data = user_data;
	}
}


/*
 * Associate a calendar formula to the calendar_canvas object.  The days
 * corresponding to the provided calendar are highlighted (selected)
 *
 * Return:
 *   The return code is the one returned by str2cal().  In particular
 *   CAL_NOERROR means no error
 */
cal_errcode_t
set_calendar (	calendar_canvas_t *ptr, const gchar *calendar,
		int *error_idx_in_str)
{
	guchar month;
	GDateDay day;
	cal_errcode_t ret;
	cal_t cal;

	if (calendar == NULL) {
		clear_calendar (ptr);
		return CAL_NOERROR;
	}

	/* Compile the formula string */
	ret = str2cal (&cal, calendar, ptr->year, error_idx_in_str);
	if (ret != CAL_NOERROR) {
		clear_calendar (ptr);
		return ret;
	}

	/* Highlight only the days matching the calendar */
	for (month = 1; month <= 12; month++) {
		for (day = 1; day <=31; day++) {
			if (calmatch (&cal, day, month, ptr->year)
				!= CAL_NOMATCH)
			{
				mark_day (ptr, day, month);
			}
			else {
				unmark_day (ptr, day, month);
			}
		}
	}
	if (ptr->current_calendar != calendar) {
		g_free (ptr->current_calendar);
		ptr->current_calendar = g_strdup (calendar);
	}
	return CAL_NOERROR;
}


/*
 * Remove all the selected day and the association between the calendar and
 * calendar_canvas object
 */
void
clear_calendar (calendar_canvas_t *ptr)
{
	guchar month;
	GDateDay day;

	for (month = 1; month <= 12; month++) {
		for (day = 1; day <=31; day++) {
			unmark_day (ptr, day, month);
		}
	}
	ptr->bit_selected_wday = 0;
	ptr->bit_selected_month = 0;
	g_free (ptr->current_calendar);
	ptr->current_calendar = NULL;
}

/*-----------------============== End Of File ==============-----------------*/
