/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* database_test_cb.c -- GUI functions for the test database window */

#include <schedwi.h>

#include <schedwi_interface.h>

#include <cursor.h>
#include <message_windows.h>
#include <schedwi_gconf.h>
#include <sql_test.h>
#include <database_test_cb.h>


/*
 * Structure used to exchange widgets pointers between callbacks functions
 */
struct database_connection_status_widgets {
	GtkLabel *label_status[SQL_TEST_NUM];
	GtkTextBuffer *message_buffer;
};
typedef struct database_connection_status_widgets
					database_connection_status_widgets_t;


/*
 * Fill the fields with the status of the connection tests
 */
static void
database_test_status (	sql_test_step_t step,
			char status, const char *err_msg, void *user_data)
{
	database_connection_status_widgets_t *widgets = 
			(database_connection_status_widgets_t *)user_data;
	GtkTextIter end;
	gchar *s;

	gtk_label_set_text (	widgets->label_status[step],
				(status == 0) ? _("Done") : _("Failed"));

	if (err_msg != NULL) {
		gtk_text_buffer_get_end_iter (widgets->message_buffer, &end);
		s = g_strdup_printf ("%s\n", err_msg);
		gtk_text_buffer_insert (widgets->message_buffer, &end, s, -1);
		g_free (s);
	}
}


/*
 * Run a database test for the assistant/druid window
 *
 * Return:
 *   0 --> Success
 *  -1 --> Failure
 */
int
database_test_go (GtkWidget *w)
{
	GtkTextIter start, end;
	const gchar *host, *user, *password, *database;
	gchar *socket_file;
	gint port;
	database_connection_status_widgets_t widgets;
	int ret;

	cursor_busy (w);

	/* Retrieve the widgets */
	widgets.message_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (
				lookup_widget (w, "textview_druid_result")));
	widgets.label_status[SQL_TEST_CONNECT] = (GtkLabel *)
			lookup_widget (w, "label_druid_status_connect");
	widgets.label_status[SQL_TEST_DATABASE] = (GtkLabel *)
			lookup_widget (w, "label_druid_status_database");
	widgets.label_status[SQL_TEST_VERSION] = (GtkLabel *)
			lookup_widget (w, "label_druid_status_version");

	/* Clear the result fields */
	gtk_label_set_text (widgets.label_status[SQL_TEST_CONNECT], "");
	gtk_label_set_text (widgets.label_status[SQL_TEST_DATABASE], "");
	gtk_label_set_text (widgets.label_status[SQL_TEST_VERSION], "");
	gtk_text_buffer_get_start_iter (widgets.message_buffer, &start);
	gtk_text_buffer_get_end_iter (widgets.message_buffer, &end);
	gtk_text_buffer_delete (widgets.message_buffer, &start, &end);

	/* Retrieve the database connection parameters from the druid */
	host = gtk_entry_get_text ((GtkEntry *)lookup_widget (w,
							"entry_druid_host")); 
	if (host != NULL && host[0] == '\0') {
		host = NULL;
	}

	user = gtk_entry_get_text ((GtkEntry *)lookup_widget (w,
							"entry_druid_user")); 
	if (user != NULL && user[0] == '\0') {
		user = NULL;
	}

	password = gtk_entry_get_text ((GtkEntry *)lookup_widget (w,
						"entry_druid_password")); 
	if (password != NULL && password[0] == '\0') {
		password = NULL;
	}

	database = gtk_entry_get_text ((GtkEntry *)lookup_widget (w,
						"entry_druid_database")); 
	if (database != NULL && database[0] == '\0') {
		database = NULL;
	}

	port = gtk_spin_button_get_value_as_int (
		(GtkSpinButton *)lookup_widget (w, "spinbutton_druid_port"));

	socket_file = gtk_file_chooser_get_filename ((GtkFileChooser *)
				lookup_widget (w, "fileentry_druid_socket"));

/*
	socket_file = gnome_file_entry_get_full_path ((GnomeFileEntry *)
				lookup_widget (w, "fileentry_druid_socket"),
				FALSE);
*/

	/* Run the test */
	ret = sql_test (host, user, password, database,
			(long int)port,
			(socket_file != NULL && socket_file[0] != '\0')
				? socket_file: NULL,
			database_test_status, &widgets);
	g_free (socket_file);
	cursor_normal (w);
	return ret;
}


/*
 * Callback for the `TEST' button
 */
void
database_test_go_clicked (GtkButton *button)
{
	GtkWidget *password_entry;
	GtkTextIter start, end;
	gchar *host, *user, *database, *socket_file;
	const gchar *password;
	gint port;
	database_connection_status_widgets_t widgets;
	int ret;

	cursor_busy (GTK_WIDGET (button));

	/* Retrieve the widgets */
	password_entry = lookup_widget (GTK_WIDGET (button),
					"entry_database_test_password");
	widgets.message_buffer = gtk_text_view_get_buffer (GTK_TEXT_VIEW (
			lookup_widget (	GTK_WIDGET (button),
					"textview_database_test_result")));
	widgets.label_status[SQL_TEST_CONNECT] = (GtkLabel *)
			lookup_widget (	GTK_WIDGET (button),
					"label_database_test_status_connect");
	widgets.label_status[SQL_TEST_DATABASE] = (GtkLabel *)
			lookup_widget (	GTK_WIDGET (button),
					"label_database_test_status_database");
	widgets.label_status[SQL_TEST_VERSION] = (GtkLabel *)
			lookup_widget (	GTK_WIDGET (button),
					"label_database_test_status_version");

	/* Clear the result fields */
	gtk_label_set_text (widgets.label_status[SQL_TEST_CONNECT], "");
	gtk_label_set_text (widgets.label_status[SQL_TEST_DATABASE], "");
	gtk_label_set_text (widgets.label_status[SQL_TEST_VERSION], "");
	gtk_text_buffer_get_start_iter (widgets.message_buffer, &start);
	gtk_text_buffer_get_end_iter (widgets.message_buffer, &end);
	gtk_text_buffer_delete (widgets.message_buffer, &start, &end);


	/* Retrieve the database connection parameters from gconf */
	host = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/host",
					NULL);
	user = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/username",
					NULL);
	database = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/database",
					NULL);
	socket_file = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/socket",
					NULL);
	port = gconf_client_get_int (	schedwi_gconf,
					GCONF_PATH "/database/port",
					NULL);
	password = gtk_entry_get_text (GTK_ENTRY (password_entry));
	
	/* Run the test */
	ret = sql_test ((host != NULL && host[0] != '\0')? host: NULL,
			(user != NULL && user[0] != '\0')? user: NULL ,
			(password != NULL && password[0] != '\0')
				? password: NULL,
			(database != NULL && database[0] != '\0')
				? database: NULL,
			(long int)port,
			(socket_file != NULL && socket_file[0] != '\0')
				? socket_file: NULL,
			database_test_status, &widgets);
	g_free (host);
	g_free (user);
	g_free (database);
	g_free (socket_file);
	cursor_normal (GTK_WIDGET (button));
	if (ret == 0) {
		info_window (_("Database Connection successful"), NULL);
	}
	else {
		warning_window (_("Database Connection failed"), NULL);
	}
}


/*
 * Initialize the new dialog
 */
static gint
database_test_init (GtkWidget *dialog)
{
	GtkWidget *username_entry;
	gchar *user;

	/*
	 * Fill the username entry
	 */
	username_entry = lookup_widget (dialog,
					"entry_database_test_username");
	user = gconf_client_get_string (schedwi_gconf,
					GCONF_PATH "/database/username",
					NULL);
	if (user != NULL) {
		gtk_entry_set_text (GTK_ENTRY (username_entry), user);
		g_free (user);
	}
	gtk_widget_set_sensitive (username_entry, FALSE);

	return 0;
}


/*
 * Create a new database test dialog
 *
 * Return:
 *   The GtkWidget of the new window (which has been `show'ed by this function)
 *   or NULL in case of error
 */
GtkWidget *
new_dialog_database_test ()
{
	GtkWidget *dialog_test;

	dialog_test = create_dialog_database_test ();
	if (database_test_init (dialog_test) != 0) {
		gtk_widget_destroy (dialog_test);
		return NULL;
	}
	else {
		gtk_widget_show (dialog_test);
		return dialog_test;
	}
}

/*-----------------============== End Of File ==============-----------------*/
