/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* message_windows.c - Message dialog windows */

#include <schedwi.h>
#include <sql_common.h>
#include <message_windows.h>

/*
 * Error window
 *
 * primary_text (if not NULL) is displayed in large bold characters
 * error_message (if not NULL) is displayed in regular characters
 */
void
error_window (const char *primary_text, const char *error_message)
{
	GtkWidget *dialog_error;
	gchar *markup;

	if (primary_text == NULL || primary_text[0] == '\0') {
		markup = g_strdup (error_message);
	}
	else {
		if (error_message == NULL || error_message[0] == '\0') {
			markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>",
					primary_text);
		}
		else {
			markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
					primary_text,
					error_message);
		}
	}
	dialog_error = gtk_message_dialog_new (NULL,
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_ERROR,
						GTK_BUTTONS_OK,
						NULL);
	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dialog_error),
					(markup == NULL) ? "": markup);
	gtk_dialog_run (GTK_DIALOG (dialog_error));
	gtk_widget_destroy (dialog_error);
	g_free (markup);
}


/*
 * Wrapper for the error_window() function.  This function is the error
 * function callback of the sql_* functions
 */
void
error_window_ignore_errno (	const char *primary_text,
				const char *error_message,
				unsigned int sql_errno)
{
	error_window (primary_text, error_message);
}


/*
 * SQL error callback function
 */
void
error_window_check_duplicate (	const char *primary_text,
				const char *error_message,
				unsigned int sql_errno)
{
	if (	   sql_errno != ER_DUP_KEY
		&& sql_errno != ER_DUP_ENTRY
		&& sql_errno != ER_DUP_UNIQUE)
	{
		error_window (primary_text, error_message);
	}
	else {
		error_window (
			_("Save failed"),
			_("An object with the same name already exists."));
	}
}


/*
 * Information window
 *
 * primary_text (if not NULL) is displayed in large bold characters
 * error_message (if not NULL) is displayed in regular characters
 */
void
info_window (const char *primary_text, const char *info_message)
{
	GtkWidget *dialog_info;
	gchar *markup;

	if (primary_text == NULL || primary_text[0] == '\0') {
		markup = g_strdup (info_message);
	}
	else {
		if (info_message == NULL || info_message[0] == '\0') {
			markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>",
					primary_text);
		}
		else {
			markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
					primary_text,
					info_message);
		}
	}
	dialog_info = gtk_message_dialog_new (NULL,
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_INFO,
						GTK_BUTTONS_OK,
						NULL);
	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dialog_info),
					(markup == NULL) ? "": markup);
	gtk_dialog_run (GTK_DIALOG (dialog_info));
	gtk_widget_destroy (dialog_info);
	g_free (markup);
}


/*
 * Warning window
 *
 * primary_text (if not NULL) is displayed in large bold characters
 * error_message (if not NULL) is displayed in regular characters
 */
void
warning_window (const char *primary_text, const char *warning_message)
{
	GtkWidget *dialog_warning;
	gchar *markup;

	if (primary_text == NULL || primary_text[0] == '\0') {
		markup = g_strdup (warning_message);
	}
	else {
		if (warning_message == NULL || warning_message[0] == '\0') {
			markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>",
					primary_text);
		}
		else {
			markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
					primary_text,
					warning_message);
		}
	}
	dialog_warning = gtk_message_dialog_new (NULL,
						GTK_DIALOG_DESTROY_WITH_PARENT,
						GTK_MESSAGE_WARNING,
						GTK_BUTTONS_OK,
						NULL);
	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dialog_warning),
					(markup == NULL) ? "": markup);
	gtk_dialog_run (GTK_DIALOG (dialog_warning));
	gtk_widget_destroy (dialog_warning);
	g_free (markup);
}


/*
 * Delete confirmation popup
 *
 * Return:
 *    TRUE --> Carry on the delete operation
 *   FALSE --> Cancel the delete operation
 */
gboolean
question_delete_window (GtkWidget *parent, const char *message)
{
	GtkWidget *dialog;
	gint ret;
	gchar *markup;

	markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>",
			message);

	dialog = gtk_message_dialog_new (
					GTK_WINDOW (parent),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_NONE,
					NULL);
	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dialog), markup);
	
	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				GTK_STOCK_CANCEL,
				GTK_RESPONSE_NO,
				GTK_STOCK_DELETE,
				GTK_RESPONSE_YES,
				NULL);
	ret = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	g_free (markup);
	if (ret == GTK_RESPONSE_YES) {
		return TRUE;
	}
	return FALSE;
}


/*
 * Overwrite confirmation popup
 *
 * Return:
 *    TRUE --> Carry on the replace operation
 *   FALSE --> Cancel the replace operation
 */
gboolean
question_replace_window (	GtkWidget *parent,
				const char *primary_text,
				const char *secondary_text)
{
	GtkWidget *dialog;
	gint ret;
	gchar *markup;

	if (secondary_text == NULL) {
		markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>",
			primary_text);

	}
	else {
		markup = g_markup_printf_escaped (
		"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
			primary_text, secondary_text);
	}

	dialog = gtk_message_dialog_new (
					GTK_WINDOW (parent),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_NONE,
					NULL);
	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dialog), markup);
	
	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				GTK_STOCK_CANCEL,
				GTK_RESPONSE_NO,
				_("Replace"),
				GTK_RESPONSE_YES,
				NULL);
	ret = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	g_free (markup);
	if (ret == GTK_RESPONSE_YES) {
		return TRUE;
	}
	return FALSE;
}


/*
 * Question/Confirmation popup
 *
 * Return:
 *    TRUE --> Carry on the operation
 *   FALSE --> Cancel the operation
 */
gboolean
question_window (	GtkWidget *parent,
			const char *primary_text,
			const char *secondary_text)
{
	GtkWidget *dialog;
	gint ret;
	gchar *markup;

	if (secondary_text == NULL) {
		markup = g_markup_printf_escaped (
			"<span weight=\"bold\" size=\"larger\">%s</span>",
			primary_text);

	}
	else {
		markup = g_markup_printf_escaped (
		"<span weight=\"bold\" size=\"larger\">%s</span>\n\n%s",
			primary_text, secondary_text);
	}

	dialog = gtk_message_dialog_new (
					GTK_WINDOW (parent),
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_MESSAGE_QUESTION,
					GTK_BUTTONS_YES_NO,
					NULL);
	gtk_message_dialog_set_markup (GTK_MESSAGE_DIALOG (dialog), markup);
	ret = gtk_dialog_run (GTK_DIALOG (dialog));
	gtk_widget_destroy (dialog);
	g_free (markup);
	if (ret == GTK_RESPONSE_YES) {
		return TRUE;
	}
	return FALSE;
}

/*-----------------============== End Of File ==============-----------------*/
