/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_hierarchy.c -- Get the job hierarchy and parameters */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_hierarchy.h>

#define SQL_GET_JOB_NAME "SELECT name,parent FROM job_main_s WHERE workload_date=%d AND id=\"%s\"" 
#define SQL_GET_PARENT "SELECT parent FROM job_main_s WHERE workload_date=%d AND id=\"%s\""
#define SQL_GET_JOB_PARAMS "SELECT %s FROM %s WHERE workload_date=%d AND job_id=\"%s\""
#define SQL_GET_JOB_PARAMS_TIME "SELECT %s FROM %s WHERE workload_date=%d AND id=\"%s\""
#define SQL_GET_JOB_PARAMS_SORTED "SELECT %s FROM %s WHERE workload_date=%d AND job_id=\"%s\" ORDER BY %s"
#define SQL_GET_JOB_ID "SELECT id,parent,name,type FROM job_main_s WHERE workload_date=%d AND name IN (%s)"


/*
 * Prepend the string s to the begining of *str (and add a / at the beginning
 * of the returned string)
 *
 * If *str is NULL or not large enough, a new string will be allocated and
 * returned in *str (*str_len is set with the new lenght)
 *
 * Return:
 *   0 --> No error (*str contains the result and must be freed by the caller)
 *  -1 --> Memory allocation error (*str is not modified)
 */
static int
prepend_path (char **str, unsigned int *str_len, const char *s)
{
	unsigned int len_s, len_str;
	int i;
	char *tmp;

#if HAVE_ASSERT_H
	assert (str != NULL && str_len != NULL && s != NULL && s[0] != '\0');
#endif

	len_s = schedwi_strlen (s);

	/*
	 * If *str is empty, simply copy s (with a prepended /) in a new string
	 */
	if (*str == NULL || *str_len == 0) {
		*str = (char *) malloc (len_s  + 100);
		if (*str == NULL) {
			*str_len = 0;
			return -1;
		}
		*str_len = len_s  + 100;
		(*str)[0] = '/';
		strcpy (*str + 1, s);
		return 0;
	}

	len_str = schedwi_strlen (*str);

	/* *str is not big enough.  Allocate a new string */
	if (*str_len < len_s + len_str + 2) {
		tmp = (char *) malloc (len_s + len_str + 100);
		if (tmp == NULL) {
			return -1;
		}
		*str_len = len_s + len_str + 100;
		tmp[0] = '/';
		strcpy (tmp + 1, s);
		strcpy (tmp + len_s + 1, *str);
		free (*str);
		*str = tmp;
		return 0;
	}

	/* Shift the string in *str to allow room to copy s */
	(*str)[len_str + len_s + 1] = '\0';
	for (i = len_str - 1; i >= 0; i--) {
		(*str)[i + len_s + 1] = (*str)[i];
	}
	(*str)[0] = '/';
	for (i = 0; i < len_s; i++) {
		(*str)[i + 1] = s[i];
	}
	return 0;
}


/*
 * Recursively construct the path (hierarchy) of a job
 *
 * Return:
 *     0 --> No error.  path is set and must be freed by the caller (free()).
 *           The working buffer buf may have been set by this function and
 *           must also be freed by the caller.
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *           path and buf must be freed by the caller.
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *           path and buf must be freed by the caller.
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 *           path and buf must be freed by the caller.
 */
static unsigned int
get_job_path_recursive (int workload_date, const char *job_id,
			char **buf, unsigned int *buf_len,
			char **path, unsigned int *path_len,
			char **err_msg)
{
	unsigned int ret;
	lwc_LL *rows;
	char **row;
	char *s;

#if HAVE_ASSERT_H
	assert (job_id != NULL && path != NULL && path_len != NULL);
#endif

	/* Get the job name and its parent */
	ret = sql_select (	buf, buf_len, err_msg, NULL, &rows, NULL,
				SQL_GET_JOB_NAME,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_END);
	if (ret != 0) {
		return ret;
	}

	/* Retrieve the result (only one row) */
	row = (char **) lwc_delStartLL (rows);
	if (row == NULL) {
		/*
		 * Nothing found - The root of the hieararchy
		 * has been reached
		 */
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		return 0;
	}
	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	/* Prepend the job name to the path */
	if (prepend_path (path, path_len, row[0]) != 0) {
		sql_free_row (row);
		if (err_msg != NULL) {
			s = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (s) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, s);
			}
		}
		return 1;
	}

	/* Recursively retrieve the parent jobset */
	ret = get_job_path_recursive (	workload_date, row[1],
					buf, buf_len, path, path_len, err_msg);
	sql_free_row (row);

	return ret;
}


/*
 * Get the path (hierarchy) of a job
 *
 * Return:
 *     0 --> No error.  path is set and must be freed by the caller (free()).
 *           If path is NULL then the provided job_id is not in the database
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 */
unsigned int
get_job_path (	int workload_date, const char *job_id,
		char **path, char **err_msg)
{
	MYSQL *sql;
	unsigned int ret;
	char *buf, *out;
	unsigned int buf_len, out_len;

#if HAVE_ASSERT_H
	assert (job_id != NULL && path != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		return 2;
	}

	/* Build the path */
	out = buf = NULL;
	out_len = buf_len = 0;
	ret = get_job_path_recursive (	workload_date, job_id,
					&buf, &buf_len,
					&out, &out_len, err_msg);
	if (ret == 0) {
		*path = out;
	}
	else {
		if (out != NULL) {
			free (out);
		}
	}

	/* Disconnect from the database */
	end_mysql ();

	if (buf != NULL) {
		free (buf);
	}

	return ret;
}


/*
 * Get the hierarchy list of a job.  This list must be passed as a parameter
 * to the get_job_parameters() and get_job_parameter() functions and must be
 * freed by the caller.
 *
 * Return:
 *     The list to be freed by the caller using
 *		lwc_delLL (lst, (void (*)(const void *))free);
 *     or NULL in case of error (if err_msg is not NULL it contains an
 *     error message which must be freed by the caller.  *err_msg may be
 *     NULL if there is not enough memory to store the error message)
 */
lwc_LL *
get_hierarchy_list (int workload_date, const char *job_id, char **err_msg)
{
	MYSQL *sql;
	unsigned int ret;
	lwc_LL *lst, *rows;
	char **row;
	char *id, *buf, *s;
	unsigned int buf_len;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif

	/* Create the result list */
	lst = lwc_newLL ();
	if (lst == NULL) {
		if (err_msg != NULL) {
			s = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (s) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, s);
			}
		}
		return NULL;
	}

	/* Initialize the first id to look in the database */
	id = (char *) malloc (schedwi_strlen (job_id) + 1);
	if (id == NULL || lwc_addEndLL (lst, id) != 0) {
		if (id != NULL) {
			free (id);
		}
		lwc_delLL (lst, NULL);
		if (err_msg != NULL) {
			s = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (s) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, s);
			}
		}
		return NULL;
	}
	strcpy (id, job_id);

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		lwc_delLL (lst, (void (*)(const void *))free);
		return NULL;
	}

	buf = NULL;
	buf_len = 0;
	while (1) {
		/* Get the parent of the job */
		ret = sql_select (&buf, &buf_len, err_msg, NULL, &rows, NULL,
				SQL_GET_PARENT,
				SQL_INT, (long int)workload_date,
				SQL_STRING, id,
				SQL_END);
		if (ret != 0) {
			end_mysql ();
			lwc_delLL (lst, (void (*)(const void *))free);
			if (buf != NULL) {
				free (buf);
			}
			return NULL;
		}

		/* Retrieve the result (only one row) */
		row = (char **) lwc_delStartLL (rows);
		if (row == NULL) {
			/*
			 * Nothing found - The root of the hierarchy
			 * has been reached
			 */
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			break;
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

		/* Copy and add the parent id to the list */
		id = (char *) malloc (schedwi_strlen (row[0]) + 1);
		if (id == NULL || lwc_addEndLL (lst, id) != 0) {
			end_mysql ();
			if (id != NULL) {
				free (id);
			}
			lwc_delLL (lst, (void (*)(const void *))free);
			sql_free_row (row);
			if (buf != NULL) {
				free (buf);
			}
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return NULL;
		}
		strcpy (id, row[0]);
		sql_free_row (row);
	}

	/* Close the database and free the working buffer */
	end_mysql ();
	if (buf != NULL) {
		free (buf);
	}

	/* Return the list */
	return lst;
}


/*
 * Convert a hierarchy list to a string of job IDs separated by commas
 *
 * Return:
 *   0 --> No error.  result is set and must be freed by the caller by free()
 *  -1 --> Memory allocation error
 */
int
hierarchy_list_to_str (lwc_LL *lst, char **result)
{
	char *s, *id;
	unsigned int i;

	if (lwc_getNumNode (lst) <= 0 || result == NULL) {
		return 0;
	}

	s = (char *) malloc (lwc_getNumNode (lst) * 26);
	if (s == NULL) {
		return -1;
	}

	i = 0;
	lwc_rewindLL (lst);
	while ((id = (char *)lwc_nextLL (lst)) != NULL) {
		strcpy (s + i, id);
		i += schedwi_strlen (s + i);
		s[i++] = ',';
	}
	if (i > 0) {
		s[i - 1] = '\0';
	}
	*result = s;
	return 0;
}


/*
 * Duplicate the hierarchy list of a job.  The nodes are not copied, they are
 * references (pointers) in the new list and must not be freed.
 *
 * Return:
 *     The list to be freed by the caller using lwc_delLL (lst, NULL);
 *     or NULL in case of error (if err_msg is not NULL it contains an
 *     error message which must be freed by the caller.  *err_msg may be
 *     NULL if there is not enough memory to store the error message)
 */
lwc_LL *
duplicate_hierarchy_list (lwc_LL *lst, char **err_msg)
{
	lwc_LL *copy_lst;
	char *id, *s;


	/* Create the result list */
	copy_lst = lwc_newLL ();
	if (copy_lst == NULL) {
		if (err_msg != NULL) {
			s = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (s) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, s);
			}
		}
		return NULL;
	}
	
	lwc_rewindLL (lst);
	while ((id = (char *)lwc_nextLL (lst)) != NULL) {
		if (lwc_addEndLL (copy_lst, id) != 0) {
			lwc_delLL (copy_lst, NULL);
			if (err_msg != NULL) {
				s = _("Memory allocation error");
				*err_msg = (char *) malloc (schedwi_strlen (s)
								+ 1);
				if (*err_msg != NULL) {
					strcpy (*err_msg, s);
				}
			}
			return NULL;
		}
	}
	return copy_lst;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved parameters and must be
 *           freed by the caller by sql_free_row().  If values_len is not
 *           NULL it contains the length of the values stored in values
 *           (without the trailing '\0') and must be freed by the caller by
 *           free().
 *           If the parameters has not been found in the hierarchy (ie. no
 *           ancestor defines these parameters), values and values_len are
 *           set to NULL.
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 */
unsigned int
get_job_parameters (	int workload_date,
			lwc_LL *lst,
			const char *tablename, const char *colnames,
			char ***values, unsigned long int **values_len,
			char **err_msg)
{
	MYSQL *sql;
	unsigned int ret;
	lwc_LL *rows, *rows_len;
	char **row;
	char *id, *buf;
	unsigned int buf_len;
	unsigned long int *row_len;

#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colnames != NULL
		&& values != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		return 2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((id = (char *)lwc_nextLL (lst)) != NULL) {
		/* Try to get the required parameter */
		ret = sql_select (&buf, &buf_len, err_msg, NULL,
				&rows, &rows_len,
				SQL_GET_JOB_PARAMS,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)workload_date,
				SQL_STRING, id,
				SQL_END);
		if (ret != 0) {
			end_mysql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/* Retrieve the result (only one row) */
		row = (char **) lwc_delStartLL (rows);
		if (row == NULL) {
			/*
			 * Nothing found - The same request will then be
			 * issued for the parent
			 */
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			lwc_delLL (rows_len, (void (*)(const void *))free);
			continue;
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

		/* Retrieve the fields len */
		row_len = (unsigned long int *) lwc_delStartLL (rows_len);
		lwc_delLL (rows_len, (void (*)(const void *))free);

		/* Copy the returned values */
		*values = row;

		/* Copy the values length */
		if (values_len != NULL) {
			*values_len = row_len;
		}
		
		/* Close the database and free the working buffer */
		end_mysql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Parameters not found in the database
	 */

	/* Close the database and free the working buffer */
	end_mysql ();
	if (buf != NULL) {
		free (buf);
	}

	*values = NULL;
	if (values_len != NULL) {
		*values_len = NULL;
	}
	return 0;
}


/*
 * Retrieve a parameter for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  value contains the retrieved parameter and must be
 *           freed by the caller (by free()).  If value_len is not NULL it
 *           contains the length of the value stored in value (without the
 *           added trailing '\0').
 *           If the parameter has not been found in the hierarchy (ie. no
 *           ancestor defines this parameter), value is set to NULL and
 *           value_len (if not NULL) is set to 0.
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 */
unsigned int
get_job_parameter (	int workload_date,
			lwc_LL *lst,
			const char *tablename, const char *colname,
			char **value, unsigned long int *value_len,
			char **err_msg)
{
	unsigned int ret;
	char **row;
	unsigned long int *row_len;
	char *s;

#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colname != NULL
		&& value != NULL);
#endif


	ret = get_job_parameters (	workload_date, lst,
					tablename, colname,
					&row, &row_len,
					err_msg);
	if (ret != 0) {
		return ret;
	}

	/* Parameter not found */
	if (row == NULL || row_len == NULL) {
		/* Return NULL in *value */
		*value = NULL;
		if (value_len != NULL) {
			*value_len = 0;
		}
		return 0;
	}

	/* Copy the returned value */
	s = (char *) malloc (row_len[0] + 1);
	if (s == NULL) {
		sql_free_row (row);
		free (row_len);
		if (err_msg != NULL) {
			s = _("Memory allocation error");
			*err_msg = (char *) malloc (schedwi_strlen (s) + 1);
			if (*err_msg != NULL) {
				strcpy (*err_msg, s);
			}
		}
		return 1;
	}
	schedwi_memcpy (s, row[0], row_len[0]);
	s[row_len[0]] = '\0';

	/* Copy the value length */
	if (value_len != NULL) {
		*value_len = row_len[0];
	}
	*value = s;
	sql_free_row (row);
	free (row_len);

	return 0;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.  The provided rows and rows_len parameters are filled with
 * the retrieved parameters.  rows_len may be NULL if the row lengths are not
 * required.  rows must be freed by the caller by
 *     lwc_delLL (rows, (void (*)(const void *))sql_free_row)
 * and rows_len must be freed by
 *     lwc_delLL (rows_len, (void (*)(const void *))free)
 * If there is no row retrieved from the database, rows (and rows_len if not
 * NULL) is set to NULL.
 *
 * Return:
 *     0 --> No error.  rows and rows_len (if not NULL) are set and must be
 *           freed by the caller by
 *               lwc_delLL (rows, (void (*)(const void *))sql_free_row)
 *               lwc_delLL (rows_len, (void (*)(const void *))free)
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 */
unsigned int
get_job_parameters_multi_row (
		int workload_date,
		lwc_LL *lst,
		const char *tablename, const char *colnames,
		const char *sort_colname,
		lwc_LL **rows, lwc_LL **rows_len,
		char **err_msg)
{
	MYSQL *sql;
	unsigned int ret;
	lwc_LL *r, *r_len;
	char *id, *buf;
	unsigned int buf_len;

#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colnames != NULL
		&& rows != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		return 2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((id = (char *)lwc_nextLL (lst)) != NULL) {
		/* Try to get the required parameter */
		if (sort_colname == NULL) {
			ret = sql_select (&buf, &buf_len, err_msg, NULL,
				&r, &r_len,
				SQL_GET_JOB_PARAMS,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)workload_date,
				SQL_STRING, id,
				SQL_END);
		}
		else {
			ret = sql_select (&buf, &buf_len, err_msg, NULL,
				&r, &r_len,
				SQL_GET_JOB_PARAMS_SORTED,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)workload_date,
				SQL_STRING, id,
				SQL_STRING_NON_ESCAPE, sort_colname,
				SQL_END);
		}

		if (ret != 0) {
			end_mysql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/*
		 * If nothing has been found, the same request will be
		 * issued for the parent
		 */
		if (lwc_getNumNode (r) <= 0) {
			lwc_delLL (r, (void (*)(const void *))sql_free_row);
			lwc_delLL (r_len, (void (*)(const void *))free);
			continue;
		}

		*rows = r;
		if (rows_len != NULL) {
			*rows_len = r_len;
		}
		else {
			lwc_delLL (r_len, (void (*)(const void *))free);
		}


		/* Close the database and free the working buffer */
		end_mysql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Parameters not found in the database
	 */
	*rows = NULL;
	if (rows_len != NULL) {
		*rows_len = NULL;
	}

	/* Close the database and free the working buffer */
	end_mysql ();
	if (buf != NULL) {
		free (buf);
	}

	return 0;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.  The provided callback function is called for each retrieved
 * row.  Its first parameter is the provided user_data object, the second
 * is the row array and the last is the array of each row size.
 * If callback returns a value other than 0, get_job_parameters_multi()
 * returns a value of 3.
 *
 * Return:
 *     0 --> No error.
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     3 --> The callback function didn't return 0
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 */
unsigned int
get_job_parameters_multi (
		int workload_date,
		lwc_LL *lst,
		const char *tablename, const char *colnames,
		const char *sort_colname,
		int (*callback)(void *, char **, unsigned long int *),
		void *user_data,
		char **err_msg)
{
	unsigned int ret;
	int callback_ret;
	lwc_LL *rows, *rows_len;
	char **row;
	unsigned long int *row_len;

#if HAVE_ASSERT_H
	assert (   lst != NULL && tablename != NULL && colnames != NULL
		&& callback != NULL);
#endif

	/* Retrieve the rows */
	ret = get_job_parameters_multi_row (	workload_date,
						lst, tablename, colnames,
						sort_colname,
						&rows, &rows_len,
						err_msg);
	if (ret != 0) {
		return ret;
	}

	/* Call the callback function for each row */
	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		row_len = (unsigned long int *) lwc_delStartLL (rows_len);
			
		callback_ret = callback (user_data, row, row_len);

		sql_free_row (row);
		if (row_len != NULL) {
			free (row_len);
		}

		if (callback_ret != 0) {
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			lwc_delLL (rows_len, (void (*)(const void *))free);
			return 3;
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	lwc_delLL (rows_len, (void (*)(const void *))free);

	return 0;
}


/*
 * Retrieve parameters for a job hierarchy.  The provided hierarchy list
 * (in lst) must have been built by get_hierarchy_list() prior to calling
 * this function.
 *
 * Return:
 *     0 --> No error.  values contains the retrieved parameters and must be
 *           freed by the caller by sql_free_row().  If values_len is not
 *           NULL it contains the length of the values stored in values
 *           (without the trailing '\0') and must be freed by the caller by
 *           free().
 *           If the parameters has not been found in the hierarchy (ie. all
 *           ancestors have a value of -1 in the first provided column),
 *           values and values_len are set to NULL.
 *     1 --> Memory allocation error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 *     2 --> Database connection error (if err_msg is not NULL it contains an
 *           error message which must be freed by the caller.  *err_msg may be
 *           NULL if there is not enough memory to store the error message).
 * other --> Database error (if err_msg is not NULL it contains an error
 *           message which must be freed by the caller.  *err_msg may be NULL
 *           if there is not enough memory to store the error message).
 *           The returned value is the one returned by mysql_errno().
 */
unsigned int
get_job_parameters_by_id (
			int workload_date,
			lwc_LL *lst,
			const char *tablename, const char *colnames,
			const char *not_defined,
			char ***values, unsigned long int **values_len,
			char **err_msg)
{
	MYSQL *sql;
	unsigned int ret;
	lwc_LL *rows, *rows_len;
	char **row;
	char *id, *buf;
	unsigned int buf_len;
	unsigned long int *row_len;

#if HAVE_ASSERT_H
	assert (   lst != NULL && not_defined != NULL
		&& tablename != NULL && colnames != NULL && values != NULL);
#endif

	/* Connect to the database */
	sql = begin_mysql (err_msg);
	if (sql == NULL) {
		return 2;
	}

	buf = NULL;
	buf_len = 0;
	lwc_rewindLL (lst);
	while ((id = (char *)lwc_nextLL (lst)) != NULL) {
		/* Try to get the required parameter */
		ret = sql_select (&buf, &buf_len, err_msg, NULL,
				&rows, &rows_len,
				SQL_GET_JOB_PARAMS_TIME,
				SQL_STRING_NON_ESCAPE, colnames,
				SQL_STRING_NON_ESCAPE, tablename,
				SQL_INT, (long int)workload_date,
				SQL_STRING, id,
				SQL_END);
		if (ret != 0) {
			end_mysql ();
			if (buf != NULL) {
				free (buf);
			}
			return ret;
		}

		/* Retrieve the result (only one row) */
		row = (char **) lwc_delStartLL (rows);
		if (row == NULL || strcmp (not_defined, row[0]) == 0) {
			/*
			 * Nothing found - The same request will then be
			 * issued for the parent
			 */
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			lwc_delLL (rows_len, (void (*)(const void *))free);
			continue;
		}
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

		/* Retrieve the fields len */
		row_len = (unsigned long int *) lwc_delStartLL (rows_len);
		lwc_delLL (rows_len, (void (*)(const void *))free);

		/* Copy the returned values */
		*values = row;

		/* Copy the values length */
		if (values_len != NULL) {
			*values_len = row_len;
		}
		
		/* Close the database and free the working buffer */
		end_mysql ();
		if (buf != NULL) {
			free (buf);
		}
		return 0;
	}

	/*
	 * Parameters not found in the database
	 */

	/* Close the database and free the working buffer */
	end_mysql ();
	if (buf != NULL) {
		free (buf);
	}

	*values = NULL;
	if (values_len != NULL) {
		*values_len = NULL;
	}
	return 0;
}


/*
 * Split the provided string using the provided separator
 *
 * Return:
 *   A string array (last element is NULL) to be freed by te caller using
 *            split_array_destroy().  If not NULL, array_len contains the
 *            array len (last NULL element not included)
 *   OR
 *   NULL in case of memory allocation error
 */
static char **
split_str (const char *str, char separator, unsigned int *array_len)
{
	unsigned int i, j, b, len, num_separator;
	char **a;

#if HAVE_ASSERT_H
	assert (str != NULL);
#endif

	len = schedwi_strlen (str);
	for (num_separator = i = 0; i < len; i++) {
		if (str[i] == separator) {
			num_separator++;
		}
	}

	a = (char **)malloc (sizeof (char *) * (num_separator + 2));
	if (a == NULL) {
		return NULL;
	}

	/* Skip the separators at the begining of the name */
	for (b = 0; b < len && str[b] == separator; b++);
	i = b;
	j = 0;
	while (i < len) {
		if (str[i] == separator) {
			a[j] = (char *)malloc (i - b + 1);
			if (a[j] == NULL) {
				for (i = 0; i < j; i++) {
					free (a[i]);
				}
				free (a);
				return NULL;
			}
			strncpy (a[j], str + b, i - b);
			a[j][i - b] = '\0';
			j++;
			/* Skip multiple separators */
			while (i < len && str[i] == separator) {
				i++;
			}
			b = i;
		}
		else {
			i++;
		}
	}
	/* Copy the last part of the string */
	if (b != i) {
		a[j] = (char *)malloc (i - b + 1);
		if (a[j] == NULL) {
			for (i = 0; i < j; i++) {
				free (a[i]);
			}
			free (a);
			return NULL;
		}
		strncpy (a[j], str + b, i - b);
		a[j][i - b] = '\0';
		j++;
	}
	a[j] = NULL;
	if (array_len != NULL) {
		*array_len = j;
	}
	return a;
}


/*
 * Free the provided string arrau
 */
static void
split_array_destroy (char **a)
{
	unsigned int i;

	if (a != NULL) {
		for (i = 0; a[i] != NULL; i++) {
			free (a[i]);
		}
		free (a);
	}
}


/*
 * Retrieve from the database the job id given its path
 *
 * Return:
 *     0 --> No error.  If not NULL, j_id, j_parent_id and j_type contain the
 *           job ID, its parent jobset ID and its type.  They must be freed by
 *           the caller using free()
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
get_job_id (	int workload_date, const char *job_name,
		char **j_id, char **j_parent_id, char **j_type,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	MYSQL *sql;
	char *err_msg = NULL;
	lwc_LL *rows;
	char **row;
	char **split, **a;
	char *s, *s_id, *s_parent_id, *s_type;
	unsigned int i, j, array_len, str_len, ret;

#if HAVE_ASSERT_H
	assert (job_name != NULL);
#endif

	/* Split the path into a jobset array */
	split = split_str (job_name, '/', &array_len);
	if (split == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return 1;
	}

	/* Empty path.  The details of the root jobset are returned */
	if (array_len == 0) {
		split_array_destroy (split);
		s_id = (char *) malloc (2);
		if (s_id == NULL) {
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}
		s_id[0] = '1';
		s_id[1] = '\0';
		s_parent_id = (char *) malloc (2);
		if (s_parent_id == NULL) {
			free (s_id);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}
		s_parent_id[0] = '0';
		s_parent_id[1] = '\0';
		s_type = (char *) malloc (2);
		if (s_type == NULL) {
			free (s_parent_id);
			free (s_id);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}
		s_type[0] = '0';
		s_type[1] = '\0';

		if (j_id != NULL) {
			*j_id = s_id;
		}
		else {
			free (s_id);
		}
		if (j_parent_id != NULL) {
			*j_parent_id = s_parent_id;
		}
		else {
			free (s_parent_id);
		}
		if (j_type != NULL) {
			*j_type = s_type;
		}
		else {
			free (s_type);
		}
		return 0;
	}

	/*
	 * Allocate memory for the array which will contain an escaped copy
	 * of the jobset names that composed the path (in the split array)
	 */
	a = (char **)malloc (sizeof (char *) * array_len);
	if (a == NULL) {
		split_array_destroy (split);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return 1;
	}

	/* Connect to the database */
	sql = begin_mysql (&err_msg);
	if (sql == NULL) {
		split_array_destroy (split);
		free (a);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 0);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 2;
	}

	/* Escape the jobset names and populate the array */
	for (str_len = i = 0; i < array_len; i++) {
		a[i] = sql_escape (sql, split[i]);
		if (a[i] == NULL) {
			end_mysql ();
			split_array_destroy (split);
			for (j = 0; j < i; j++) {
				free (a[j]);
			}
			free (a);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return 1;
		}
		str_len += schedwi_strlen (a[i]) + 2;
	}

	/*
	 * This string will contain the jobset names in a format suitable for
	 * the SQL request
	 */
	s = (char *) malloc (str_len + array_len + 1);
	if (s == NULL) {
		end_mysql ();
		split_array_destroy (split);
		for (j = 0; j < array_len; j++) {
			free (a[j]);
		}
		free (a);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return 1;
	}

	/* Contatenate the escaped jobset names in s */
	for (j = i = 0; i < array_len; i++) {
		s[j++] = '"';
		strcpy (s + j, a[i]);
		j += schedwi_strlen (a[i]);
		s[j++] = '"';
		s[j++] = ',';
		free (a[i]);
	}
	free (a);
	s[j - 1] = '\0';	/* Remove the last `,' */

	/* Retrieve all the jobsets details */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_GET_JOB_ID,
				SQL_INT, (long int)workload_date,
				SQL_STRING_NON_ESCAPE, s,
				SQL_END);
	end_mysql ();
	free (s);
	if (ret != 0) {
		split_array_destroy (split);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* Check that the retrieved jobsets are part of the provided path */
	row = NULL;
	s = "0";
	for (i = 0; i < array_len; i++) {
		lwc_rewindLL (rows);
		while ((row = (char **)lwc_nextLL (rows)) != NULL) {
			if (	   strcmp (row[2], split[i]) == 0
				&& strcmp (row[1], s) == 0)
			{
				s = row[0];
				break;
			}
		}
		/* Fail to find a path component in the retrieved jobsets */
		if (row == NULL) {
			split_array_destroy (split);
			lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Unknown jobset"),
						ret);
			}
			return 2;
		}
	}
	split_array_destroy (split);

	/* Copy the details of the `leaf' job/jobset */
	s_id = (char *)malloc (schedwi_strlen (row[0]) + 1);
	if (s_id == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return 1;
	}
	strcpy (s_id, row[0]);

	s_parent_id = (char *)malloc (schedwi_strlen (row[1]) + 1);
	if (s_parent_id == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		free (s_id);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return 1;
	}
	strcpy (s_parent_id, row[1]);

	s_type = (char *)malloc (schedwi_strlen (row[3]) + 1);
	if (s_type == NULL) {
		lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
		free (s_id);
		free (s_parent_id);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"),
					0);
		}
		return 1;
	}
	strcpy (s_type, row[3]);

	lwc_delLL (rows, (void (*)(const void *)) sql_free_row);

	if (j_id != NULL) {
		*j_id = s_id;
	}
	else {
		free (s_id);
	}
	if (j_parent_id != NULL) {
		*j_parent_id = s_parent_id;
	}
	else {
		free (s_parent_id);
	}
	if (j_type != NULL) {
		*j_type = s_type;
	}
	else {
		free (s_type);
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
