/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_jobsets.c -- jobsets management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_jobsets.h>

#define SQL_JOBSET_LIST "SELECT id,parent,name FROM job_main_s WHERE workload_date=%d AND type=0 ORDER BY parent DESC,name"
#define SQL_JOBSET_SEARCH "SELECT id,parent,name,type FROM job_main_s WHERE workload_date=%d AND name LIKE \"%%%s%%\""


/*
 * Retrieve from the database the jobset list
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved jobsets and must be freed
 *           by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_jobset_list (int workload_date, lwc_LL **rows,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows, NULL,
				SQL_JOBSET_LIST,
				SQL_INT, (long int) workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Retrieve from the database the jobsets matching the provided name
 *
 * For each finded job/jobset, the provided find_job() function is called with
 * the following parameters:
 *     1) The provided workload date
 *     2) the provided user_data_find_job parameter
 *     3) the job/jobset ID
 *     4) the parent jobset ID
 *     5) the job/jobset name
 *     6) the type ("0" for jobset, "1" for job)
 * find_job() must return 0 if successful or an other code in case of error.
 *
 * Return:
 *   0 --> No error
 *  -1 --> The find_job() function returned with a code different from 0
 *  -2 --> SQL error (if error_func() is not NULL, it is called with
 *         user_data_error_func as its first parameter and the error message
 *         as the second parameter)
 */
int
sql_jobset_search (int workload_date, const char *search_str,
	int (*find_job)(int, void *, const char *, const char *,
			const char *, const char *),
	void *user_data_find_job,
	void (*error_func)(void *, const char *, unsigned int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	unsigned int ret;
	char **row;

#if HAVE_ASSERT_H
	assert (search_str != NULL && find_job != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQL_JOBSET_SEARCH,
				SQL_INT, (long int) workload_date,
				SQL_STRING, search_str,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return -2;
	}
	while ((row = (char **) lwc_delStartLL (rows)) != NULL) {
		if (find_job (	workload_date, user_data_find_job,
				row[0], row[1], row[2], row[3]) != 0)
		{
			sql_free_row (row);
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return -1;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, NULL);
	return 0;
}


/*
 * Get the canvas size for the provided jobset ID
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_jobset_get_canvas_size (int workload_date, const char *jobset_id,
			void (*set_size)(const char *, const char *, void *),
			void *user_data_set_size,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;
	char **row;
	lwc_LL *hierarchy_list;

#if HAVE_ASSERT_H
	assert (jobset_id != NULL);
#endif

	hierarchy_list = get_hierarchy_list (	workload_date, jobset_id,
						&err_msg);
	if (hierarchy_list == NULL) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, 1);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return 1;
	}

	ret = get_job_parameters (	workload_date, hierarchy_list,
					"jobset_background_s",
					"canvas_width,canvas_height",
					&row, NULL, &err_msg);
	if (ret != 0) {
		lwc_delLL (hierarchy_list, (void (*)(const void *))free);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	lwc_delLL (hierarchy_list, (void (*)(const void *))free);

	if (set_size != NULL) {
		set_size (row[0], row[1], user_data_set_size);
	}

	sql_free_row (row);
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
