/* Schedwi
   Copyright (C) 2007 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_status.c -- job status management functions */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <sql_common.h>
#include <sql_status.h>

#define SQLSTATUS_WORKLOAD_LIST "SELECT DISTINCT workload_date FROM job_status ORDER BY workload_date"
#define SQLSTATUS_GET "SELECT status,start_time,retry_num,error_msg FROM job_status WHERE workload_date=%d AND job_id=\"%s\""
#define SQLSTATUS_SET "REPLACE INTO job_status (workload_date,job_id,status,start_time,retry_num,error_msg) VALUES (%d,\"%s\",%d,%u,%d,\"%s\")"
#define SQLSTATUS_DEL "DELETE FROM job_status WHERE workload_date=%d"
#define SQLSTATUS_DEL_TABLES "DELETE FROM %s_s WHERE workload_date=%d"
#define SQLSTATUS_COPY "REPLACE INTO %s_s SELECT *,%d FROM %s"

/*
 * List of tables to copy.  These tables will be copied in tables with the
 * same name but suffixed by `_s'
 */
static char * status_tables[] = {	"job_main",
					"job_start_limit",
					"job_max_duration",
					"job_retries",
					"job_retries_interval",
					"job_username",
					"job_file_out",
					"job_file_err",
					"job_loadenv",
					"job_success_return_code",
					"job_command",
					"job_arguments",
					"environments",
					"environment_var",
					"job_environment",
					"host_environment",
					"job_host",
					"calendars",
					"constraint_file",
					"jobset_background",
					"job_icon_default",
					"job_icon_completed",
					"job_icon_failed",
					"job_icon_running",
					"job_icon_waiting",
					"jobset_icon_default",
					"jobset_icon_completed",
					"jobset_icon_failed",
					"jobset_icon_running",
					"jobset_icon_waiting",
					"link",
					NULL
				};


/*
 * Get all the available workload from the job_status database table
 *
 * Return:
 *     0 --> No error.  rows contains the retrieved workload date and must be
 *           freed by the caller by
 *              lwc_delLL (rows, (void (*)(const void *)) sql_free_row);
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_status_workload_list (lwc_LL **rows,
			void (*error_func)(void *, const char *, unsigned int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (rows != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, rows, NULL,
				SQLSTATUS_WORKLOAD_LIST,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Get the status of the provided job/jobset
 *
 * Return:
 *     0 --> No error.  status contains the retrieved status and start_time
 *           the associated start time.
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_status_get (int workload_date,
		const char *job_id,
		int *status,
		long int *start_time,
		int *retry_num,
		char **message,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	unsigned int ret;
	char **row, *s;

#if HAVE_ASSERT_H
	assert (job_id != NULL && status != NULL);
#endif

	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows, NULL,
				SQLSTATUS_GET,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_END);

	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	row = (char **) lwc_delStartLL (rows);

	/* Message */
	if (message != NULL) {
		if (row == NULL || row[3] == NULL || row[3][0] == '\0') {
			*message = NULL;
		}
		else {
			s = (char *) malloc (schedwi_strlen (row[3]) + 1);
			if (s == NULL) {
				sql_free_row (row);
				lwc_delLL (rows,
					(void (*)(const void *))sql_free_row);
				if (error_func != NULL) {
					error_func (user_data_error_func,
						_("Memory allocation error"),
						0);
				}
				return 1;
			}
			strcpy (s, row[3]);
			*message = s;
		}
	}

	/* Status */
	*status = (row == NULL) ? 0 : atoi (row[0]);

	/* Status time */
	if (start_time != NULL) {
		*start_time = (row == NULL) ? 0 : strtol (row[1], NULL, 0);
	}

	/* Current retry number */
	*retry_num = (row == NULL) ? 0 : atoi (row[2]);

	sql_free_row (row);
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return 0;
}


/*
 * Set the status of the provided job/jobset
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_status_set (int workload_date,
		const char *job_id,
		int status,
		unsigned int start_time,
		int retry_num,
		const char *error_message,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret;

#if HAVE_ASSERT_H
	assert (job_id != NULL);
#endif


	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL,
				SQLSTATUS_SET,
				SQL_INT, (long int)workload_date,
				SQL_STRING, job_id,
				SQL_INT, (long int)status,
				SQL_INT, (long int)start_time,
				SQL_INT, (long int)retry_num,
				SQL_STRING, (error_message != NULL)
							? error_message: "",
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Delete a workload
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_status_delete (int workload_date,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret, err, i;

	/* First, delete the entry in the job_status table */
	ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL,
				SQLSTATUS_DEL,
				SQL_INT, (long int)workload_date,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	/* And then delete the entries for all the other tables */
	err = 0;
	for (i = 0; status_tables[i] != NULL; i++) {
		ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL,
					SQLSTATUS_DEL_TABLES,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_END);
		if (ret != 0) {
			err = ret;
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
		}
	}

	if (err != 0) {
		return err;
	}
	return 0;
}


/*
 * Create a new workload by copying tables
 *
 * Return:
 *     0 --> No error
 *     1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 * other --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
unsigned int
sql_status_copy_to_workload (int workload_date,
		void (*error_func)(void *, const char *, unsigned int),
		void *user_data_error_func)
{
	char *err_msg = NULL;
	unsigned int ret, i;

	/* Copy all the tables */
	for (i = 0; status_tables[i] != NULL; i++) {
		ret = sql_non_select (  NULL, NULL, &err_msg, NULL, NULL,
					SQLSTATUS_COPY,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_INT, (long int)workload_date,
					SQL_STRING_NON_ESCAPE,
							status_tables[i],
					SQL_END);
		if (ret != 0) {
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						err_msg, ret);
			}
			if (err_msg != NULL) {
				free (err_msg);
			}
			
			/* Try to remove the successfully copied tables */
			sql_status_delete (workload_date, NULL, NULL);
			return ret;
		}
	}

	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
