/* Schedwi
   Copyright (C) 2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* reg_register_clnt.c -- Send a registration request to the server */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif


#include <lib_functions.h>
#include <lwc_log.h>
#include <net_utils_sock.h>
#include <parse_json_result.h>
#include <cert_utils.h>
#include <reg_register_clnt.h>


/*
 * Send the request to the server (schedwireg)
 *
 * Return:
 *   0 --> No error.
 *  -1 --> Error (a message has been logged)
 *   1 --> Error on the server side.  In that case, and if not NULL, `srv_msg'
 *         contains the error message from the server (`srv_msg' may be NULL
 *         or empty if the server didn't send any message). It must be freed by
 *         the caller.  Also, `reason' is set if not NULL.  It contains the
 *         error code which indentify the error.
 */
int
send_request (	int sock, const char *local_port, char ssl, const char *csr,
		char **srv_msg, error_reason_t *reason)
{
	char *s, *p;
	char *buff;
	size_t len;
	ssize_t nb_read;
	int ret;

#if HAVE_LIBGNUTLS
	char *serialized;
#endif

#if HAVE_ASSERT_H
	assert (   local_port != NULL
		&& (ssl == 0 || (csr != NULL && *csr != '\0')));
#endif

	/* Convert the provided port to a number */
	p = port2number (local_port);
	if (p == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

#if HAVE_LIBGNUTLS
	/* Convert the PEM CSR to a string */
	if (ssl != 0) {
		serialized = PEM2string (csr);
		if (serialized == NULL) {
			free (p);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}
#endif

	/* Send the request */
	s = "[ \"register\", { \"port\" : ";
	net_write_sock (sock, s, schedwi_strlen (s));
	net_write_sock (sock, p, schedwi_strlen (p));
	free (p);
	s = ", \"ssl\" : ";
	net_write_sock (sock, s, schedwi_strlen (s));

#if HAVE_LIBGNUTLS
	if (ssl != 0) {
		s = "true, \"csr\" : \"";
		net_write_sock (sock, s, schedwi_strlen (s));
		net_write_sock (sock, serialized, schedwi_strlen (serialized));
		free (serialized);
		s = "\" } ]";
		net_write_sock (sock, s, schedwi_strlen (s));
	}
	else
#endif
	{
		s = "false } ]";
		net_write_sock (sock, s, schedwi_strlen (s));
	}

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_sock (sock, &buff, &len);
	if (nb_read < 0) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	ret = parse_json_result (buff, nb_read, srv_msg, reason);
	if (buff != NULL) {
		free (buff);
	}
	if (ret == 0) {
		return 0;
	}
	else if (ret < 0) {
		return -1;
	}
	return 1;
}

/*-----------------============== End Of File ==============-----------------*/
