# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to create a new calendar."""

import sys
import getopt

import path_cal
import set_options
from tables.calendars import calendars
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: mk [OPTION]... CAL...
        Create new calendars.""")
    set_options.usage_options()

def mk(sql_session, current_cwd, arguments):
    """Create a new calendar.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the mk command (list of calendars)

    """
    # Parse the parameters
    try:
        optlist, args = set_options.parse_args(arguments)
    except getopt.GetoptError, err:
        sys.stderr.write("mk: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write("mk: missing the calendar name to create\n")
        return 1
    paths = list()
    for arg in args:
        basename, p = path_cal.get_new_paths(sql_session, arg,
                                             current_cwd, True)
        if not p:
            sys.stderr.write("mk: `%s': no such directory\n" % arg)
            return 1
        paths.append ([basename, p])
    if not paths:
        sys.stderr.write("mk: no such directory\n")
        return 1
    session = sql_session.open_session()
    for basename, dirs in paths:
        for p in dirs:
            # Check that the calendar to create does not already exist
            full_path = path_cal.PathCal(session, basename, p)
            if full_path:
                sys.stderr.write("mk: %s already exists\n" % full_path)
                sql_session.cancel_session(session)
                return 1
            cal = calendars(parent=p.id[-1],
                            name=basename,
                            entry_type=0,
                            description='',
                            formula='')
            session.add(cal)
            if set_options.set(session, cal, optlist):
                sql_session.cancel_session(session)
                return 1
    sql_session.close_session(session)
    return 0

