# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to create a new calendar directory."""

import sys
import getopt

import path_cal
from tables.calendars import calendars
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: mkdir DIRECTORY...
        Create the calendar DIRECTORY(ies), if they do not already exist.

        Option:
          -h, --help  display this help.
    """)

def mkdir(sql_session, current_cwd, arguments):
    """Create a new calendar directory.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the mkdir command (list
                 of calendars)

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("mkdir: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write("mkdir: missing directory name to create\n")
        return 1
    paths = list()
    for arg in args:
        basename, p = path_cal.get_new_paths(sql_session, arg,
                                             current_cwd, True)
        if not p:
            sys.stderr.write("mkdir: `%s': no such directory\n" % arg)
            return 1
        paths.append ([basename, p])
    if not paths:
        sys.stderr.write("mkdir: no such directory\n")
        return 1
    session = sql_session.open_session()
    for basename, dirs in paths:
        for p in dirs:
            # Check that the directory to create does not already exist
            full_path = path_cal.PathCal(session, basename, p)
            if full_path:
                sys.stderr.write("mkdir: %s already exists\n" % full_path)
                sql_session.cancel_session(session)
                return 1
            cal = calendars(parent=p.id[-1],
                            name=basename,
                            entry_type=1,
                            description='',
                            formula='')
            session.add(cal)
    sql_session.close_session(session)
    return 0

