# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to remove a calendar directory."""

import sys
import getopt

import path_cal
from tables.calendars import calendars
from tables.calendars_s import calendars_s
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: rmdir DIRECTORY...
        Remove the calendar DIRECTORY(ies), if they are empty.
        See `help rm' to remove directories and their contents recursively.

        Option:
          -h, --help  display this help.
    """)

def rmdir(sql_session, current_cwd, arguments, workload=None):
    """Remove an empty calendar directory.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the rmdir command (list
                 of calendars)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("rmdir: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    if args:
        for arg in args:
            p = path_cal.get_paths(sql_session, arg, current_cwd, True,
                                   workload=workload)
            if not p:
                sys.stderr.write("rmdir: `%s': no such directory\n" % arg)
            else:
                paths.extend (p)
    else:
        sys.stderr.write("rmdir: missing directory name to remove\n")
        return 1
    if not paths:
        return 1
    error = False
    session = sql_session.open_session()
    for p in paths:
        if workload is None:
            q = session.query(calendars)
        else:
            q = session.query(calendars_s)
            q = q.filter(calendars_s.workload_date == workload)
        try:
            cal = q.filter_by(id=p.id[-1]).one()
        except:
            sys.stderr.write("rmdir: no such directory\n")
            error = True
            continue
        if workload is None:
            q = session.query(calendars)
        else:
            q = session.query(calendars_s)
            q = q.filter(calendars_s.workload_date == workload)
        if q.filter_by(parent=cal.id).count():
            sys.stderr.write("rmdir: `%s': directory not empty\n" %p)
            error = True
            continue
        session.delete(cal)
    sql_session.close_session(session)
    return 1 if error else 0

