# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to list environment groups."""

import sys
import getopt

from tables.environments import environments
from tables.environments_s import environments_s
import environments_utils
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: ls [OPTION]... [ENV]...
    List information about the ENV groups (all the environment groups by
    default).  Sort entries alphabetically.

    Options:
      -l          use a long listing format according to the following schema:
                     NUM    number of environment variables in the group
                     NAME   environment group name
      -h, --help  display this help.

    Example:
    > ls -l
     10 Default
      5 Hive
    """)

def _print_env(env, long_listing):
    """Print the environment group name.

    Arguments:
    env -- environment group object
    long_listing -- should a long listing (number of variables inside the
                    group and the environment group name) be printed or just
                    the name.

    """
    if not long_listing:
        print env.name
        return
    print "%3d %s" % (len(env.environment_var), env.name)

def ls(sql_session, arguments, workload=None):
    """List environments groups.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the ls command (list of env groups)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, 'lh', ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("ls: " + str(err) + "\n")
        return 1
    long_listing = False
    for o, a in optlist:
        if o == "-l":
            long_listing = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    session = sql_session.open_session()
    if args:
        for a in args:
            env_list = environments_utils.name2env_list(session, a, workload)
            if not env_list:
                sys.stderr.write("ls: `%s': no such environment group\n" % a)
            else:
                for e in env_list:
                    _print_env(e, long_listing)
    else:
        if workload is None:
            for e in session.query(environments).order_by(environments.name):
                _print_env(e, long_listing)
        else:
            q = session.query(environments_s)
            q = q.filter(environments_s.workload_date == workload)
            for e in q.order_by(environments_s.name):
                _print_env(e, long_listing)
    sql_session.close_session(session)
    return 0

