# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to get the jobs, jobsets and hosts that use an environment group."""

import sys
import getopt

import path
from tables.hosts import hosts
from tables.job_main import job_main
from tables.job_environment import job_environment
from tables.host_environment import host_environment
from tables.job_main_s import job_main_s
from tables.job_environment_s import job_environment_s
from tables.host_environment_s import host_environment_s
import environments_utils
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: whatuses ENV...
        List the jobs, jobsets and hosts that use the given environment groups.

        Option:
          -h, --help  display this help. 
    """)

def is_used(session, env, workload):
    """Tell if the environment group is used.

    Arguments:
    session -- SQLAlchemy session
    env -- environment group object
    workload -- workload to use

    """
    if workload is None:
        q = session.query(job_environment)
        if q.filter(job_environment.env_id == env.id).count():
            return True
        q = session.query(host_environment)
        if q.filter(host_environment.env_id == env.id).count():
            return True
    else:
        q = session.query(job_environment_s)
        q = q.filter(job_environment_s.workload_date == workload)
        if q.filter(job_environment_s.env_id == env.id).count():
            return True
        q = session.query(host_environment_s)
        q = q.filter(host_environment_s.workload_date == workload)
        if q.filter(host_environment_s.env_id == env.id).count():
            return True
    return False

_FIELD_WIDTH = 10

def _print_val(title, fmt=None, val=None):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    fmt -- format string for the parameter value
    val -- value

    """
    if fmt is not None:
        if val is not None:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt % val
        else:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt
    else:
        print "%s :" % title.rjust(_FIELD_WIDTH)

def _print_whatuses(session, env, workload):
    """Print the jobs, jobsets and hosts that use an environment group.

    Arguments:
    session -- SQLAlchemy session
    env -- environment group object
    workload -- workload to use

    """
    print "%s:" % env.name
    if workload is None:
        q = session.query(job_main).filter(job_environment.env_id == env.id)
        q = q.filter(job_environment.job_id == job_main.id)
        q = q.order_by(job_main.parent)
    else:
        q = session.query(job_main_s).filter(job_environment_s.env_id == env.id)
        q = q.filter(job_environment_s.job_id == job_main_s.id)
        q = q.order_by(job_main_s.parent)
    for job in q:
        _print_val("Job" if job.type else "Jobset", "%s",
                   path.id2path(session, job.id, workload))
    if workload is None:
        q = session.query(hosts).filter(host_environment.env_id == env.id)
        q = q.filter(host_environment.host_id == hosts.id)
        q = q.order_by(hosts.hostname)
    else:
        q = session.query(hosts).filter(host_environment_s.env_id == env.id)
        q = q.filter(host_environment_s.host_id == hosts.id)
        q = q.order_by(hosts.hostname)
    for h in q:
        _print_val("Host", "%s (%s)", (h.hostname, h.portnum))

def whatuses(sql_session, arguments, workload=None):
    """Show what is referencing an environment group.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the whatuses command
                 (list of env groups)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("whatuses: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write("whatuses: missing operand\n")
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        env_list = environments_utils.name2env_list(session, a, workload)
        if not env_list:
            sys.stderr.write("whatuses: `%s': no such environment group\n" % a)
            error = True
        else:
            first = True
            for e in env_list:
                if not first:
                    print
                _print_whatuses(session, e, workload)
                first = False
    sql_session.close_session(session)
    return 0 if not error else 1

