# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to copy jobs and jobsets."""

import sys
import getopt

import path
import rm
from tables.job_main import job_main
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: cp [OPTION]... SOURCE DEST
          or:  cp [OPTION]... SOURCE... JOBSET
        Copy SOURCE to DEST, or multiple SOURCE(s) to JOBSET.

        Options:
          -f, --force          never prompt for confirmation and overwrite the
                               destination if it already exists
          -r, -R, --recursive  copy jobsets recursively
    """)

def _relink(sql_session, job_assoc, job_links):
    """Recreate dependency links to the newly copied jobs/jobsets.

    Arguments:
    sql_session -- SQLAlchemy session
    job_assoc -- dictionnary of sourceJobID -> newJobID
    job_links -- dictionnary of jobIDs referenced by new jobs
                 destJobId -> [ newJobID, ...]

    """
    session = sql_session.open_session()
    for k in job_links.keys():
        if k in job_assoc:
            for job_id in job_links[k]:
                try:
                    job = session.query(job_main).filter_by(id=job_id).one()
                except:
                    continue
                for link in job.links:
                    if link.job_id_destination == k:
                        link.job_id_destination = job_assoc[k]
    sql_session.close_session(session)

def _cp_recursive(sql_session, src_job, dest_jobset,
                  job_assoc, job_links, base=None):
    """Recursively copy a job/jobset."""
    session = sql_session.open_session()
    new_job = src_job.copy()
    new_job.parent = dest_jobset.id
    if base is not None:
        new_job.name = base
    session.add(new_job)
    sql_session.close_session(session)
    job_assoc[src_job.id] = new_job.id
    for link in src_job.links:
        if link.job_id_destination in job_links:
            job_links[link.job_id_destination].append(new_job.id)
        else:
            job_links[link.job_id_destination] = [ new_job.id ]
    if src_job.type == 1:
        return 0
    session = sql_session.open_session()
    jobs = session.query(job_main).filter_by(parent=src_job.id).all()
    sql_session.close_session(session)
    for job in jobs:
        _cp_recursive(sql_session, job, new_job, job_assoc, job_links)
    return 0

def _rename(sql_session, cwd, src_path, dst, recursive):
    """Recursively rename a job/jobset."""
    (base, dest_path) = path.get_new_paths(sql_session, dst, cwd)
    lp = len(dest_path)
    if lp == 0:
        sys.stderr.write("cp: no such destination jobset\n")
        return 1
    if lp > 1:
        sys.stderr.write("cp: `%s/%s': no such job or jobset\n" % \
                            (dest_path[0], base))
        return 1
    session = sql_session.open_session()
    try:
        djbset = session.query(job_main).filter_by(id=dest_path[0].id[-1]).one()
    except:
        sys.stderr.write("cp: no such destination jobset\n")
        sql_session.cancel_session(session)
        return 1
    try:
        src_job = session.query(job_main).filter_by(id=src_path.id[-1]).one()
    except:
        sys.stderr.write("cp: no such job or jobset\n")
        sql_session.cancel_session(session)
        return 1
    sql_session.close_session(session)
    # The source is a jobset and the recursive option (-r) is not set
    if src_job.type == 0 and not recursive:
        sys.stderr.write("cp: omitting jobset `%s'\n" % src_path)
        return 1
    job_assoc = {}
    job_links = {}
    ret = _cp_recursive(sql_session, src_job, djbset,
                        job_assoc, job_links, base)
    if ret == 0:
        _relink(sql_session, job_assoc, job_links)
    return ret

def _cp_multi(sql_session, cwd, paths, recursive, force):
    dest = paths.pop()
    session = sql_session.open_session()
    try:
        dest_jobset = session.query(job_main).filter_by(id=dest.id[-1]).one()
    except:
        sys.stderr.write("cp: no such destination jobset\n")
        sql_session.cancel_session(session)
        return 1
    sql_session.close_session(session)
    if len(paths) > 1:
        if dest_jobset.type != 0:
            sys.stderr.write("cp: target `%s' is not a jobset\n" % dest)
            return 1
    else:
        if dest.id[-1] == paths[0].id[-1]:
            sys.stderr.write("cp: cannot copy `%s' to itself\n" % dest)
            return 1
        if dest_jobset.type != 0:
            if not force:
                sys.stderr.write("cp: target `%s' already exists\n" % dest)
                return 1
            else:
                session = sql_session.open_session()
                rm.rm_recursive(session, dest, force, recursive=True)
                sql_session.close_session(session)
                return _rename(sql_session, cwd, paths[0], str(dest), recursive)
    job_assoc = {}
    job_links = {}
    for p in paths:
        session = sql_session.open_session()
        try:
            job = session.query(job_main).filter_by(id=p.id[-1]).one()
        except:
            sys.stderr.write("cp: no such job or jobset\n")
            sql_session.cancel_session(session)
            return 1
        sql_session.close_session(session)
        if job.type == 0:
            if not recursive:
                sys.stderr.write("cp: omitting jobset `%s'\n" % p)
                return 1
            try:
                dest.id.index(job.id)
            except:
                _cp_recursive(sql_session, job, dest_jobset,
                              job_assoc, job_links)
            else:
                sys.stderr.write(
                     "cp: cannot copy `%s' to a sub-jobset of itself\n" % p)
                return 1
        else:
            _cp_recursive(sql_session, job, dest_jobset, job_assoc, job_links)
    _relink(sql_session, job_assoc, job_links)
    return 0                

def cp(sql_session, current_cwd, arguments):
    """Copy a job or jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the cp command (list
                 of jobs/jobsets)

    """
    try:
        optlist, args = getopt.getopt(arguments, "frR", ["force", "recursive"])
    except getopt.GetoptError, err:
        sys.stderr.write("cp: " + str(err) + "\n")
        return 1
    force = False
    recursive = False
    for o, a in optlist:
        if o in ("-f", "--force"):
            force = True
        elif o in ("-r", "-R", "--recursive"):
            recursive = True
    paths = list()
    num_args = len(args)
    dest_rename = None
    if args:
        for i in range(0, num_args):
            p = path.get_paths(sql_session, args[i], current_cwd)
            # An argument is an unknow path (if it's not the last one)
            if not p:
                if i + 1 != num_args:
                    sys.stderr.write("cp: `%s': no such job or jobset\n" % \
                                     args[i])
                    return 1
                else:
                    dest_rename = args[i]
            paths.extend(p)
    else:
        sys.stderr.write("cp: missing job or jobset operand\n")
        return 1
    if not paths:
        sys.stderr.write("cp: `%s': no such job or jobset\n" % args[0])
        return 1
    lp = len(paths)
    # Copy and rename
    if lp == 1:
        if num_args < 2:
            sys.stderr.write("cp: no enough operand or unknown job/jobset\n")
            return 1
        return _rename(sql_session, current_cwd, paths[0], args[1], recursive)
    # Copy
    if dest_rename is not None:
        sys.stderr.write("cp: target `%s' does not exist\n" % dest_rename)
        return 1
    return _cp_multi(sql_session, current_cwd, paths, recursive, force)

