# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to create new jobs."""

import sys
import getopt

from sqlalchemy.orm.exc import NoResultFound

import path
import set_options
from tables.job_main import job_main

def usage():
    """Print a usage message on STDOUT."""
    print """Usage: mk [OPTION]... JOB...
Create new jobs.

Options:
  --enable=True|False     enable (True) or disable (False) the job.
  --host=HOSTNAME[:PORT]  host on which the job must be running.  If not set,
                          same as the parent jobset.
  --calendar=CAL_NAME     calendar name. If not set, same as the parent jobset.
  --start=TIME            start time of the job (HH:MM).
  --command=COMMAND       command to run (if set to `-', no command will be
                          started and the job will complete immediately)
                          If not set, use the one defined by the parent jobset.
  --parameter=PARAMETER   a parameter to pass to the command.  This option can
                          be specified several times to define more than one
                          parameter.  If not set, use the parameters defined
                          by the parent jobset.
  --return=RETURN_CODE    a return code from the command equal or below this
                          value means success.  If not set, use the one set
                          by the parent jobset.
  --user=USERNAME         username under which the command must be run.
                          If not set, same as the parent jobset.
  --stdout=FILE           command output file (stored on the Schedwi agent).
                          If not set, same as the parent jobset.
  --stderr=FILE           command error file (stored on the Schedwi agent).
                          If not set, same as the parent jobset.
  --duration=MINUTES      maximum number of minutes for the job to
                          complete once running.  If not set, same as the
                          parent jobset.
  --retries=NUM           number of retries when the job fails.
                          If not set, same as the parent jobset.
  --retries-int=MINUTES   number of minutes between retries.
                          If not set, same as the parent jobset.
  --limit=MINUTES         number of minutes for the job to be started past its
                          start time (number of minutes or HH:MM)
                          If not set, same as the parent jobset.
  --user-env=True|False   load (True) or not (False) the user account
                          environment (~/.profile or ~/.bashrc for instance) on
                          the remote host before running the job command.
                          If not set, same as the parent jobset.
  --addenv=ENV_NAME       add an environment group to the job.  This
                          option can be specified several times.
  --noenv                 do not use any environment group for this job.
  --addfile=HOSTNAME[:PORT],True|False,FILE     add a file to check before
                          starting the job.  The value is composed of
                          three fields separated by a comma.  The first one is
                          the remote agent hostname on which to check the file.
                          The second specifies if the file must be present
                          (True) or not (False).  The last one is the file or
                          directory to check.
  --description=TEXT      a description.
  -h, --help              display this help.
    """

def mk(sql_session, current_cwd, current_cwd_cal, arguments):
    """Create new jobs.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the mk command (list
                 of jobs/jobsets)

    """
    # Parse the parameters
    try:
        optlist, args = set_options.parse_args(arguments)
    except getopt.GetoptError, err:
        sys.stderr.write("mk: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write("mk: missing the job name to create\n")
        return 1
    paths = list()
    for arg in args:
        basename, p = path.get_new_paths(sql_session, arg, current_cwd)
        paths.append ([basename, p])
    if not paths:
        sys.stderr.write("mk: no such jobset\n")
        return 1
    session = sql_session.open_session()
    for basename, dirs in paths:
        for p in dirs:
            # Check that the job to create does not already exist
            try:
                full_path = path.Path(session, basename, p)
            except NoResultFound:
                pass
            else:
                sys.stderr.write("mk: %s already exists\n" % full_path)
                sql_session.cancel_session(session)
                return 1
            job = job_main( parent=p.id[-1],
                            name=basename,
                            type=1,
                            enabled=1,
                            description='',
                            x=0, y=0,
                            cal_id=0,
                            start_time=-1)
            session.add(job)
            if set_options.set( session, current_cwd, current_cwd_cal,
                                job, optlist) != 0:
                sql_session.cancel_session(session)
                return 1
    sql_session.close_session(session)
    return 0

