# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to move or rename jobs and jobsets."""

import sys
import getopt

import path
from tables.job_main import job_main
from help import print_trim

def usage():
    """Print a usage message on STDOUT."""
    print_trim("""Usage: mv SOURCE DEST
          or:  mv SOURCE... JOBSET
        Rename SOURCE to DEST, or move SOURCE(s) to JOBSET.

        Option:
          -h, --help  display this help.
    """)

def rename(sql_session, cwd, src_path, dst):
    """Rename the given job/jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    cwd -- current working jobset (a path.Path object)
    src_path -- the source job/jobset to copy
    dst -- the destination

    """
    (base, dest_path) = path.get_new_paths(sql_session, dst, cwd)
    lp = len(dest_path)
    if lp == 0:
        sys.stderr.write("mv: no such destination jobset\n")
        return 1
    if lp > 1:
        sys.stderr.write("mv: `%s/%s': no such job or jobset\n" % \
                            (dest_path[0], base))
        return 1
    session = sql_session.open_session()
    dstjobset = session.query(job_main).filter_by(id=dest_path[0].id[-1]).one()
    try:
        src_job = session.query(job_main).filter_by(id=src_path.id[-1]).one()
    except:
        sys.stderr.write("mv: `%s': no such job or jobset\n" % src_path)
        sql_session.cancel_session(session)
        return 1
    src_job.parent = dstjobset.id
    src_job.name = base
    sql_session.close_session(session)
    return 0

def move(sql_session, paths):
    """Move jobs/jobsets in a new jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    paths -- list of jobs/jobsets to move (the last item is the destination)

    """

    dest = paths.pop()
    error = False
    session = sql_session.open_session()
    try:
        dest_jobset = session.query(job_main).filter_by(id=dest.id[-1]).one()
    except:
        sys.stderr.write("mv: no such destination job or jobset\n")
        sql_session.cancel_session(session)
        return 1
    if len(paths) > 1:
        if dest_jobset.type != 0:
            sys.stderr.write("mv: target `%s' is not a jobset\n" % dest)
            sql_session.cancel_session(session)
            return 1
        for p in paths:
            try:
                job = session.query(job_main).filter_by(id=p.id[-1]).one()
            except:
                sys.stderr.write("mv: `%s': no such job or jobset\n" % p)
                sql_session.cancel_session(session)
                return 1
            try:
                dest.id.index(job.id)
            except:
                job.parent = dest_jobset.id
            else:
                sys.stderr.write(
                    "mv: cannot move `%s' to a sub-jobset of itself\n" % \
                    job.name)
                sql_session.cancel_session(session)
                return 1
    else:
        try:
            job = session.query(job_main).filter_by(id=paths[0].id[-1]).one()
        except:
            sys.stderr.write("mv: `%s': no such job or jobset\n" % paths[0])
            sql_session.cancel_session(session)
            return 1
        if dest_jobset.type == 0:
            try:
                dest.id.index(job.id)
            except:
                job.parent = dest_jobset.id
            else:
                sys.stderr.write(
                    "mv: cannot move `%s' to a sub-jobset of itself\n" % \
                    job.name)
                sql_session.cancel_session(session)
                return 1
        else:
            sys.stderr.write("mv: target `%s' already exists\n" % dest)
            sql_session.cancel_session(session)
            return 1
    sql_session.close_session(session)
    return 0

def mv(sql_session, current_cwd, arguments):
    """Rename a job or jobset.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working jobset (a path.Path object)
    arguments -- list of arguments given to the mv command (list
                 of jobs/jobsets)

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write("mv: " + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    num_args = len(args)
    dest_rename = None
    if args:
        for i in range(0, num_args):
            p = path.get_paths(sql_session, args[i], current_cwd)
            # An argument is an unknow path (if it's not the last one)
            if not p:
                if i + 1 != num_args:
                    sys.stderr.write("mv: `%s': no such job or jobset\n" % \
                                        args[i])
                    return 1
                else:
                    dest_rename = args[i]
            paths.extend(p)
    else:
        sys.stderr.write("mv: missing job or jobset operand\n")
        return 1
    if not paths:
        sys.stderr.write("mv: `%s': no such job or jobset\n" % args[0])
        return 1
    lp = len(paths)
    # Rename
    if lp == 1:
        if num_args < 2:
            sys.stderr.write("mv: no enough operand or unknown job/jobset\n")
            return 1
        return rename(sql_session, current_cwd, paths[0], args[1])
    # Move
    if dest_rename is not None:
        sys.stderr.write("mv: target `%s' does not exist\n" % dest_rename)
        return 1
    return move(sql_session, paths)

