# Schedwi
# Copyright (C) 2011 Herve Quatremain
# 
# This file is part of Schedwi.
# 
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
# 
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to convert calendar path strings to calendar path objects."""

from sqlalchemy.orm.exc import NoResultFound
import sqlalchemy.orm.session

from tables.calendars import calendars
from tables.calendars_s import calendars_s

def _id2name(session, ids, workload=None):
    """Convert a calendar ID to its name."""
    names = list()
    for i in ids:
        if i > 1:
            if workload is None:
                q = session.query(calendars.name)
            else:
                q = session.query(calendars_s.name)
                q = q.filter(calendars_s.workload_date == workload)
            try:
                name, = q.filter_by(id=i).one()
            except NoResultFound:
                return None
            names.append(name)
    return '/' + '/'.join(names)

def _id2list(session, id, workload=None):
    """Convert a calendar ID to its hierarchy list.

    Return the list of IDs for the root (`/') to the given calendar.

    """
    ids = [ id ]
    while id > 0:
        if workload is None:
            q = session.query(calendars.parent)
        else:
            q = session.query(calendars_s.parent)
            q = q.filter(calendars_s.workload_date == workload)
        try:
            parent_id, = q.filter_by(id=id).one()
        except NoResultFound:
            return None
        ids.append(parent_id)
        id = parent_id
    ids.reverse()
    return ids

def _name2id(session, name_list, cwd=None, only_dir=False, workload=None):
    """Convert a calendar name to its list of IDs.

    Arguments:
    session -- SQLAlchemy session
    name_list -- list of calendar name parts
    cwd -- current working directory
    only_dir -- whether to only return the IDs if it's a calendar directory
    workload -- workload to use

    """
    if cwd is None:
        ids = [ 1 ]
    else:
        ids = list(cwd.id)
    for cal_name in name_list:
        if not cal_name or cal_name == '.':
            continue
        if cal_name == '..':
            try:
                ids.pop()
            except:
                pass
            if not ids:
                ids = [ 1 ]
            continue
        # Glob to SQL
        name = cal_name.replace("*","%").replace("?","_")
        if workload is None:
            q = session.query(calendars.id).filter(calendars.name.like(name))
            if only_dir:
                q = q.filter_by(entry_type=1)
            res = q.filter_by(parent=ids[-1]).order_by(calendars.name).first()
        else:
            q = session.query(calendars_s.id)
            q = q.filter(calendars_s.workload_date == workload)
            q = q.filter(calendars_s.name.like(name))
            if only_dir:
                q = q.filter_by(entry_type=1)
            res = q.filter_by(parent=ids[-1]).order_by(calendars_s.name).first()
        if res is None:
            return None
        ids.append(res[0])
    return ids


class PathCal(object):

    def __init__(self, sql_session, path=None, cwd=None, only_dir=False,
                 id=None, workload=None):
        """Build a new PathCal object.

        Arguments:
        sql_session -- SQLAlchemy session
        path -- calendar path (only if `id' is None)
        cwd -- current working directory.  By default `/'
        only_dir -- whether to only consider directories
        id -- calendar ID
        workload -- workload to use

        """
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            session = sql_session.open_session()
        else:
            session = sql_session
        if id is None:
            if path[0] == '/':
                self.id = _name2id(session, path.split('/'), None,
                                   only_dir, workload)
            else:
                self.id = _name2id(session, path.split('/'), cwd,
                                   only_dir, workload)
        else:
            self.id = _id2list(session, id, workload)
        if self.id is None:
            self.path = None
        else:
            self.path = _id2name(session, self.id, workload)
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)

    def __str__(self):
        return self.path

    def __cmp__(self, other):
        return cmp(self.id[-1], other.id[-1])

    def __nonzero__(self):
        if self.path is not None and self.id is not None:
            return True
        return False

def _sql_path_multi(session, h, id, res, only_dir=False, workload=None):
    if not h:
        # Check for duplicates
        for lst in res:
            for i in range(0, len(id)):
                if id[i] != lst[i]:
                    break
            else:
                return res
        res.append(id)
        return res
    cal_name = h.pop(0)
    if not cal_name or cal_name == '.':
        return _sql_path_multi(session, h, id, res, only_dir, workload)
    if cal_name == '..':
        if len(id) > 1:
            id.pop()
        return _sql_path_multi(session, h, id, res, only_dir, workload)
    # Glob to SQL
    cn = cal_name.replace("*","%").replace("?","_")
    if workload is None:
        q = session.query(calendars.id)
        q = q.filter(calendars.name.like(cn))
    else:
        q = session.query(calendars_s.id)
        q = q.filter(calendars_s.workload_date == workload)
        q = q.filter(calendars_s.name.like(cn))
    q = q.filter_by(parent=id[-1])
    if only_dir:
        q = q.filter_by(entry_type=1)
    for i, in q:
        myid = list(id)
        myid.append(i)
        try:
            _sql_path_multi(session, list(h), myid, res, only_dir, workload)
        except NoResultFound:
            pass
    return res

def get_paths(sql_session, path, cwd=None, only_dir=False, workload=None):
    """Return a list of calendars matching the given path."""
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    cals = list()
    if cwd is None or not path or path[0] == '/':
        _sql_path_multi(session, path.split('/'), [1], cals, only_dir, workload)
    else:
        _sql_path_multi(session, path.split('/'), list(cwd.id), cals, only_dir,
                        workload)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    res = list()
    for cal in cals:
        res.append(PathCal(sql_session, id=cal[-1], workload=workload))
    return res

def _split_path(path):
    try:
        pos = path.rindex("/")
    except ValueError:
        return ("./", path)
    return (path[:pos], path[pos+1:])

def get_new_paths(sql_session, path, cwd=None, only_dir=False, workload=None):
    """Split the given path in its basename and dirname.

    Return the tuple (basename, list_cal) with basename the calendar base name
    and list_cal a list of PathCal objects.

    """
    base, name = _split_path(path)
    res = get_paths(sql_session, base, cwd, only_dir, workload)
    return (name, res)

