-- Schedwi
--   Copyright (C) 2010 Herve Quatremain
-- 
--   This file is part of Schedwi.
-- 
--   Schedwi is free software; you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation; either version 3 of the License, or
--   (at your option) any later version.
-- 
--   Schedwi is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
-- 
--   You should have received a copy of the GNU General Public License
--   along with this program.  If not, see <http://www.gnu.org/licenses/>.
-- 

-- PostgreSQL tables definitions for the Schedwi server
-- 
-- Process this script with the psql command to create the required tables.
-- 
-- Example:
--
--  psql schedwidb < schedwi_postgresql.sql
-- 
-- The database - schedwidb in the example - must have been already created
-- prior to running the previous command.  As an example, the following command
-- create a schedwidb database:
--
--   createdb schedwidb
--
-- Also, a role for the current user must have been created.  This can be done
-- via the postgres account.  As an example:
--  su postgres
--     psql
--       CREATE ROLE schedwi;
--       ALTER ROLE schedwi LOGIN;
--       ALTER ROLE schedwi CREATEDB;
-- 

-- 
-- Version of the schedwi database and installation date
-- 
CREATE TABLE schedwi_version (
 version_number varchar(10),
 installed_date timestamp,

 UNIQUE (version_number)
);


-- 
-- Calendar definition.  Calendars are organized in a tree in this table
-- 
--          id --> Calendar ID
--      parent --> Parent folder
--        name --> Calendar/folder name
--  entry_type --> 0 (calendar) or 1 (folder)
-- description --> Host description
--     formula --> Calendar formula (if entry_type is 0)
--      
CREATE TABLE calendars (
  id            bigserial PRIMARY KEY,
  parent        bigint DEFAULT 0 NOT NULL,
  name          varchar(255) DEFAULT '' NOT NULL,
  entry_type    smallint DEFAULT 0 NOT NULL,
  description   text NOT NULL,
  formula       text NOT NULL,

  UNIQUE (parent,name)
);


-- 
-- Main job/jobset parameters
-- 
--           id --> Job/jobset ID
--       parent --> Parent jobset ID.  O means the jobset is the root one
--         name --> Name
--         type --> 0 (jobset) or 1 (job)
--      enabled --> 0 (disabled - this job/jobset is ignored) or 1 (enabled)
--  description --> Description
--      x and y --> Coordinates of the job/jobset icon in the canvas
--       cal_id --> Calendar ID (see table calendars).  0 means no calendar and
--                  the job/jobset will be run everyday
--   start_time --> Start time (HHMM). For example, for the job to start at
--                  9:17am, the start_time will be 917.  If -1 the job/jobset
--                  will start as soon as its parent jobset is running
-- 
CREATE TABLE job_main (
 id             bigserial PRIMARY KEY,
 parent         bigint DEFAULT 0 NOT NULL,
 name           varchar(255) DEFAULT '' NOT NULL,
 type           smallint DEFAULT 1 NOT NULL,
 enabled        smallint DEFAULT 0 NOT NULL,
 description    text NOT NULL,
 x              int,
 y              int,
 cal_id         bigint DEFAULT 0 NOT NULL,
 start_time     smallint DEFAULT -1 NOT NULL,

 FOREIGN KEY (cal_id) REFERENCES calendars(id)
);


-- 
-- Number of minutes for the job to be started past it start time
-- (see start_time in table job_main).  0 means no limit
-- 
CREATE TABLE job_start_limit (
 job_id         bigint DEFAULT 0 NOT NULL,
 start_limit    smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Max number of minutes for the job/jobset to complete once running.  0 means
-- no limit
-- 
CREATE TABLE job_max_duration (
 job_id         bigint DEFAULT 0 NOT NULL,
 max_duration   smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Number of retries when the job/jobset fail
-- 
CREATE TABLE job_retries (
 job_id         bigint DEFAULT 0 NOT NULL,
 retries        smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Number of minutes between retries
-- 
CREATE TABLE job_retries_interval (
 job_id           bigint DEFAULT 0 NOT NULL,
 retries_interval smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Username of the account under which the job command will be run
-- on the remote host
-- 
CREATE TABLE job_username (
 job_id         bigint DEFAULT 0 NOT NULL,
 username       varchar(100) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Output file name (stdout) for the job command on the remote host
-- 
CREATE TABLE job_file_out (
 job_id         bigint DEFAULT 0 NOT NULL,
 file_out       varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Error file name (stderr) for the job command on the remote host
-- 
CREATE TABLE job_file_err (
 job_id         bigint DEFAULT 0 NOT NULL,
 file_err       varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Specifies if the user account environment ($HOME/.profile or $HOME/.basrc for
-- instance) on the remote host must be loaded before running the job command
-- 0 --> Do not load the environment
-- 1 --> Load the environment
-- 
CREATE TABLE job_loadenv (
 job_id         bigint DEFAULT 0 NOT NULL,
 loadenv        smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Job command return code which means that the job is successful.  All
-- returned code lower or equal to this value mean that the job is successful.
-- All returned code greater than this value mean the job failed.
-- 
CREATE TABLE job_success_return_code (
 job_id         bigint DEFAULT 0 NOT NULL,
 success_ret    smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Remote command.  If command is `-' then there is no command associated
-- with the job and the job will immediately completed.  This may be used
-- to define steps like the Start or the End of a chain.
-- 
CREATE TABLE job_command (
 job_id         bigint DEFAULT 0 NOT NULL,
 command        varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Command arguments
-- A position of -1 means that the command has no arguments.
-- If for a specified job there is no row in this table, its parent
-- command arguments are used instead (and recursively up to the root
-- jobset until arguments are found).
-- 
CREATE TABLE job_arguments (
 job_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 argument       varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (job_id,position),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Group of environment variables
-- 
CREATE TABLE environments (
 id             bigserial PRIMARY KEY,
 name           varchar(100) DEFAULT '' NOT NULL,
 description    text NOT NULL,

 UNIQUE (name)
);


-- 
-- Environment variables
-- 
--     env_id --> Group of the environment variable (see table environments)
--   position --> Position in the group
--    env_key --> Name
--  env_value --> Value
-- 
CREATE TABLE environment_var (
 env_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 env_key        varchar(255) DEFAULT '' NOT NULL,
 env_value      varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (env_id,position),
 FOREIGN KEY (env_id) REFERENCES environments(id) ON DELETE CASCADE
);


-- 
-- Associate environments with jobs.  A position of -1 means that the job
-- does not have any environment.
-- The final job environment is the concatenation of all its parents 
-- environments plus its associated host environment (see
-- table host_environment)
-- 
CREATE TABLE job_environment (
 job_id         bigint DEFAULT 0 NOT NULL,
 env_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id,env_id),
 FOREIGN KEY (env_id) REFERENCES environments(id) ON DELETE CASCADE,
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Schedwi agent hosts
-- 
--          id --> Host ID
--    hostname --> Agent host name (a name of an address in dot notation) 
--     portnum --> Agent TCP port number (or a name in /etc/services)
--   sslenable --> Use SSL for the connexion (1 use SSL or O dont use SSL)
--     sslcert --> Agent's certificate (only used if sslenable is 1)
-- description --> Host description
-- 
CREATE TABLE hosts (
 id             bigserial PRIMARY KEY,
 hostname       varchar(255) DEFAULT '' NOT NULL,
 portnum        varchar(50) DEFAULT '2006' NOT NULL,
 sslenable      smallint DEFAULT 1 NOT NULL,
 sslcert        text NOT NULL,
 description    text NOT NULL,

 UNIQUE (hostname,portnum)
);


-- 
-- Associate environments with a host
-- 
CREATE TABLE host_environment (
 host_id        bigint DEFAULT 0 NOT NULL,
 env_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,

 PRIMARY KEY (host_id,env_id),
 FOREIGN KEY (host_id) REFERENCES hosts(id) ON DELETE CASCADE,
 FOREIGN KEY (env_id) REFERENCES environments(id) ON DELETE CASCADE
);


-- 
-- Associate a job/jobset to a host
-- 
CREATE TABLE job_host (
 job_id         bigint DEFAULT 0 NOT NULL,
 host_id        bigint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE,
 FOREIGN KEY (host_id) REFERENCES hosts(id)
);


-- 
-- File name to check
-- 
--      job_id --> Job ID
--     host_id --> Host ID on which the file presence must be checked
--    filename --> File name (variables are allowed in the name)
--       exist --> 0: file mustn't exist    1: file must exist
-- 
CREATE TABLE constraint_file (
 job_id         bigint DEFAULT 0 NOT NULL,
 host_id        bigint DEFAULT 0 NOT NULL,
 filename       text,
 exist          smallint,

 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE,
 FOREIGN KEY (host_id) REFERENCES hosts(id)
);


-- 
-- Canvas background image for a jobset
-- 
--                   job_id --> Jobset ID
--                    image --> Serialized GDK image
--          background_mode --> 0: Solid color
--                              1: Vertical gradient
--                              2: Horizontal gradient
--                              3: Image in top/left cornet
--                              4: Image centered
--                              5: Image tiled all over the canvas
--                              6: Image scaled (ratio width/height keeped)
--                              7: Image stretched to fill all the canvas
--   gradient_color_primary --> Background color (or first color for the
--                              gradient). Format: #RRGGBB
-- gradient_color_secondary --> Last color of the gradient
--             canvas_width --> Width of the canvas for this jobset
--            canvas_height --> Height of the canvas for this jobset
-- 
CREATE TABLE jobset_background (
 job_id                    bigint DEFAULT 0 NOT NULL,
 image                     bytea,
 background_mode           smallint DEFAULT 0 NOT NULL,
 gradient_color_primary    VARCHAR(100) DEFAULT '#FFFFFF' NOT NULL, 
 gradient_color_secondary  VARCHAR(100) DEFAULT '#808080' NOT NULL,
 canvas_width              int DEFAULT 640 NOT NULL,
 canvas_height             int DEFAULT 480 NOT NULL,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);
 

-- 
-- Job icon (used during design)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_default (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Job icon (Completed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_completed (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Job icon (Failed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_failed (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Job icon (Running)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_running (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Job icon (Waiting)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_waiting (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Jobset icon (used during design)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_default (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Jobset icon (Completed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_completed (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Jobset icon (Failed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_failed (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Jobset icon (Running)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_running (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Jobset icon (Waiting)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_waiting (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 
 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Chaining between jobs/jobsets
-- 
CREATE TABLE link (
 job_id_source             bigint DEFAULT 0 NOT NULL,
 job_id_destination        bigint DEFAULT 0 NOT NULL,
 required_status           smallint DEFAULT 1 NOT NULL,

 PRIMARY KEY (job_id_source,job_id_destination),
 FOREIGN KEY (job_id_source) REFERENCES job_main(id) ON DELETE CASCADE,
 FOREIGN KEY (job_id_destination) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- User commands
-- 
--      workload_date --> Workload date
--             job_id --> Job/jobset ID
--           username --> Name of the user who submits the command
--        insert_time --> Date/time at which the command was added in this table
--     command_status --> 0 (waiting to be run), 1 (Done) or 2 (Failed)
--            command --> 1 (change the status of the job) or 2 (stop the job)
--          parameter --> If command is 1, requested new status for the job
-- 
CREATE TABLE commands (
 workload_date  int DEFAULT 0 NOT NULL,
 job_id         bigint DEFAULT 0 NOT NULL,
 username       varchar(100) DEFAULT '' NOT NULL,
 insert_time    TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
 command_status smallint DEFAULT 0 NOT NULL, -- 0: To run   1: Done   2: Failed
 command        smallint DEFAULT 0 NOT NULL,
 parameter      smallint DEFAULT 0 NOT NULL,

 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);


-- 
-- Statistics
-- 
CREATE TABLE job_stat (
 job_id         bigint DEFAULT 0 NOT NULL,
 num_success    bigint DEFAULT 0 NOT NULL,
 num_failed     bigint DEFAULT 0 NOT NULL,
 total_duration bigint DEFAULT 0 NOT NULL,
 last_duration  bigint DEFAULT 0 NOT NULL,

 PRIMARY KEY (job_id),
 FOREIGN KEY (job_id) REFERENCES job_main(id) ON DELETE CASCADE
);



-- ---------------------------------------
-- -- Tables for the running job status --
-- ---------------------------------------


-- 
-- Calendar definition.  Calendars are organized in a tree in this table
-- 
--          id --> Calendar ID
--      parent --> Parent folder
--        name --> Calendar/folder name
--  entry_type --> 0 (calendar) or 1 (folder)
-- description --> Host description
--     formula --> Calendar formula (if entry_type is 0)
--      
CREATE TABLE calendars_s (
 id             bigint DEFAULT 0 NOT NULL,
 parent         bigint DEFAULT 0 NOT NULL,
 name           varchar(255) DEFAULT '' NOT NULL,
 entry_type     smallint DEFAULT 0 NOT NULL,
 description    text NOT NULL,
 formula        text NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,id),
 UNIQUE (workload_date,parent,name)
);


-- 
-- Main job/jobset parameters
-- 
-- workload_date --> Workload date
--            id --> Job/jobset ID
--        parent --> Parent jobset ID.  O means the jobset is the root one
--          name --> Name
--          type --> 0 (jobset) or 1 (job)
--       enabled --> 0 (disabled - this job/jobset is ignored) or 1 (enabled)
--   description --> Description
--       x and y --> Coordinates of the job/jobset icon in the canvas
--        cal_id --> Calendar ID (see table calendars).  0 means no calendar and
--                   the job/jobset will be run everyday
--    start_time --> Start time (HHMM). For example, for the job to start at
--                   9:17am, the start_time will be 917.  If -1 the job/jobset
--                   will start as soon as its parent jobset is running
-- 
CREATE TABLE job_main_s (
 id             bigint DEFAULT 0 NOT NULL,
 parent         bigint DEFAULT 0 NOT NULL,
 name           varchar(255) DEFAULT '' NOT NULL,
 type           smallint DEFAULT 1 NOT NULL,
 enabled        smallint DEFAULT 0 NOT NULL,
 description    text NOT NULL,
 x              int,
 y              int,
 cal_id         bigint DEFAULT 0 NOT NULL,
 start_time     smallint DEFAULT -1 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,id),
 FOREIGN KEY (cal_id,workload_date) REFERENCES calendars_s(id,workload_date)
);


-- 
-- Number of minutes for the job to be started past it start time
-- (see start_time in table job_main_s).  0 means no limit
-- 
CREATE TABLE job_start_limit_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 start_limit    smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Max number of minutes of the job/jobset to complete once running.  0 means
-- no limit
-- 
CREATE TABLE job_max_duration_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 max_duration   smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Number of retries when the job/jobset fail
-- 
CREATE TABLE job_retries_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 retries        smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Number of minutes between retries
-- 
CREATE TABLE job_retries_interval_s (
 job_id           bigint DEFAULT 0 NOT NULL,
 retries_interval smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Username of the account under which the job command will be run
-- on the remote host
-- 
CREATE TABLE job_username_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 username       varchar(100) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Output file name (stdout) for the job command on the remote host
-- 
CREATE TABLE job_file_out_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 file_out       varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Error file name (stderr) for the job command on the remote host
-- 
CREATE TABLE job_file_err_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 file_err       varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Specifies if the user account environment ($HOME/.profile or $HOME/.basrc for
-- instance) on the remote host must be loaded before running the job command
-- 0 --> Do not load the environment
-- 1 --> Load the environment
-- 
CREATE TABLE job_loadenv_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 loadenv        smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Job command return code which means that the job is successful.  All
-- returned code lower or equal to this value mean that the job is successful.
-- All returned code greater than this value mean the job failed.
-- 
CREATE TABLE job_success_return_code_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 success_ret    smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Remote command.  If command is `-' then there is no command associated
-- with the job and the job will immediately completed.  This may be used
-- to define steps like the Start or the End of a chain.
-- 
CREATE TABLE job_command_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 command        varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Command arguments
-- A position of -1 means that the command has no arguments.
-- If for a specified job there is no row in this table, its parent
-- command arguments are used instead (and recursively up to the root
-- jobset until arguments are found).
-- 
CREATE TABLE job_arguments_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 argument       varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id,position),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Group of environment variables
-- 
CREATE TABLE environments_s (
 id             bigint DEFAULT 0 NOT NULL,
 name           varchar(100) DEFAULT '' NOT NULL,
 description    text NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,id),
 UNIQUE (workload_date,name)
);


-- 
-- Environment variables
-- 
--     env_id --> Group of the environment variable (see table environments_s)
--   position --> Position in the group
--    env_key --> Name
--  env_value --> Value
-- 
CREATE TABLE environment_var_s (
 env_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 env_key        varchar(255) DEFAULT '' NOT NULL,
 env_value      varchar(255) DEFAULT '' NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,env_id,position),
 FOREIGN KEY (env_id,workload_date) REFERENCES environments_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Associate environments with jobs.  A position of -1 means that the job
-- does not have any environment.
-- The final job environment is the concatenation of all its parents 
-- environments plus its associated host environment (see
-- table host_environment_s)
-- 
CREATE TABLE job_environment_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 env_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id,env_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE,
 FOREIGN KEY (env_id,workload_date) REFERENCES environments_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Associate environments with a host
-- 
CREATE TABLE host_environment_s (
 host_id        bigint DEFAULT 0 NOT NULL,
 env_id         bigint DEFAULT 0 NOT NULL,
 position       smallint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,host_id,env_id),
 FOREIGN KEY (host_id) REFERENCES hosts(id),
 FOREIGN KEY (env_id,workload_date) REFERENCES environments_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Associate a job/jobset to a host
-- 
CREATE TABLE job_host_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 host_id        bigint DEFAULT 0 NOT NULL,
 workload_date  int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE,
 FOREIGN KEY (host_id) REFERENCES hosts(id)
);


-- 
-- File name to check
-- 
--      job_id --> Job ID
--     host_id --> Host ID on which the file presence must be checked
--    filename --> File name (variables are allowed in the name)
--       exist --> 0: file mustn't exist    1: file must exist
-- 
CREATE TABLE constraint_file_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 host_id        bigint DEFAULT 0 NOT NULL,
 filename       text,
 exist          smallint,
 workload_date  int DEFAULT 0 NOT NULL,

 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE,
 FOREIGN KEY (host_id) REFERENCES hosts(id)
);


-- 
-- Current status of jobs/jobsets
-- 
--      workload_date --> Workload date
--             job_id --> Job/jobset ID
--             status --> 1: Waiting
--                        2: Running
--                        3: Completed
--                        4: Failed
--         start_time --> Date/time at which this status was set
--          error_msg --> Message giving a reason of the current status
-- 
CREATE TABLE job_status (
 workload_date  int DEFAULT 0 NOT NULL,
 job_id         bigint DEFAULT 0 NOT NULL,
 status         smallint DEFAULT 1 NOT NULL,
 start_time     bigint DEFAULT 0 NOT NULL,
 retry_num      smallint DEFAULT 0 NOT NULL,
 error_msg      varchar(255) DEFAULT '' NOT NULL,

 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Canvas background image for a jobset
-- 
--                   job_id --> Jobset ID
--                    image --> Serialized GDK image
--          background_mode --> 0: Solid color
--                              1: Vertical gradient
--                              2: Horizontal gradient
--                              3: Image in top/left cornet
--                              4: Image centered
--                              5: Image tiled all over the canvas
--                              6: Image scaled (ratio width/height keeped)
--                              7: Image stretched to fill all the canvas
--   gradient_color_primary --> Background color (or first color for the
--                              gradient). Format: #RRGGBB
-- gradient_color_secondary --> Last color of the gradient
--             canvas_width --> Width of the canvas for this jobset
--            canvas_height --> Height of the canvas for this jobset
-- 
CREATE TABLE jobset_background_s (
 job_id                    bigint DEFAULT 0 NOT NULL,
 image                     bytea,
 background_mode           smallint DEFAULT 0 NOT NULL,
 gradient_color_primary    VARCHAR(100) DEFAULT '#FFFFFF' NOT NULL, 
 gradient_color_secondary  VARCHAR(100) DEFAULT '#808080' NOT NULL,
 canvas_width              int DEFAULT 640 NOT NULL,
 canvas_height             int DEFAULT 480 NOT NULL,
 workload_date             int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);
 

-- 
-- Job icon (used during design)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_default_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Job icon (Completed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_completed_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Job icon (Failed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_failed_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Job icon (Running)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_running_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Job icon (Waiting)
-- The icon is a serialized GDK image
-- 
CREATE TABLE job_icon_waiting_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Jobset icon (used during design)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_default_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Jobset icon (Completed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_completed_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Jobset icon (Failed)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_failed_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Jobset icon (Running)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_running_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Jobset icon (Waiting)
-- The icon is a serialized GDK image
-- 
CREATE TABLE jobset_icon_waiting_s (
 job_id         bigint DEFAULT 0 NOT NULL,
 icon           bytea,
 workload_date  int DEFAULT 0 NOT NULL,
 
 PRIMARY KEY (workload_date,job_id),
 FOREIGN KEY (job_id,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);


-- 
-- Chaining between jobs/jobsets
-- 
CREATE TABLE link_s (
 job_id_source             bigint DEFAULT 0 NOT NULL,
 job_id_destination        bigint DEFAULT 0 NOT NULL,
 required_status           smallint DEFAULT 1 NOT NULL,
 workload_date             int DEFAULT 0 NOT NULL,

 PRIMARY KEY (workload_date,job_id_source,job_id_destination),
 FOREIGN KEY (job_id_source,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE,
 FOREIGN KEY (job_id_destination,workload_date) REFERENCES job_main_s(id,workload_date) ON DELETE CASCADE
);



-- 
-- Create the root job
-- 
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (0,0,'',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1,0,'Calendars',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (2,1,'Custom',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (3,1,'Samples',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (4,3,'Every day',0,'*','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (5,3,'Every Monday',0,'*/Monday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (6,3,'Every Tuesday',0,'*/Tuesday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (7,3,'Every Wednesday',0,'*/Wednesday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (8,3,'Every Thursday',0,'*/Thursday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (9,3,'Every Friday',0,'*/Friday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (10,3,'Every Saturday',0,'*/Saturday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (11,3,'Every Sunday',0,'*/Sunday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (12,3,'Every weekday',0,'*/Open','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (13,3,'Every weekend',0,'*/Saturday,Sunday','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (14,3,'First day of the month',0,'*/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (15,3,'Last day of the month',0,'*/-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (16,3,'First open day of the month',0,'*/Open/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (17,3,'Last open day of the month',0,'*/Open/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (18,3,'Worldwide',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (19,18,'Germany',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (20,19,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (21,19,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (22,19,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (23,19,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (24,19,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (25,19,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (26,19,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (27,19,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (28,19,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (29,19,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (30,19,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (31,19,'German Reunification Day',0,'10/3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (32,19,'Reformation Day',0,'10/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (33,19,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (34,19,'Repentance Day',0,'11/Wed/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (35,19,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (36,19,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (37,19,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (38,19,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (39,19,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (40,19,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (41,18,'Czech Republic',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (42,41,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (43,41,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (44,41,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (45,41,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (46,41,'Liberation Day',0,'5/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (47,41,'Saint Cyril and Saint Methodius Day',0,'7/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (48,41,'Jan Hus Day',0,'7/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (49,41,'Czech Statehood Day',0,'9/28','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (50,41,'National Day',0,'10/28','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (51,41,'Struggle for Freedom and Democracy Day',0,'11/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (52,41,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (53,41,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (54,41,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (55,41,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (56,41,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (57,18,'Belgium',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (58,57,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (59,57,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (60,57,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (61,57,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (62,57,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (63,57,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (64,57,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (65,57,'Flemish Community Day',0,'7/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (66,57,'National Day',0,'7/21','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (67,57,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (68,57,'French Community Day',0,'9/27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (69,57,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (70,57,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (71,57,'Armistice Day',0,'11/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (72,57,'Dynasty Day',0,'11/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (73,57,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (74,57,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (75,57,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (76,57,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (77,18,'Austria',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (78,77,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (79,77,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (80,77,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (81,77,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (82,77,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (83,77,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (84,77,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (85,77,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (86,77,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (87,77,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (88,77,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (89,77,'National Day',0,'10/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (90,77,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (91,77,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (92,77,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (93,77,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (94,77,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (95,77,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (96,18,'Bulgaria',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (97,96,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (98,96,'Liberation Day',0,'3/3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (99,96,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (100,96,'Orthodox Easter',0,'Paskha','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (101,96,'Orthodox Easter Monday',0,'Paskha+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (102,96,'Saint George''s Day',0,'5/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (103,96,'Saint Cyril and Saint Methodius Day',0,'5/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (104,96,'Unification Day',0,'9/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (105,96,'Independence Day',0,'9/22','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (106,96,'Leaders of the Bulgarian Revival Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (107,96,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (108,96,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (109,96,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (110,96,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (111,96,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (112,18,'Croatia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (113,112,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (114,112,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (115,112,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (116,112,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (117,112,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (118,112,'Antifacism Day',0,'6/22','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (119,112,'National Day',0,'6/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (120,112,'Thanksgiving Day',0,'8/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (121,112,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (122,112,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (123,112,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (124,112,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (125,112,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (126,112,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (127,18,'Denmark',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (128,127,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (129,127,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (130,127,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (131,127,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (132,127,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (133,127,'Store Bededag',0,'Easter+26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (134,127,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (135,127,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (136,127,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (137,127,'Constitution Day',0,'6/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (138,127,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (139,127,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (140,127,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (141,127,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (142,127,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (143,18,'Estonia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (144,143,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (145,143,'Independence Day',0,'2/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (146,143,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (147,143,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (148,143,'Spring Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (149,143,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (150,143,'Victory Day',0,'6/23','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (151,143,'Midsummer''s Day',0,'6/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (152,143,'Independence Restoration Day',0,'8/20','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (153,143,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (154,143,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (155,143,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (156,143,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (157,18,'France',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (158,157,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (159,157,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (160,157,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (161,157,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (162,157,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (163,157,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (164,157,'Victory Day',0,'5/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (165,157,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (166,157,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (167,157,'Bastille Day',0,'7/14','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (168,157,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (169,157,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (170,157,'Armistice Day',0,'11/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (171,157,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (172,157,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (173,157,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (174,18,'Hungary',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (175,174,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (176,174,'Revolution Day',0,'3/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (177,174,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (178,174,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (179,174,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (180,174,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (181,174,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (182,174,'Constitution Day',0,'8/20','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (183,174,'Republic Day',0,'10/23','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (184,174,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (185,174,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (186,174,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (187,174,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (188,174,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (189,18,'Italy',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (190,189,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (191,189,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (192,189,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (193,189,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (194,189,'Liberation Day',0,'4/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (195,189,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (196,189,'Republic Day',0,'6/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (197,189,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (198,189,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (199,189,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (200,189,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (201,189,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (202,189,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (203,189,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (204,18,'Latvia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (205,204,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (206,204,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (207,204,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (208,204,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (209,204,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (210,204,'Ligo Day',0,'6/23','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (211,204,'Saint John''s Day',0,'6/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (212,204,'National Day',0,'11/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (213,204,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (214,204,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (215,204,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (216,204,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (217,204,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (218,18,'The Netherlands',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (219,218,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (220,218,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (221,218,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (222,218,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (223,218,'Queen''s Day',0,'4/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (224,218,'Liberation Day',0,'5/5/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (225,218,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (226,218,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (227,218,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (228,218,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (229,218,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (230,218,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (231,218,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (232,18,'Poland',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (233,232,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (234,232,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (235,232,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (236,232,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (237,232,'Constitution Day',0,'5/3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (238,232,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (239,232,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (240,232,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (241,232,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (242,232,'Independence Day',0,'11/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (243,232,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (244,232,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (245,232,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (246,232,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (247,18,'Portugal',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (248,247,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (249,247,'Shrove Tuesday',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (250,247,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (251,247,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (252,247,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (253,247,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (254,247,'Liberation Day',0,'4/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (255,247,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (256,247,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (257,247,'National Day',0,'6/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (258,247,'Saint Anthony''s Day',0,'6/13','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (259,247,'Saint John''s Day',0,'6/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (260,247,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (261,247,'Republic Day',0,'10/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (262,247,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (263,247,'Independence Day',0,'12/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (264,247,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (265,247,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (266,247,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (267,247,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (268,247,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (269,18,'Romania',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (270,269,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (271,269,'Second Day of New Year',0,'1/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (272,269,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (273,269,'Orthodox Easter',0,'Paskha','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (274,269,'Orthodox Easter Monday',0,'Paskha+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (275,269,'National Day',0,'12/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (276,269,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (277,269,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (278,269,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (279,269,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (280,18,'Slovakia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (281,280,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (282,280,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (283,280,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (284,280,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (285,280,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (286,280,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (287,280,'Liberation Day',0,'5/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (288,280,'Saint Cyril and Saint Methodius Day',0,'7/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (289,280,'Slovak National Uprising Anniversary',0,'8/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (290,280,'Constitution Day',0,'9/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (291,280,'Our Lady of Sorrows'' Day',0,'9/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (292,280,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (293,280,'Freedom and Democracy Day',0,'11/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (294,280,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (295,280,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (296,280,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (297,280,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (298,280,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (299,18,'Slovenia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (300,299,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (301,299,'Second Day of New Year',0,'1/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (302,299,'Cultural Day',0,'2/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (303,299,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (304,299,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (305,299,'Resistance Day',0,'4/27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (306,299,'Labour Day',0,'5/1-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (307,299,'National Day',0,'6/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (308,299,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (309,299,'Reformation Day',0,'10/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (310,299,'Remembrance Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (311,299,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (312,299,'Independence Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (313,299,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (314,299,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (315,18,'Spain',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (316,315,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (317,315,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (318,315,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (319,315,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (320,315,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (321,315,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (322,315,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (323,315,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (324,315,'National Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (325,315,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (326,315,'Constitution Day',0,'12/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (327,315,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (328,315,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (329,315,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (330,315,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (331,18,'Switzerland',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (332,331,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (333,331,'Saint Berchtold''s Day',0,'1/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (334,331,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (335,331,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (336,331,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (337,331,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (338,331,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (339,331,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (340,331,'National Day',0,'8/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (341,331,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (342,331,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (343,331,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (344,331,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (345,331,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (346,331,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (347,331,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (348,331,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (349,331,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (350,18,'United Kingdom',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (351,350,'New Year''s Day',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (352,350,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (353,350,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (354,350,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (355,350,'Early May Bank Holiday',0,'5/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (356,350,'Spring Bank Holiday',0,'5/Mon/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (357,350,'Summer Bank Holiday',0,'8/Mon/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (358,350,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (359,350,'Boxing Day',0,'12/26/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (360,350,'Christmas Holiday',0,'12/25/Sat+3,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (361,350,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (362,350,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (363,18,'Brazil',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (364,363,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (365,363,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (366,363,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (367,363,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (368,363,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (369,363,'Tiradentes Day',0,'4/21','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (370,363,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (371,363,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (372,363,'Independence Day',0,'9/7','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (373,363,'Our Lady Aparecida Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (374,363,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (375,363,'Republic Day',0,'11/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (376,363,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (377,363,'Daylight Saving Time (start)',0,'11/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (378,363,'Daylight Saving Time (end)',0,'2/Sun/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (379,18,'Russia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (380,379,'New Year''s Day',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (381,379,'Second Day of New Year',0,'1/2/Sat+2,Sun+2,Mon+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (382,379,'Orthodox Christmas Day',0,'1/7/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (383,379,'Orthodox New Year',0,'1/14','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (384,379,'Defenders of the Motherland Day',0,'2/23/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (385,379,'Women''s Day',0,'3/8/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (386,379,'Orthodox Good Friday',0,'Paskha-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (387,379,'Orthodox Easter',0,'Paskha','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (388,379,'Orthodox Easter Monday',0,'Paskha+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (389,379,'Labour Day',0,'5/1-2/Sat+2,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (390,379,'Victory Day',0,'5/9/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (391,379,'Independence Day',0,'6/12/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (392,379,'Accord and Reconciliation Day',0,'11/7/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (393,379,'Constitution Day',0,'12/12/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (394,379,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (395,379,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (396,18,'Mexico',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (397,396,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (398,396,'Constitution Day',0,'2/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (399,396,'Benito Juarez''s Birthday',0,'3/21','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (400,396,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (401,396,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (402,396,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (403,396,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (404,396,'Battle of Puebla Day',0,'5/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (405,396,'President''s State of the Nation Address',0,'9/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (406,396,'Independence Day',0,'9/16','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (407,396,'Dia de la Raza / Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (408,396,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (409,396,'Revolution Day',0,'11/20','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (410,396,'Virgin of Guadalupe Day',0,'12/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (411,396,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (412,396,'Daylight Saving Time (start)',0,'4/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (413,396,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (414,18,'Philippines',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (415,414,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (416,414,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (417,414,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (418,414,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (419,414,'Day of Valor (Bataan Day)',0,'4/9','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (420,414,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (421,414,'Independence Day',0,'6/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (422,414,'Manila Day',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (423,414,'Heroes'' Day',0,'8/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (424,414,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (425,414,'Bonifacio Day',0,'11/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (426,414,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (427,414,'Rizal Day',0,'12/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (428,414,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (429,18,'United States',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (430,429,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (431,429,'New Year''s Day Holiday',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (432,429,'Martin Luther King''s Day',0,'1/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (433,429,'Presidents''Day',0,'2/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (434,429,'Memorial Day',0,'5/Mon/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (435,429,'Independence Day',0,'7/4/Sat-1,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (436,429,'Labour Day',0,'9/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (437,429,'Columbus Day',0,'10/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (438,429,'Veterans Day',0,'11/11/Sat-1,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (439,429,'Thanksgiving Day',0,'11/Thu/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (440,429,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (441,429,'Christmas Holiday',0,'12/25/Sat-1,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (442,429,'New Year''s Day Holiday (next year)',0,'12/25/Sat+6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (443,429,'Daylight Saving Time (start)',0,'4/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (444,429,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (445,18,'Congo (Dem. Rep.)',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (446,445,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (447,445,'Martyr''s Day',0,'1/4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (448,445,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (449,445,'Independence Day',0,'6/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (450,445,'Parents'' Day',0,'8/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (451,445,'Youth Day',0,'10/14','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (452,445,'Army Day',0,'11/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (453,445,'New Regime Day',0,'11/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (454,445,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (455,18,'Ukraine',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (456,455,'New Year''s Day',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (457,455,'Orthodox Christmas Day',0,'1/7/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (458,455,'Orthodox New Year',0,'1/14/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (459,455,'Women''s Day',0,'3/8/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (460,455,'Orthodox Good Friday',0,'Paskha-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (461,455,'Orthodox Easter',0,'Paskha','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (462,455,'Orthodox Easter Monday',0,'Paskha+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (463,455,'Labour Day',0,'5/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (464,455,'Victory Day',0,'5/9/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (465,455,'Radunitsa (Orthodox Trinity)',0,'Paskha+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (466,455,'Radunitsa Monday',0,'Paskha+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (467,455,'Constitution Day',0,'6/28/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (468,455,'Independence Day',0,'8/24/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (469,455,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (470,455,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (471,18,'South Africa',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (472,471,'New Year''s Day',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (473,471,'Human Righths''s Day',0,'3/21/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (474,471,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (475,471,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (476,471,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (477,471,'Freedom Day',0,'4/27/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (478,471,'Workers'' Day',0,'5/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (479,471,'Youth Day',0,'6/16/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (480,471,'Women''s Day',0,'8/9/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (481,471,'Heritage Day',0,'9/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (482,471,'Reconciliation Day',0,'12/16/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (483,471,'Christmas Day',0,'12/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (484,471,'Day of Goodwill',0,'12/26/Mon+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (485,18,'Colombia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (486,485,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (487,485,'Epiphany',0,'1/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (488,485,'San Jose Day',0,'3/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (489,485,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (490,485,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (491,485,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (492,485,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (493,485,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (494,485,'Sagrado Corazon Day',0,'Easter+71','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (495,485,'Saint Peter and Saint Paul''s Day',0,'7/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (496,485,'Independence Day',0,'7/20','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (497,485,'Battle of Boyaca',0,'8/7','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (498,485,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (499,485,'Dia de la Raza',0,'10/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (500,485,'All Saints'' Day',0,'11/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (501,485,'Cartagena Independence Day',0,'11/Mon/-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (502,485,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (503,485,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (504,18,'Argentina',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (505,504,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (506,504,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (507,504,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (508,504,'May Day Revolution',0,'5/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (509,504,'Malvinas Day',0,'6/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (510,504,'Flag Day',0,'6/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (511,504,'Independence Day',0,'7/9','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (512,504,'Death of General Jose de San Martin',0,'8/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (513,504,'Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (514,504,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (515,504,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (516,18,'Canada',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (517,516,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (518,516,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (519,516,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (520,516,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (521,516,'Victoria Day',0,'5/Mon/-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (522,516,'Saint Jean Baptiste',0,'6/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (523,516,'Canada Day',0,'7/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (524,516,'Civic Holiday',0,'8/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (525,516,'Labour Day',0,'9/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (526,516,'Thanksgiving Day',0,'10/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (527,516,'Remembrance Day',0,'11/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (528,516,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (529,516,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (530,516,'Daylight Saving Time (start)',0,'4/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (531,516,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (532,18,'Peru',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (533,532,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (534,532,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (535,532,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (536,532,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (537,532,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (538,532,'Inti Raymi Day',0,'6/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (539,532,'Saint Peter and Saint Paul''s Day',0,'6/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (540,532,'Independence Day',0,'7/28-29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (541,532,'Santa Rosa de Lima Day',0,'8/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (542,532,'Battle of Angamos Day',0,'10/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (543,532,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (544,532,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (545,532,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (546,18,'Venezuela',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (547,546,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (548,546,'Epiphany',0,'1/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (549,546,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (550,546,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (551,546,'San Jose Day',0,'3/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (552,546,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (553,546,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (554,546,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (555,546,'Declaration of Independence Day',0,'4/19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (556,546,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (557,546,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (558,546,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (559,546,'Battle of Carabobo Day',0,'6/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (560,546,'Saint Peter and Saint Paul''s Day',0,'6/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (561,546,'Independence Day',0,'7/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (562,546,'Simon Bolivar''s Birthday',0,'7/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (563,546,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (564,546,'Dia de la Raza / Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (565,546,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (566,546,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (567,546,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (568,546,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (569,18,'Mozambique',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (570,569,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (571,569,'Heroes'' Day',0,'2/3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (572,569,'Women''s Day',0,'4/7','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (573,569,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (574,569,'Independence Day',0,'6/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (575,569,'Victory Day',0,'9/7','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (576,569,'Armed Forces Day',0,'9/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (577,569,'Maputo Day',0,'11/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (578,569,'Christmas Day / Family Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (579,18,'Australia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (580,579,'New Year''s Day',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (581,579,'Australia Day',0,'1/26/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (582,579,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (583,579,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (584,579,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (585,579,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (586,579,'ANZAC Day',0,'4/25/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (587,579,'Queen''s Birthday',0,'6/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (588,579,'Melbourne Cup Day',0,'11/Tue/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (589,579,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (590,579,'Christmas Holiday',0,'12/25/Sat+2,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (591,579,'Boxing Day',0,'12/26/Sat+2,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (592,579,'Daylight Saving Time (start)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (593,579,'Daylight Saving Time (end)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (594,18,'Madagascar',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (595,594,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (596,594,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (597,594,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (598,594,'Memorial Day',0,'3/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (599,594,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (600,594,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (601,594,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (602,594,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (603,594,'O .A.U. Day',0,'5/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (604,594,'Independence Day',0,'6/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (605,594,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (606,594,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (607,594,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (608,18,'Chile',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (609,608,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (610,608,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (611,608,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (612,608,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (613,608,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (614,608,'Navy day',0,'5/21','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (615,608,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (616,608,'Saint Peter and Saint Paul''s Day',0,'6/26/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (617,608,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (618,608,'Independence Day',0,'9/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (619,608,'Armed Forces Day',0,'9/19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (620,608,'Columbus Day',0,'10/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (621,608,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (622,608,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (623,608,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (624,608,'Daylight Saving Time (start)',0,'10/Sat/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (625,608,'Daylight Saving Time (end)',0,'3/Sat/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (626,18,'Ecuador',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (627,626,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (628,626,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (629,626,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (630,626,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (631,626,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (632,626,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (633,626,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (634,626,'Battle of Pichincha Day',0,'5/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (635,626,'Simon Bolivar Day',0,'7/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (636,626,'Independence Day',0,'8/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (637,626,'Guayaquil Day',0,'10/9','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (638,626,'Dia de la Raza / Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (639,626,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (640,626,'Cuenca Day',0,'11/3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (641,626,'Quito Day',0,'12/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (642,626,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (643,626,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (644,18,'Guatemala',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (645,644,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (646,644,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (647,644,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (648,644,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (649,644,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (650,644,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (651,644,'Army Day',0,'6/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (652,644,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (653,644,'Independence Day',0,'9/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (654,644,'Revolution Day',0,'10/20','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (655,644,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (656,644,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (657,644,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (658,644,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (659,18,'Angola',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (660,659,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (661,659,'Memorial Day',0,'1/4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (662,659,'Liberation Day',0,'2/4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (663,659,'Mardi Gras',0,'Easter-40','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (664,659,'Women''s Day',0,'3/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (665,659,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (666,659,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (667,659,'Peace Day',0,'4/4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (668,659,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (669,659,'O .A.U. Day',0,'5/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (670,659,'children''s Day',0,'6/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (671,659,'National Day',0,'9/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (672,659,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (673,659,'Independence Day',0,'11/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (674,659,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (675,18,'Zimbabwe',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (676,675,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (677,675,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (678,675,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (679,675,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (680,675,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (681,675,'Independence Day',0,'4/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (682,675,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (683,675,'Africa day',0,'5/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (684,675,'Heroes'' Day',0,'8/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (685,675,'Armed Forces Day',0,'8/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (686,675,'Unity Day',0,'12/22/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (687,675,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (688,675,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (689,18,'Cuba',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (690,689,'Liberation Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (691,689,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (692,689,'Rebellion Anniversary',0,'7/25-27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (693,689,'Independence Wars Day',0,'10/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (694,689,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (695,689,'Daylight Saving Time (start)',0,'4/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (696,689,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (697,18,'Greece',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (698,697,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (699,697,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (700,697,'Shrove Monday',0,'Paskha-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (701,697,'Ash Wednesday',0,'Paskha-46','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (702,697,'Independence Day',0,'3/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (703,697,'Orthodox Good Friday',0,'Paskha-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (704,697,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (705,697,'Orthodox Easter',0,'Paskha','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (706,697,'Orthodox Easter Monday',0,'Paskha+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (707,697,'Whit Sunday',0,'Paskha+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (708,697,'Whit Monday',0,'Paskha+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (709,697,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (710,697,'Ochi Day',0,'10/28','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (711,697,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (712,697,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (713,697,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (714,697,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (715,18,'Luxembourg',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (716,715,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (717,715,'Carnival',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (718,715,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (719,715,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (720,715,'May Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (721,715,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (722,715,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (723,715,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (724,715,'National Day / Grand Duc''s Birthday',0,'6/23/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (725,715,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (726,715,'Luxembourg City Kermesse',0,'9/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (727,715,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (728,715,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (729,715,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (730,715,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (731,715,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (732,18,'Ireland',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (733,732,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (734,732,'Saint Patrick''s Day',0,'3/17/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (735,732,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (736,732,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (737,732,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (738,732,'May Bank Holiday',0,'5/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (739,732,'June Bank Holiday',0,'6/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (740,732,'Summer Bank Holiday',0,'8/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (741,732,'Hallowe''en Bank Holiday',0,'10/Mon/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (742,732,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (743,732,'Saint Stephen''s Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (744,732,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (745,732,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (746,18,'Cyprus',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (747,746,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (748,746,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (749,746,'Clean Monday / Start of Lent',0,'Paskha-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (750,746,'Greek National Day',0,'3/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (751,746,'Greek Cypriot National Day',0,'4/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (752,746,'Orthodox Good Friday',0,'Paskha-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (753,746,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (754,746,'Orthodox Easter',0,'Paskha','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (755,746,'Orthodox Easter Monday',0,'Paskha+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (756,746,'Orthodox Easter Tuesday',0,'Paskha+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (757,746,'Whit Sunday',0,'Paskha+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (758,746,'Whit Monday',0,'Paskha+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (759,746,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (760,746,'Independence Day',0,'10/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (761,746,'Ochi Day',0,'10/28','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (762,746,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (763,746,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (764,746,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (765,746,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (766,18,'Malta',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (767,766,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (768,766,'Saint Paul''s Shipwreck',0,'2/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (769,766,'Saint Joseph''s Day',0,'3/19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (770,766,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (771,766,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (772,766,'Freedom Day',0,'3/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (773,766,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (774,766,'Commemoration of 1919 Uprising',0,'6/7','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (775,766,'Saint Peter and Saint Paul''s Day',0,'6/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (776,766,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (777,766,'Our Lady of the Victories'' Day',0,'9/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (778,766,'Independence Day',0,'9/21','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (779,766,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (780,766,'Republic Day',0,'12/13','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (781,766,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (782,766,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (783,766,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (784,18,'Lithuania',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (785,784,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (786,784,'Independence Day',0,'2/16','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (787,784,'Restoration Day',0,'3/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (788,784,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (789,784,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (790,784,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (791,784,'Statehood Day / Mindaugas'' Coronation',0,'7/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (792,784,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (793,784,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (794,784,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (795,784,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (796,784,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (797,784,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (798,18,'Sweden',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (799,798,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (800,798,'Epiphany Eve',0,'1/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (801,798,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (802,798,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (803,798,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (804,798,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (805,798,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (806,798,'Valborg''s Eve',0,'4/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (807,798,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (808,798,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (809,798,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (810,798,'National Day',0,'6/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (811,798,'Midsummer''s Eve',0,'6/Fri/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (812,798,'Midsummer''s Day',0,'6/Sat/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (813,798,'All Hallow''s Eve',0,'11/Fri/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (814,798,'All Saints'' Day',0,'11/Sat/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (815,798,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (816,798,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (817,798,'Second Day of Christmas',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (818,798,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (819,798,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (820,798,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (821,18,'Finland',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (822,821,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (823,821,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (824,821,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (825,821,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (826,821,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (827,821,'May Day / Vappu',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (828,821,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (829,821,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (830,821,'Midsummer''s Eve',0,'6/Fri/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (831,821,'Midsummer''s Day',0,'6/Sat/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (832,821,'All Saints'' Day',0,'11/Sat/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (833,821,'Independence Day',0,'12/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (834,821,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (835,821,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (836,821,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (837,821,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (838,821,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (839,18,'Norway',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (840,839,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (841,839,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (842,839,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (843,839,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (844,839,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (845,839,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (846,839,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (847,839,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (848,839,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (849,839,'Constitution Day',0,'5/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (850,839,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (851,839,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (852,839,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (853,839,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (854,18,'Bolivia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (855,854,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (856,854,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (857,854,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (858,854,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (859,854,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (860,854,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (861,854,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (862,854,'Independence Day',0,'8/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (863,854,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (864,854,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (865,18,'Paraguay',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (866,865,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (867,865,'Battle of Cerro Cora Day',0,'3/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (868,865,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (869,865,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (870,865,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (871,865,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (872,865,'Independence Day',0,'5/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (873,865,'Chaco Armistice Day',0,'6/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (874,865,'Founding of Asuncion Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (875,865,'Battle of Boqueron Day',0,'9/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (876,865,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (877,865,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (878,865,'Daylight Saving Time (start)',0,'9/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (879,865,'Daylight Saving Time (end)',0,'4/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (880,18,'New Zealand',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (881,880,'New Year''s Day',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (882,880,'Second Day of New Year',0,'1/2/Sat+2,Sun+2,Mon+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (883,880,'Waitangi Day',0,'2/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (884,880,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (885,880,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (886,880,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (887,880,'ANZAC Day',0,'4/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (888,880,'Queen''s Birthday',0,'6/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (889,880,'Labour Day',0,'10/Mon/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (890,880,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (891,880,'Boxing Day',0,'12/26/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (892,880,'Christmas Holiday',0,'12/25/Sat+3,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (893,880,'Daylight Saving Time (start)',0,'10/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (894,880,'Daylight Saving Time (end)',0,'3/Sun/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (895,18,'Bahamas',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (896,895,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (897,895,'New Year''s Holiday',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (898,895,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (899,895,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (900,895,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (901,895,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (902,895,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (903,895,'Labour Day',0,'6/Fri/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (904,895,'Independence Day',0,'7/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (905,895,'Emancipation Day',0,'8/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (906,895,'Discovery Day',0,'10/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (907,895,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (908,895,'Christmas Holiday',0,'12/25/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (909,895,'Boxing Day',0,'12/26/Sat+2,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (910,895,'Daylight Saving Time (start)',0,'4/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (911,895,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (912,18,'Barbados',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (913,912,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (914,912,'New Year''s Holiday',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (915,912,'Errol Barrow Day',0,'1/21/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (916,912,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (917,912,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (918,912,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (919,912,'Heroes''Day',0,'4/28/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (920,912,'Labour Day',0,'5/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (921,912,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (922,912,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (923,912,'Kadooment Day',0,'8/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (924,912,'Emancipation Day',0,'8/Tue/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (925,912,'Independence Day',0,'11/30/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (926,912,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (927,912,'Christmas Holiday',0,'12/25/Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (928,912,'Boxing Day',0,'12/26/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (929,18,'Belize',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (930,929,'New Year''s Day',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (931,929,'Baron Bliss Day',0,'3/9/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (932,929,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (933,929,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (934,929,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (935,929,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (936,929,'Labour Day',0,'5/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (937,929,'Commonwealth Day',0,'5/24/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (938,929,'National Day',0,'9/10/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (939,929,'Independence Day',0,'9/21/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (940,929,'Pan American Day / Columbus Day',0,'10/12/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (941,929,'Garifuna Day',0,'11/19/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (942,929,'Bank Holiday',0,'12/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (943,929,'Christmas Day',0,'12/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (944,929,'Second Day of Christmas',0,'12/26/Sun+1,Mon+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (945,18,'Bermuda',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (946,945,'New Year''s Day',0,'1/1/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (947,945,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (948,945,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (949,945,'Bermuda Day',0,'5/24/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (950,945,'Queen''s Birthday',0,'6/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (951,945,'Emancipation Day',0,'7/Thu/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (952,945,'Somers Day',0,'7/Fri/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (953,945,'Labour Day',0,'9/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (954,945,'Remembrance Day',0,'11/11/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (955,945,'Christmas Day',0,'12/25/Sat+2,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (956,945,'Boxing Day',0,'12/26/Sat+2,Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (957,18,'Botswana',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (958,957,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (959,957,'Second Day of New Year',0,'1/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (960,957,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (961,957,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (962,957,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (963,957,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (964,957,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (965,957,'Sir Seretse Khama Day',0,'7/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (966,957,'President''s Day (first day)',0,'7/Tue/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (967,957,'President''s Day (second day)',0,'7/Wed/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (968,957,'Botswana Day (first day)',0,'9/30','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (969,957,'Botswana Day (second day)',0,'10/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (970,957,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (971,957,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (972,957,'Third Day of Christmas',0,'12/27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (973,18,'Costa Rica',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (974,973,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (975,973,'San Jose Day',0,'3/19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (976,973,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (977,973,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (978,973,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (979,973,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (980,973,'Juan Santamaria Day',0,'4/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (981,973,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (982,973,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (983,973,'Saint Peter and Saint Paul''s Day',0,'6/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (984,973,'Guanacoste Annexation Day',0,'7/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (985,973,'Virgin of the Angels Day',0,'8/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (986,973,'Independence Day',0,'9/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (987,973,'Dia de la Raza / Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (988,973,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (989,973,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (990,973,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (991,973,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (992,18,'Zambia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (993,992,'New Year''s Day',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (994,992,'Youth Day',0,'3/Fri/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (995,992,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (996,992,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (997,992,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (998,992,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (999,992,'Labour Day',0,'5/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1000,992,'Africa Day',0,'5/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1001,992,'Heroes'' Day',0,'7/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1002,992,'Unity Day',0,'7/Tue/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1003,992,'Farmers'' Day',0,'8/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1004,992,'Independence Day',0,'10/24/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1005,992,'Christmas Day',0,'12/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1006,18,'Uruguay',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1007,1006,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1008,1006,'Epiphany',0,'1/Mon/-4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1009,1006,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1010,1006,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1011,1006,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1012,1006,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1013,1006,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1014,1006,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1015,1006,'Landing of the 33 Patriots Day',0,'4/17/Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1016,1006,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1017,1006,'Battle of Las Piedras Day',0,'5/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1018,1006,'Jose Artigas'' Birthday',0,'6/19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1019,1006,'Constitution Day',0,'7/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1020,1006,'Independence Day',0,'8/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1021,1006,'Dia de la Raza / Columbus Day',0,'10/10/Sat+2,Sun+1,Tue+6,Wed+5,Thu+4,Fri+3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1022,1006,'All Souls'' Day',0,'11/2/Tue-1,Wed+5,Thu+4,Fri+3,Sat+2,Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1023,1006,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1024,18,'Seychelles',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1025,1024,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1026,1024,'Second Day of New Year',0,'1/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1027,1024,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1028,1024,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1029,1024,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1030,1024,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1031,1024,'Liberation Day',0,'6/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1032,1024,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1033,1024,'National Day',0,'6/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1034,1024,'Independence Day',0,'6/29','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1035,1024,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1036,1024,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1037,1024,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1038,1024,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1039,18,'Puerto Rico',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1040,1039,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1041,1039,'Epiphany',0,'1/6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1042,1039,'De Hostos'' Birthday',0,'1/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1043,1039,'Martin Luther King''s Day',0,'1/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1044,1039,'Presidents''Day',0,'2/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1045,1039,'Emancipation Day',0,'3/22','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1046,1039,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1047,1039,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1048,1039,'De Diego''s Birthday',0,'4/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1049,1039,'Memorial Day',0,'5/Mon/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1050,1039,'U.S. Independence Day',0,'7/4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1051,1039,'Munoz Rivera''s Birthday',0,'7/Mon/Third','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1052,1039,'Constitution Day',0,'7/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1053,1039,'Barbosa''s Birthday',0,'7/28','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1054,1039,'Labour Day',0,'9/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1055,1039,'Columbus Day',0,'10/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1056,1039,'Veterans'' Day',0,'11/11','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1057,1039,'Discovery Day',0,'11/20','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1058,1039,'Thanksgiving Day',0,'11/Thu/Fourth','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1059,1039,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1060,18,'Papua New Guinea',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1061,1060,'New Year''s Day',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1062,1060,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1063,1060,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1064,1060,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1065,1060,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1066,1060,'Queen''s Birthday',0,'6/Mon/Second','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1067,1060,'Remembrance Day',0,'7/23/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1068,1060,'Independence Day',0,'9/16/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1069,1060,'Christmas Day',0,'12/25/Sun+2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1070,1060,'Boxing Day',0,'12/26/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1071,18,'Panama',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1072,1071,'New Year''s Day',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1073,1071,'Martyrs'' Day',0,'1/9/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1074,1071,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1075,1071,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1076,1071,'Ash Wednesday',0,'Easter-46','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1077,1071,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1078,1071,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1079,1071,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1080,1071,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1081,1071,'Labour Day',0,'5/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1082,1071,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1083,1071,'Independence from Columbia Day',0,'11/3/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1084,1071,'Flag Day',0,'11/4/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1085,1071,'Colon Independence Day',0,'11/5/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1086,1071,'Cry of Independence Day',0,'11/10/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1087,1071,'Independence from Spain Day',0,'11/28/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1088,1071,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1089,1071,'Christmas Day',0,'12/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1090,1071,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1091,18,'Nicaragua',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1092,1091,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1093,1091,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1094,1091,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1095,1091,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1096,1091,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1097,1091,'Labour Day',0,'5/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1098,1091,'Revolution Day',0,'7/19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1099,1091,'Santo Domingo Day',0,'8/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1100,1091,'San Jacinto Battle Day',0,'9/14','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1101,1091,'Independence Day',0,'9/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1102,1091,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1103,1091,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1104,1091,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1105,18,'Monaco',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1106,1105,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1107,1105,'Saint Devote''s Day',0,'1/27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1108,1105,'Mardi Gras',0,'Easter-40','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1109,1105,'Mi-careme',0,'Easter-19','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1110,1105,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1111,1105,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1112,1105,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1113,1105,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1114,1105,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1115,1105,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1116,1105,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1117,1105,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1118,1105,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1119,1105,'Immaculate Conception Day',0,'12/8','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1120,1105,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1121,1105,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1122,1105,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1123,1105,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1124,1105,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1125,18,'North Korea',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1126,1125,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1127,1125,'Kim Jong II''s Anniversary',0,'2/16-17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1128,1125,'Kim II Sung''s Birthday',0,'4/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1129,1125,'Army Day',0,'4/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1130,1125,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1131,1125,'Victory Day',0,'7/27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1132,1125,'Freedom Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1133,1125,'Communist Party Foundation',0,'9/9','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1134,1125,'Workers'' Party Foundation',0,'10/10','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1135,1125,'Constitution Day',0,'12/27','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1136,18,'Jamaica',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1137,1136,'New Year''s Day',0,'1/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1138,1136,'Ash Wednesday',0,'Easter-46','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1139,1136,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1140,1136,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1141,1136,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1142,1136,'Labour Day',0,'5/23/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1143,1136,'Emancipation Day',0,'8/1/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1144,1136,'Independence Day',0,'8/6/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1145,1136,'National Heroes''Day',0,'10/16','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1146,1136,'Christmas Day',0,'12/25/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1147,1136,'Boxing Day',0,'12/26/Sun+1,Mon+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1148,18,'Iceland',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1149,1148,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1150,1148,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1151,1148,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1152,1148,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1153,1148,'Easter Monday',0,'Easter+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1154,1148,'First Day of Summer',0,'4/Thu/-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1155,1148,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1156,1148,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1157,1148,'Whit Sunday',0,'Easter+49','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1158,1148,'Whit Monday',0,'Easter+50','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1159,1148,'National Day',0,'6/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1160,1148,'Commerce Day',0,'8/Mon/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1161,1148,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1162,1148,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1163,1148,'Boxing Day',0,'12/26','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1164,1148,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1165,18,'Honduras',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1166,1165,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1167,1165,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1168,1165,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1169,1165,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1170,1165,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1171,1165,'America''s Day',0,'4/14','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1172,1165,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1173,1165,'Independence Day',0,'9/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1174,1165,'Francisco Morozan''s Birthday',0,'10/3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1175,1165,'Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1176,1165,'Armed Forces Day',0,'10/21','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1177,1165,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1178,18,'Haiti',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1179,1178,'Independence Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1180,1178,'Ancestors''Day',0,'1/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1181,1178,'Carnival (first day)',0,'Easter-48','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1182,1178,'Carnival (second day)',0,'Easter-47','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1183,1178,'Ash Wednesday',0,'Easter-46','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1184,1178,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1185,1178,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1186,1178,'Pan American Day',0,'4/14','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1187,1178,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1188,1178,'Ascension Day',0,'Easter+39','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1189,1178,'Flag Day / University Day',0,'5/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1190,1178,'Corpus Christi Day',0,'Easter+60','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1191,1178,'Assumption Day',0,'8/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1192,1178,'Jean-Jacques Dessalines'' Death',0,'10/17','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1193,1178,'United Nations'' Day',0,'10/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1194,1178,'All Saints'' Day',0,'11/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1195,1178,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1196,1178,'Vertieres Battle Day',0,'11/18','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1197,1178,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1198,18,'El Salvador',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1199,1198,'New Year''s Day',0,'1/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1200,1198,'Holy Wednesday',0,'Easter-4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1201,1198,'Maundy Thursday',0,'Easter-3','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1202,1198,'Good Friday',0,'Easter-2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1203,1198,'Holy Saturday',0,'Easter-1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1204,1198,'Easter',0,'Easter','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1205,1198,'Labour Day',0,'5/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1206,1198,'Bank Holiday',0,'6/30/Sun+1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1207,1198,'Transfiguration',0,'8/4','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1208,1198,'San Salvador Days',0,'8/5-6','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1209,1198,'Independence Day',0,'9/15','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1210,1198,'Columbus Day',0,'10/12','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1211,1198,'All Souls'' Day',0,'11/2','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1212,1198,'Cry of Independence Day',0,'11/5','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1213,1198,'Christmas Eve',0,'12/24','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1214,1198,'Christmas Day',0,'12/25','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1215,1198,'New Year''s Eve',0,'12/31','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1216,18,'Egypt',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1217,1216,'Daylight Saving Time (start)',0,'4/Fri/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1218,1216,'Daylight Saving Time (end)',0,'9/Thu/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1219,18,'Namibia',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1220,1219,'Daylight Saving Time (start)',0,'9/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1221,1219,'Daylight Saving Time (end)',0,'4/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1222,18,'Iraq',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1223,1222,'Daylight Saving Time (start)',0,'4/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1224,1222,'Daylight Saving Time (end)',0,'10/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1225,18,'Lebanon',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1226,1225,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1227,1225,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1228,18,'Kirgizstan',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1229,1228,'Daylight Saving Time (start)',0,'3/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1230,1228,'Daylight Saving Time (end)',0,'10/Sun/Last','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1231,18,'Syria',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1232,1231,'Daylight Saving Time (start)',0,'4/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1233,1231,'Daylight Saving Time (end)',0,'10/1','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1234,18,'Tonga',1,'','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1235,1234,'Daylight Saving Time (start)',0,'11/Sun/First','');
INSERT INTO calendars (id,parent,name,entry_type,formula,description) VALUES (1236,1234,'Daylight Saving Time (end)',0,'1/Sun/Last','');
INSERT INTO job_main VALUES (1,0,'Root',0,1,'Root Jobset',100,100,4,0);
INSERT INTO job_start_limit VALUES (1, 1440);
INSERT INTO job_max_duration VALUES (1, 0);
INSERT INTO job_retries VALUES (1, 0);
INSERT INTO job_retries_interval VALUES (1, 5);
INSERT INTO job_username VALUES (1,'nobody');
INSERT INTO job_file_out VALUES (1,'/dev/null');
INSERT INTO job_file_err VALUES (1,'/dev/null');
INSERT INTO job_loadenv  VALUES (1,0);
INSERT INTO job_success_return_code VALUES (1,0);
INSERT INTO job_arguments VALUES (1,-1,'');
INSERT INTO environments VALUES (1,'Default','Default environment');
INSERT INTO job_environment VALUES (1,1,-1);
INSERT INTO hosts (id,hostname,description,sslenable,sslcert) VALUES (1,'127.0.0.1','Local agent',0,'');
INSERT INTO job_host VALUES (1,1);
INSERT INTO jobset_background VALUES (1,'',2,'#678cc2','#d7e1ef',800,600);


-- 
-- Insert the icons (binary SQL scripts)
-- 

INSERT INTO schedwi_version VALUES ('0.1.3',now());

-- End Of File --
