/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* schedwiping.c -- Test the connection with agents (schedwiclnt) */

#include <schedwi.h>

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif

#if HAVE_SIGNAL_H
#include <signal.h>
#endif

#if HAVE_GETOPT_H
#include <getopt.h>
#endif

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <utils.h>
#include <conf.h>
#include <conf_srv.h>
#include <module.h>
#include <net_utils.h>
#include <net_utils_sock.h>
#include <net_utils_ssl.h>
#include <sql_hosts.h>
#include <lib_functions.h>

static const char *configuration_file = NULL;


/*
 * Print a help message to stderr
 */
void
help (const char * prog_name)
{
	int i;
	char * msg[] = {
N_("Test the connection to a Schedwi agent (schedwiclnt)."),
N_("This program must be run on the same host as the Schedwi server."),
"",
#if HAVE_GETOPT_LONG
N_("  -c, --config=FILE  use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -h, --help         display this help and exit"),
N_("  -V, --version      print version information and exit"),
#else /* HAVE_GETOPT_LONG */
N_("  -c FILE            use the configuration file FILE rather than"),
N_("                     the default one"),
N_("  -h                 display this help and exit"),
N_("  -V                 print version information and exit"),
#endif /* ! HAVE_GETOPT_LONG */
"",
"~",
N_("Exit status is 1 on error (0 means no error)."),
NULL
	};

#if HAVE_ASSERT_H
	assert (prog_name != NULL && configuration_file != NULL);
#endif

	fprintf (stderr, _("Usage: %s [OPTION]... HOST[:PORT]\n"), prog_name);

	for (i = 0; msg[i] != NULL; i++) {
		if (msg[i][0] == '~') {
			fputs (	_("The default configuration file is "),
				stderr);
			fputs (configuration_file, stderr);
		}
		else {
			fputs ((msg[i][0] == '\0')? msg[i]: _(msg[i]), stderr);
		}
		fputc ('\n', stderr);
	}

	fputc ('\n', stderr);
	fputs (_("Report bugs to "), stderr);
	fputs (PACKAGE_BUGREPORT, stderr);
	fputc ('\n', stderr);
}


/*
 * Error callback function for the sql_host_get_main_row_names() function
 */
static void
sql_error_logger (void *data, const char *msg, int err_code)
{
	if (msg != NULL) {
		fputs (msg, stderr);
		fputc ('\n', stderr);
	}
	else {
		fputs (
		_("Database error while retrieving the agent host details\n"),
			stderr);
	}
}


/*
 * Main function
 *
 * The exit code is 1 in case of error or 0 if killed (SIGTERM)
 */
int
main (int argc, char **argv)
{
	const char *server_key, *server_crt, *ca_crt, *ca_crl;
	const char *prog_name;
	char quick_random;
	char *agent_name, *result_msg, *port;
	char **names;
	row_item_t *host_details;
	int ret, ret_read;
	dbi_conn sql_handler;
	char *err_msg;
	schedwi_BIO *b;
	char *s = "[ \"version\" ]";

#if HAVE_GETOPT_LONG
	int option_index;
	struct option long_options[] =
	{
		{"help",       0, 0, 'h'},
		{"version",    0, 0, 'V'},
		{"config",     1, 0, 'c'},
		{0, 0, 0, 0}
	};
#endif

#if HAVE_SETLOCALE
	setlocale (LC_ALL, "");
#endif
#if HAVE_BINDTEXTDOMAIN
	bindtextdomain (PACKAGE, LOCALEDIR);
#endif
#if HAVE_TEXTDOMAIN
	textdomain (PACKAGE);
#endif


	/* Set default values for options */
	configuration_file = SCHEDWI_DEFAULT_CONFFILE_SRV;

	prog_name = base_name (argv[0]);

	/* Parse options */
	while (1) {
#if HAVE_GETOPT_LONG
		option_index = 0;
		ret = getopt_long (argc, argv, "hVc:",
					long_options, &option_index);
#else
		ret = getopt (argc, argv, "hVc:");
#endif

		if (ret == -1) {
			break;
		}

		switch (ret) {
			case 'h':
				help (prog_name);
				return 0;
			case 'V':
				version (prog_name);
				return 0;
			case 'c':
				configuration_file = optarg;
				break;
			default:
				help (prog_name);
				return 1;
		}
	}

	/* Agent host name missing */
	if (optind >= argc) {
		fputs (_("Agent host name required\n"), stderr);
		help (prog_name);
		return 1;
	}

	/* Too many parameters */
	if (optind + 1 != argc) {
		fputs (_("Too many parameters\n"), stderr);
		help (prog_name);
		return 1;
	}

	agent_name = argv[optind];

	/*
	 * Initialize the module engine
	 */
	if (module_init () != 0) {
		return 1;
	}

	/*
	 * Read the configuration file
	 */
	ret = conf_init_srv (configuration_file);
	switch (ret) {
		case -1:
			fputs (_("Memory allocation error\n"), stderr);
			break;

		case -2:
			perror (configuration_file);
			break;
	}
	if (ret != 0) {
		module_exit ();
		return 1;
	}

	ret =  conf_get_param_string ("SSLCACertificateFile", &ca_crt);
	ret += conf_get_param_string ("SSLCertificateFile", &server_crt);
	ret += conf_get_param_string ("SSLCertificateKeyFile", &server_key);
	ret += conf_get_param_string ("SSLCACRLFile", &ca_crl);
	ret += conf_get_param_bool ("SSLQuickRandom", &quick_random);
#if HAVE_ASSERT_H
	assert (ret == 0);
#endif

	/*
	 * Initialize the GnuTLS layer
	 */
	if (net_init_gnutls (quick_random) != 0) {
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}

	/*
	 * Initialize the network
	 */
	if (net_init (	server_crt, server_key, ca_crt, ca_crl,
			 quick_random) != 0)
	{
		net_destroy ();
		conf_destroy_srv ();
		module_exit ();
		return 1;
	}

	/*
	 * Database connection
	 */
	err_msg = NULL;
	sql_handler = begin_sql (&err_msg);
	if (sql_handler == NULL) {
		if (err_msg != NULL) {
			fprintf (stderr,
				_("Failed to connect to database: %s\n"),
				err_msg);
			free (err_msg);
		}
		else {
			fputs (_("Failed to connect to database\n"), stderr);
		}
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}

	/* Retrieve all the names of the agent from the DNS */
	if (build_client_names_from_name (agent_name, &names, &port) != 0) {
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}

	/* Retrieve the agent details from the database */
	if (sql_host_get_main_row_names (names, port, &host_details,
					sql_error_logger, NULL) != 0)
	{
		free (port);
		destroy_client_names (names);
		end_sql ();
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}
	free (port);
	destroy_client_names (names);
	end_sql ();

	/* Establish the connection to the agent */
	b = net_client (host_details[2].value_string,  /* TCP port */
			host_details[1].value_string,  /* Agent host name */
			(char)sql_row_item2ll (&(host_details[3])),
			host_details[4].value_string,  /* Certificate */
			(unsigned int)(host_details[4].len));
	sql_free_row (host_details);
	if (b == NULL) {
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}

	if (net_write (b, s, schedwi_strlen (s)) != 0) {
		net_close (b);
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}

	/* Read the result */
	result_msg = NULL;
	ret_read = net_read_result (b, &result_msg);
	net_close (b);

	/* System error */
	if (ret_read < 0) {
		if (result_msg != NULL) {
			free (result_msg);
		}
		conf_destroy_srv ();
		module_exit ();
		net_destroy ();
		return 1;
	}

	/* Success */
	if (result_msg != NULL) {
		printf (_("The agent v%s is running.\n"), result_msg);
		free (result_msg);
	}
	else {
		fputs (_("The agent is running.\n"), stdout);
	}

	/* Destroy the configuration file object */
	conf_destroy_srv ();

	/* Unload the modules */
	module_exit ();

	/* Clean the network layer */
	net_destroy ();

	return 0;
}

/*------------------------======= End Of File =======------------------------*/
