/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* sql_db_checks.c -- Database consistency checks */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <calendar.h>
#include <parsevar.h>
#include <lib_functions.h>
#include <sql_db_checks.h>


/*
 * Check the references IDs
 * For unknown IDs, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The value of the provided `field_source' in table `table_source'
 *    - The value of the provided `field_output' in table `table_source'
 *
 * Return:
 *     2 --> Erroneous IDs
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_id (
	const char *table_source, const char *field_source,
	const char *table_destination, const char *field_destination,
	const char *field_output,
	int (*callback)(void *,  row_item_t *, row_item_t *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT %s,%s FROM %s WHERE %s NOT IN \
				(SELECT %s FROM %s)",
			SQL_STRING_NON_ESCAPE, field_source,
			SQL_STRING_NON_ESCAPE, field_output,
			SQL_STRING_NON_ESCAPE, table_source,
			SQL_STRING_NON_ESCAPE, field_source,
			SQL_STRING_NON_ESCAPE, field_destination,
			SQL_STRING_NON_ESCAPE, table_destination,
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> field_source
		 * row[1] --> field_output
		 */
		callback (user_data, &(row[0]), &(row[1]));
		sql_free_row (row);
		ret = 2;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check for orphans jobs/jobsets (without existing parent).
 * For each orphan, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The job/jobset ID as a string
 *    - The job/jobset name
 *    - The type (0 --> jobset, 1 --> Job)
 *    - The job/jobset description
 *
 * Return:
 *     2 --> Orphans were found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_job_main_check_parent (
	int (*callback)(void *, unsigned long long int,
			const char *, char, const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT id,name,type,description \
			FROM job_main \
			WHERE parent <> 0 \
			AND parent NOT IN (SELECT id FROM job_main)",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Job ID
		 * row[1] --> Name
		 * row[2] --> Type (0 -> jobset, 1 --> job)
		 * row[3] --> Description
		 */
		callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
			row[1].value_string,
			(char)sql_row_item2ll (&(row[2])),
			row[3].value_string);
		sql_free_row (row);
		ret = 2;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check for jobs/jobsets with a missing calendar or a calendar folder.
 * For such jobs/jobsets, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The job/jobset ID as a string
 *    - The job/jobset name
 *    - The type (0 --> jobset, 1 --> Job)
 *    - The job/jobset description
 *
 * Return:
 *     2 --> Missing calendars were found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_job_main_check_calendar (
	int (*callback)(void *, unsigned long long int,
			const char *, char, const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT id,name,type,description \
			FROM job_main \
			WHERE cal_id <> 0 AND cal_id NOT IN \
				(SELECT id FROM calendars WHERE entry_type=0)",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Job ID
		 * row[1] --> Name
		 * row[2] --> Type (0 -> jobset, 1 --> job)
		 * row[3] --> Description
		 */
		callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
			row[1].value_string,
			(char)sql_row_item2ll (&(row[2])),
			row[3].value_string);
		sql_free_row (row);
		ret = 2;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check for jobs/jobsets with a wrong start time
 * For such jobs/jobsets, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The job/jobset ID as a string
 *    - The job/jobset name
 *    - The type (0 --> jobset, 1 --> Job)
 *    - The job/jobset description
 *    - The erroneous start_time
 *
 * Return:
 *     2 --> Erroneous start time were found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_job_main_check_time (
	int (*callback)(void *, unsigned long long int, const char *, char,
			const char *, int),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT id,name,type,description,start_time \
			FROM job_main \
			WHERE start_time < -1 OR start_time > 2359",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Job ID
		 * row[1] --> Name
		 * row[2] --> Type (0 -> jobset, 1 --> job)
		 * row[3] --> Description
		 * row[4] --> Start time
		 */
		callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
			row[1].value_string,
			(char)sql_row_item2ll (&(row[2])),
			row[3].value_string,
			(int)sql_row_item2ll (&(row[4])));
		sql_free_row (row);
		ret = 2;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check for missing SSL certificate associated with hosts.
 * For such hosts, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The host ID as a string
 *    - The host name
 *    - The TCP port number as a string
 *    - The host description
 *
 * Return:
 *     2 --> Missing SSL certificates were found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_hosts_check_cert (
	int (*callback)(void *, unsigned long long int, const char *,
			const char *, const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT id,hostname,portnum,description \
			FROM hosts \
			WHERE sslenable = 1 \
				AND (sslcert IS NULL OR LENGTH(sslcert) = 0)",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Host ID
		 * row[1] --> Host name
		 * row[2] --> TCP port number
		 * row[3] --> Description
		 */
		callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
			row[1].value_string,
			row[2].value_string,
			row[3].value_string);
		sql_free_row (row);
		ret = 2;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check the calendar formulas
 * On formula error, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The calendar ID as a string
 *    - The calendar name
 *    - The calendar description
 *    - The calendar formula
 *
 * Return:
 *     2 --> Erroneous formulas were found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_calendars_check_formula (
	int (*callback)(void *, unsigned long long int, const char *,
			const char *, const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;
	cal_t cal;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT id,name,description,formula \
			FROM calendars WHERE entry_type = 0",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Calendar ID
		 * row[1] --> Name
		 * row[2] --> Description
		 * row[3] --> Formula
		 */
		if (str2cal (	&cal, row[3].value_string,
				2007, NULL) != CAL_NOERROR)
		{
			callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
				row[1].value_string,
				row[2].value_string,
				row[3].value_string);
			ret = 2;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check for orphans calendars (without existing parent).
 * For each orphan, the callback() function is called with the following
 * parameters:
 *    - The provided user_data
 *    - The calendar ID as a string
 *    - The calendar name
 *    - The type (0 --> calendar, 1 --> folder)
 *    - The calendar description
 *
 * Return:
 *     2 --> Orphans were found
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_calendars_check_parent (
	int (*callback)(void *, unsigned long long int, const char *, char,
			const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT id,name,entry_type,description \
			FROM calendars \
			WHERE parent <> 0 \
			AND parent NOT IN (SELECT id FROM calendars)",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Calendar ID
		 * row[1] --> Name
		 * row[2] --> Type (0 -> calendar, 1 --> folder)
		 * row[3] --> Description
		 */
		callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
				row[1].value_string,
				(char)sql_row_item2ll (&(row[2])),
				row[3].value_string);
		sql_free_row (row);
		ret = 2;
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	return ret;
}


/*
 * Check for invalid environment variable values.
 * For each erroneous variable, the callback() function is called with the
 * following parameters:
 *    - The provided user_data
 *    - The environment variable ID as a string
 *    - The environment variable name
 *    - The environment variable value
 *
 * Return:
 *     2 --> Errors were found.
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_environment_var_check_value (
	int (*callback)(void *, unsigned long long int, const char *,
			const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	int ret;
	char *out;
	size_t out_len;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (NULL, NULL, &err_msg, NULL, &rows,
			"SELECT env_id,env_key,env_value \
			FROM environment_var",
			SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	out = NULL;
	out_len = 0;
	ret = 0;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Environment variable ID
		 * row[1] --> Name
		 * row[2] --> Value
		 */
		if (parse_string (	row[2].value_string,
					row[2].len,
					NULL, &out, &out_len) != 0)
		{
			callback (user_data,
			(unsigned long long int) sql_row_item2ll (&(row[0])),
				row[1].value_string,
				row[2].value_string);
			ret = 2;
		}
		sql_free_row (row);
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);
	if (out != NULL) {
		free (out);
	}
	return ret;
}


/*
 * Remove the item from the provided table
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_delete1 (	const char *table_name,
			const char *field_name,
			unsigned long long int id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (table_name != NULL && field_name != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"DELETE FROM %s WHERE %s=%ld",
				SQL_STRING_NON_ESCAPE, table_name,
				SQL_STRING_NON_ESCAPE, field_name,
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the item from the provided table
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_delete2 (	const char *table_name,
			const char *field_name1,
			const char *field_name2,
			unsigned long long int id1,
			const char *id2,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (	   table_name != NULL
			&& field_name1 != NULL && field_name2 != NULL
			&& id2 != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"DELETE FROM %s WHERE %s=%ld AND %s=%s",
				SQL_STRING_NON_ESCAPE, table_name,
				SQL_STRING_NON_ESCAPE, field_name1,
				SQL_INT, (long int)id1,
				SQL_STRING_NON_ESCAPE, field_name2,
				SQL_STRING, id2,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Remove the SSL enabled flag from the provided host ID
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_disable_ssl (	unsigned long long int id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"UPDATE hosts SET sslenable=0 WHERE id=%ld",
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Fix the calendar ID (i.e. set cal_id to 0) in the job/jobset which
 * ID is provided
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_fix_cal_id (	unsigned long long int id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"UPDATE job_main SET cal_id=0 WHERE id=%ld",
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Fix the start time (i.e. set start_time to -1) in the job/jobset which
 * ID is provided
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_fix_start_time (unsigned long long int id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"UPDATE job_main SET start_time=-1 \
				WHERE id=%ld",
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Create a new jobset at the root of the jobset tree.  This new jobset will
 * contain the detected orphan jobs/jobsets
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_create_lost_found_jobset (const char *name, const char *description,
			int x, int y,
			unsigned long long int *id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (name != NULL && description != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, id, NULL, "job_main",
		"INSERT INTO job_main \
		 (parent,name,type,enabled,description,x,y,cal_id,start_time) \
		VALUES (1,%s,0,0,%s,%d,%d,0,-1)",
				SQL_STRING, name,
				SQL_STRING, description,
				SQL_INT, (long int)x,
				SQL_INT, (long int)y,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Re-attach the provided orphan job/jobset to the lost+found jobset
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_attach_orphan_job (unsigned long long int id,
			unsigned long long int parent_id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"UPDATE job_main SET parent=%ld \
				WHERE id=%ld",
				SQL_INT, (long int)parent_id,
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Create a new calendar folder at the root of the tree.  This new folder will
 * contain the detected orphan calendars
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_create_lost_found_cal_folder (const char *name,
			const char *description,
			unsigned long long int *id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (name != NULL && description != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, id, NULL, "calendars",
		"INSERT INTO calendars (parent,name,entry_type,description) \
		VALUES (1,%s,1,%s)",
				SQL_STRING, name,
				SQL_STRING, description,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Re-attach the provided orphan calendar/folder to the lost+found folder
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_attach_orphan_cal (unsigned long long int id,
			unsigned long long int parent_id,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;


	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"UPDATE calendars SET parent=%ld \
				WHERE id=%ld",
				SQL_INT, (long int)parent_id,
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}


/*
 * Comment out the erroneous calendar formula
 *
 * Return:
 *     0 --> No error
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_check_comment_out_formula (unsigned long long int id,
			const char *new_formula,
			void (*error_func)(void *, const char *, int),
			void *user_data_error_func)
{
	char *err_msg = NULL;
	int ret;

#if HAVE_ASSERT_H
	assert (new_formula != NULL);
#endif

	ret = sql_non_select (	NULL, NULL, &err_msg, NULL, NULL, NULL,
				"UPDATE calendars SET formula=%s \
				WHERE id=%ld",
				SQL_STRING, new_formula,
				SQL_INT, (long int)id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
