/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * sql_get_environment.c -- Get the environment for a job searching its
 *                          ancestors (the job inherit the environments of its
 *                          ancestors).
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <sql_common.h>
#include <sql_hierarchy.h>
#include <sql_get_environment.h>


#define SQL_GET_ENVIRONMENT_REQ "SELECT job_environment_s.job_id,job_environment_s.position,environment_var_s.position,environment_var_s.env_key,environment_var_s.env_value FROM environment_var_s,job_environment_s WHERE job_environment_s.workload_date=%d AND environment_var_s.workload_date=%d AND job_environment_s.job_id IN (%s) AND job_environment_s.env_id=environment_var_s.env_id ORDER BY job_environment_s.position,environment_var_s.position"
#define SQL_GET_ENVIRONMENT_HOST_REQ "SELECT host_environment_s.position,environment_var_s.position,environment_var_s.env_key,environment_var_s.env_value FROM environment_var_s,host_environment_s WHERE host_environment_s.workload_date=%d AND environment_var_s.workload_date=%d AND host_environment_s.host_id=%ld AND host_environment_s.env_id=environment_var_s.env_id ORDER BY host_environment_s.position,environment_var_s.position"


/*
 * Get the environment from the database for a job specified by the
 * provided hierarchy list (in lst).  For each environment variable,
 * the callback() function is called with the following parameters:
 *    - The position
 *    - The variable name
 *    - The variable value
 * If the callback function return a value other than 0, the
 * sql_get_environment() exits with a return code of 2.
 *
 * Return:
 *     2 --> The callback function returned a value other than 0
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_get_environment (int workload_date, lwc_LL *lst, unsigned int next_pos,
	int (*callback)(void *, unsigned int, const char *, const char *),
	void *user_data,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	char *hierarchy_str;
	unsigned int i, j, hierarchy_len;
	int ret;
	lwc_LL *rows, **lists, *result_list;
	row_item_t *row;
	unsigned long long int jobid, *id;

#if HAVE_ASSERT_H
	assert (lwc_getNumNode (lst) > 0 && callback != NULL);
#endif


	/* Create the array of environment variable lists for each job ID */
	hierarchy_len = lwc_getNumNode (lst);
	lists = (lwc_LL **) malloc (hierarchy_len * sizeof (lwc_LL *));
	if (lists == NULL) {
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	for (i = 0; i < hierarchy_len; i++) {
		lists[i] = lwc_newLL ();
		if (lists[i] == NULL) {
			for (j = 0; j < i; j++) {
				lwc_delLL (lists[j], NULL);
			}
			free (lists);
			if (error_func != NULL) {
				error_func (	user_data_error_func,
						_("Memory allocation error"),
						0);
			}
			return -1;
		}
	}

	/* Build the job ID list to be used by the SQL request */
	if (hierarchy_list_to_str (lst, &hierarchy_str) != 0) {
		for (i = 0; i < hierarchy_len; i++) {
			lwc_delLL (lists[i], NULL);
		}
		free (lists);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	/* SQL request */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_GET_ENVIRONMENT_REQ,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)workload_date,
				SQL_STRING_NON_ESCAPE, hierarchy_str,
				SQL_END);
	free (hierarchy_str);
	if (ret != 0) {
		for (i = 0; i < hierarchy_len; i++) {
			lwc_delLL (lists[i], NULL);
		}
		free (lists);
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Job ID
		 * row[1] --> Position 1
		 * row[2] --> Position 2
		 * row[3] --> Environment variable name
		 * row[4] --> Environment variable value
		 */
		jobid = (unsigned long long int)sql_row_item2ll (&(row[0]));
		lwc_rewindLL (lst);
		for (	i = 0;
			   (id = (unsigned long long int *) lwc_nextLL (lst)) != NULL
			&& *id != jobid;
			i++);
		if (id != NULL) {
			if (lwc_addEndLL (lists[i], row) != 0) {
				for (i = 0; i < hierarchy_len; i++) {
					lwc_delLL (lists[i],
					(void (*)(const void *))sql_free_row);
				}
				free (lists);
				sql_free_row (row);
				lwc_delLL (rows,
					(void (*)(const void *))sql_free_row);
				if (error_func != NULL) {
					error_func (user_data_error_func,
						_("Memory allocation error"),
						0);
				}
				return -1;
			}
		}
		else {
			sql_free_row (row);
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	/* Create the result list */
	result_list = lwc_newLL ();
	if (result_list == NULL) {
		for (i = 0; i < hierarchy_len; i++) {
			lwc_delLL (lists[i],
					(void (*)(const void *))sql_free_row);
		}
		free (lists);
		if (error_func != NULL) {
			error_func (	user_data_error_func,
					_("Memory allocation error"), 0);
		}
		return -1;
	}

	i = hierarchy_len;
	do {
		i--;
		while ((row = (row_item_t *)lwc_delStartLL (lists[i])) != NULL)
		{
			/*
			 * row[0] --> Job ID
			 * row[1] --> Position 1
			 * row[2] --> Position 2
			 * row[3] --> Environment variable name
			 * row[4] --> Environment variable value
			 */

			/* No environment for the current job if pos1 == -1 */
			if (sql_row_item2ll (&(row[1])) < 0) {
				lwc_emptyLL (result_list,
					(void (*)(const void *))sql_free_row);
				lwc_emptyLL (lists[i],
					(void (*)(const void *))sql_free_row);
				sql_free_row (row);
				continue;
			}

			if (lwc_addEndLL (result_list, row) != 0) {
				for (i = 0; i < hierarchy_len; i++) {
					lwc_delLL (lists[i],
					(void (*)(const void *))sql_free_row);
				}
				free (lists);
				lwc_delLL (result_list,
					(void (*)(const void *))sql_free_row);
				sql_free_row (row);
				if (error_func != NULL) {
					error_func (user_data_error_func,
						_("Memory allocation error"),
						0);
				}
				return -1;
			}
		}
	} while (i > 0);

	for (i = 0; i < hierarchy_len; i++) {
		lwc_delLL (lists[i], (void (*)(const void *))sql_free_row);
	}
	free (lists);

	while ((row = (row_item_t *) lwc_delStartLL (result_list)) != NULL) {
		/*
		 * row[0] --> Job ID
		 * row[1] --> Position 1
		 * row[2] --> Position 2
		 * row[3] --> Environment variable name
		 * row[4] --> Environment variable value
		 */
		ret = callback (user_data, next_pos++,
				row[3].value_string, row[4].value_string);
		sql_free_row (row);
		if (ret != 0) {
			lwc_delLL (	result_list,
					(void (*)(const void *))sql_free_row);
			return 2;
		}
	}
	lwc_delLL (result_list, (void (*)(const void *))sql_free_row);
	return 0;
}


/*
 * Get the environment from the database for a host specified by the
 * provided host_id string.  For each environment variable,
 * the callback() function is called with the following parameters:
 *    - The position
 *    - The variable name
 *    - The variable value
 * If the callback function return a value other than 0, the
 * sql_get_environment_host() function exits with a return code of 2.
 * If not NULL, next_pos is set with the position of the last environment
 * variable.  This value may be used in the call to the sql_get_environment()
 * function.
 *
 * Return:
 *     2 --> The callback function returned a value other than 0
 *     0 --> No error.
 *    -1 --> Memory allocation error (if error_func() is not NULL, it is called
 *           with user_data_error_func as its first parameter and the error
 *           message as the second parameter)
 *    -2 --> SQL error (if error_func() is not NULL, it is called with
 *           user_data_error_func as its first parameter and the error message
 *           as the second parameter)
 */
int
sql_get_environment_host (int workload_date, unsigned long long int host_id,
	int (*callback)(void *, unsigned int, const char *, const char *),
	void *user_data,
	unsigned int *next_pos,
	void (*error_func)(void *, const char *, int),
	void *user_data_error_func)
{
	char *err_msg = NULL;
	lwc_LL *rows;
	row_item_t *row;
	unsigned int i;
	int ret;

#if HAVE_ASSERT_H
	assert (callback != NULL);
#endif

	/* SQL request */
	ret = sql_select (	NULL, NULL, &err_msg, NULL, &rows,
				SQL_GET_ENVIRONMENT_HOST_REQ,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)workload_date,
				SQL_INT, (long int)host_id,
				SQL_END);
	if (ret != 0) {
		if (error_func != NULL) {
			error_func (user_data_error_func, err_msg, ret);
		}
		if (err_msg != NULL) {
			free (err_msg);
		}
		return ret;
	}

	i = 1;
	while ((row = (row_item_t *) lwc_delStartLL (rows)) != NULL) {
		/*
		 * row[0] --> Position 1
		 * row[1] --> Position 2
		 * row[2] --> Environment variable name
		 * row[3] --> Environment variable value
		 */
		ret = callback (user_data, i++,
				row[2].value_string, row[3].value_string);
		sql_free_row (row);
		if (ret != 0) {
			lwc_delLL (rows, (void (*)(const void *))sql_free_row);
			return 2;
		}
	}
	lwc_delLL (rows, (void (*)(const void *))sql_free_row);

	if (next_pos != NULL) {
		*next_pos = i;
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
