/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * job_parameters.h -- Execution parameters for a job
 *
 * The parameters (members of the job_parameters structure) are the
 * following:
 *
 *        command --> The command to be run
 *       username --> User name to use to run the command
 *    file_stdout --> File name of the command stdout redirection
 *    file_stderr --> File name of the command stderr redirection
 *    cgroup_name --> Linux Control Group name (path from a controller mount
 *                    directory)
 *       load_env --> Do the user environement should be load (1) or
 *                    not (0).  If yes (1), the user shell is run with the
 *                    command as a parameter (ie. /bin/sh -c command)
 *      arguments --> Arguments list
 *        environ --> Environement to load (several other variables will be
 *                    added, like SHELL, PATH, LOGNAME, ...)
 */

#ifndef _JOB_PARAMETERS_H
#define _JOB_PARAMETERS_H

#include <schedwi.h>
#include <env_utils.h>
#include <args_utils.h>

/* Allow the use in C++ code.  */
#ifdef __cplusplus
extern "C" {
#endif

struct job_parameters {
	char *command;
	char *username;
	char *file_stdout;
	char *file_stderr;
	char *cgroup_name;
	int load_env;
	argument_t arguments;
	environment_t environ;
};
typedef struct job_parameters job_parameters_t;


extern void init_job_parameters OF((job_parameters_t *j));
extern void destroy_job_parameters OF((job_parameters_t *j));
extern int add_command_job_parameters OF((	job_parameters_t *j,
						const char *command,
						size_t len));
extern int add_username_job_parameters OF((	job_parameters_t *j,
						const char *username,
						size_t len));
extern int add_stdout_job_parameters OF((	job_parameters_t *j,
						const char *file,
						size_t len));
extern int add_stderr_job_parameters OF((	job_parameters_t *j,
						const char *file,
						size_t len));
extern int add_cgroup_job_parameters OF((	job_parameters_t *j,
						const char *cgroup_name,
						size_t len));
extern void add_loadenv_job_parameters OF((job_parameters_t *j, int flag));
extern int add_arg_job_parameters OF((	job_parameters_t *j, const char *arg,
					size_t len));
extern int add_env_job_parameters OF((	job_parameters_t *j,
					const char *key, const char *val));
extern int concat_env_job_parameters OF((job_parameters_t *j,
					environment_t *to_concat));

#ifdef __cplusplus
}
#endif  /* C++ */

#endif /* job_parameters.h */

/*-----------------============== End Of File ==============-----------------*/
