/* Schedwi
   Copyright (C) 2007-2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 * job_run.c -- Running parameters for a job
 *
 * The parameters (members of the job_run structure) are the
 * following:
 *
 *        command --> The command to be run
 *    file_stdout --> File name of the command stdout redirection
 *    file_stderr --> File name of the command stderr redirection
 *    cgroup_name --> Linux Control Group name (path from a controller mount
 *                    directory)
 *           work --> Working directory (a chdir(2) will be done in this
 *                    directory before running (exec) the user command)
 *          shell --> User default shell
 *            uid --> User ID
 *            gid --> User group ID
 *           mask --> umask
 *            lim --> ulimit
 *           nice --> Scheduling priority (nice) for this user
 *      arguments --> Arguments list (included argv[0], the command name)
 *        environ --> Environement to load
 */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_STRINGS_H
#include <strings.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_PWD_H
#include <pwd.h>
#endif

#if HAVE_FCNTL_H
#include <fcntl.h>
#endif

#if HAVE_TIME_H
#include <time.h>
#endif
#if TM_IN_SYS_TIME
#include <sys/time.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <parsevar.h>
#include <job_run.h>


/*
 * Initialize a job_run structure
 */
void
init_job_run (job_run_t *j)
{
	if (j != NULL) {
		j->command = NULL;
		j->file_stderr = NULL;
		j->file_stdout = NULL;
		j->cgroup_name = NULL;
		j->work = NULL;
		j->shell = NULL;
		j->uid = SCHEDWI_UID_NOBODY;
		j->gid = SCHEDWI_GID_NOGROUP;
		j->mask = SCHEDWI_DEFAULT_UMASK;
		j->lim = SCHEDWI_DEFAULT_ULIMIT;
		j->nice = SCHEDWI_DEFAULT_NICE;
		init_argument (&(j->arguments));
		init_environment (&(j->environ));
	}
}


/*
 * Free the content of a job_paramters structure
 */
void
destroy_job_run (job_run_t *j)
{
	if (j != NULL) {
		if (j->command != NULL) {
			free (j->command);
		}
		if (j->file_stderr != NULL) {
			free (j->file_stderr);
		}
		if (j->file_stdout != NULL) {
			free (j->file_stdout);
		}
		if (j->cgroup_name != NULL) {
			free (j->cgroup_name);
		}
		if (j->work != NULL) {
			free (j->work);
		}
		if (j->shell != NULL) {
			free (j->shell);
		}
		destroy_argument (&(j->arguments));
		destroy_environment (&(j->environ));
		init_job_run (j);
	}
}


/*
 * Return the working directory.  The returned buffer must be freed by the
 * caller.
 * If `home_dir' exists, it will be the working directory.  If not,
 * the directory specified by the TMPDIR environment variable will be used.
 * If TMPDIR is not set, /tmp will be used.
 *
 * Return:
 *    The working directory (string to be freed by the caller) or
 *    NULL on memory allocation error
 */
static char *
get_workdir (const char *home_dir)
{
	char *work, *tmp;
	struct stat stat_buff;

	/* Check that home_dir exists and is a directory */
	if (	   home_dir == NULL
		|| stat (home_dir, &stat_buff) != 0
#ifndef STAT_MACROS_BROKEN
		|| !S_ISDIR (stat_buff.st_mode)
#endif
		)
	{
#if HAVE_GETENV
		tmp = getenv ("TMPDIR");
#else
		tmp = NULL;
#endif
		if (tmp == NULL) {
			work = (char *)malloc (
				schedwi_strlen (SCHEDWI_DEFAULT_TMP) + 1);
			if (work == NULL) {
				return NULL;
			}
			strcpy (work, SCHEDWI_DEFAULT_TMP);
			return work;
		}
		work = (char *)malloc (schedwi_strlen (tmp) + 1);
		if (work == NULL) {
			return NULL;
		}
		strcpy (work, tmp);
		return work;
	}
	work = (char *) malloc (schedwi_strlen (home_dir) + 1);
	if (work == NULL) {
		return  NULL;
	}
	strcpy (work, home_dir);
	return work;
}


/*
 * Parse the /etc/login.defs (or /etc/default/login) to get the default
 * umask, ulimit and PATH parameters.  The QUOTAS_ENAB keyword is also
 * checked to define if the password GECOS must be checked for ulimit, umask
 * and nice.
 * `mask', `ulim' and `path' are set (`path' MUST NOT be freed by the caller)
 * using the values from the login.defs file.  If not defined in this file,
 * `mask' and `ulim' are set to default values (077 for mask and 0 (unlimited)
 * for ulim) and `path' to NULL.
 * If the PATH is not defined in thi file, the PATH environment variable
 * will not be set.
 *
 * Return:
 *    0 --> No error.  The GECOS in /etc/passwd must be parsed for ulimit,
 *          umask and nice
 *    1 --> No error.  The GECOS must be ignored
 *   -1 --> Memory allocation error (mask, ulim and path are not modified)
 */
static int
parse_login_defaults (mode_t *mask, rlim_t *ulim, char **path)
{
	static time_t mtime = 0;
	static mode_t msk = SCHEDWI_DEFAULT_UMASK;
	static rlim_t lim = SCHEDWI_DEFAULT_ULIMIT;
	static int parse_gecos_ret = 0;
	static char *default_path = NULL;

	struct stat stat_buff;
	int fd, i;
	char *buff, *ln, *sep;
	char *file_name;

	/* File not found */
	file_name = "/etc/login.defs";
	if (stat (file_name, &stat_buff) != 0) {
		file_name = "/etc/default/login";
		if (stat (file_name, &stat_buff) != 0) {
			mtime = 0;
			msk = SCHEDWI_DEFAULT_UMASK;
			lim = SCHEDWI_DEFAULT_ULIMIT;
			parse_gecos_ret = 0;
			if (default_path != NULL) {
				free (default_path);
				default_path = NULL;
			}

			if (mask != NULL) {
				*mask = msk;
			}
			if (ulim != NULL) {
				*ulim = lim;
			}
			if (path != NULL) {
				*path = default_path;
			}
			return parse_gecos_ret;
		}
	}

	/* The file hasn't been modified since the last read */
	if (stat_buff.st_mtime == mtime) {
		if (mask != NULL) {
			*mask = msk;
		}
		if (ulim != NULL) {
			*ulim = lim;
		}
		if (path != NULL) {
			*path = default_path;
		}
		return parse_gecos_ret;
	}

	buff = (char *)malloc (stat_buff.st_size);
	if (buff == NULL) {
		return -1;
	}

	/* Initialization */
	msk = SCHEDWI_DEFAULT_UMASK;
	lim = SCHEDWI_DEFAULT_ULIMIT;
	parse_gecos_ret = 0;
	if (default_path != NULL) {
		free (default_path);
		default_path = NULL;
	}

	/* Fail to open the file or to read it */
	fd = open (file_name, O_RDONLY);
	if (fd < 0 || read (fd, buff, stat_buff.st_size) <= 0) {
		if (fd >= 0) {
			close (fd);
		}
		free (buff);
		if (mask != NULL) {
			*mask = msk;
		}
		if (ulim != NULL) {
			*ulim = lim;
		}
		if (path != NULL) {
			*path = default_path;
		}
		return parse_gecos_ret;
	}
	close (fd);

	ln = buff;
	/* Parse every line */
	do {
		sep = strchr (ln, '\n');
		if (sep != NULL) {
			*sep = '\0';
		}

		for (i = 0; isspace (ln[i]); i++);

		/* umask */
		if (	   schedwi_strncasecmp (ln + i, "UMASK", 5) == 0
			&& isspace (ln[i + 5]))
		{
			msk = strtol (ln + i + 5, NULL, 0);
		}

		/* ulimit */
		else {
		if (	   schedwi_strncasecmp (ln + i, "ULIMIT", 6) == 0
			&& isspace (ln[i + 6]))
		{
			lim = strtol (ln + i + 6, NULL, 0);
		}

		/* Do we have to check the GECOS in /etc/passwd? */
		else {
		if (	   schedwi_strncasecmp (ln + i, "QUOTAS_ENAB", 11) == 0
			&& isspace (ln[i + 11]))
		{
			i += 11;
			while (isspace (ln[i])) {
				i++;
			}
			if (schedwi_strncasecmp (ln + i, "no", 2) == 0) {
				parse_gecos_ret = 1;
			}
			else {
				parse_gecos_ret = 0;
			}
		}

		/* PATH environment variable */
		else {
		if (	   schedwi_strncasecmp (ln + i, "ENV_PATH", 8) == 0
			&& isspace (ln[i + 8]))
		{
			i += 8;
			while (isspace (ln[i])) {
				i++;
			}
			if (schedwi_strncasecmp (ln + i, "PATH=", 5) == 0) {
				i += 5;
				while (isspace (ln[i])) {
					i++;
				}
			}

			if (default_path != NULL) {
				free (default_path);
			}
			default_path = (char *)malloc (
						schedwi_strlen (ln + i) + 1);
			if (default_path == NULL) {
				free (buff);
				return -1;
			}
			strcpy (default_path, ln + i);
		}

		/* PATH environment variable */
		else {
		if (	   schedwi_strncasecmp (ln + i, "PATH", 4) == 0
			&& isspace (ln[i + 4]))
		{
			i += 4;
			while (isspace (ln[i])) {
				i++;
			}
			if (schedwi_strncasecmp (ln + i, "PATH=", 5) == 0) {
				i += 5;
				while (isspace (ln[i])) {
					i++;
				}
			}

			if (default_path != NULL) {
				free (default_path);
			}
			default_path = (char *)malloc (
						schedwi_strlen (ln + i) + 1);
			if (default_path == NULL) {
				free (buff);
				return -1;
			}
			strcpy (default_path, ln + i);
		}}}}}

		if (sep != NULL) {
			ln = sep + 1;
		}
	} while (sep != NULL);

	free (buff);

	if (mask != NULL) {
		*mask = msk;
	}
	if (ulim != NULL) {
		*ulim = lim;
	}
	if (path != NULL) {
		*path = default_path;
	}

	mtime = stat_buff.st_mtime;

	return parse_gecos_ret;
}


/*
 * Given a /etc/password GECOS field, and parsing the /etc/login.defs (or
 * /etc/default/login), fill the following members of the job_run structure:
 *
 *          mask --> umask
 *           lim --> ulimit
 *          nice --> scheduling priority (nice) retrieved only from the
 *                   gecos field
 *       environ --> the initial PATH environment variable is set
 *
 * First, the umask, ulimit and PATH are retrieved from the /etc/login.defs
 * (or /etc/default/login).  Then, if the QUOTAS_ENAB parameter in this
 * file is not `no', umask, ulimit and nice are retrieved from the GECOS
 * field of the /etc/password file (see password(5)).
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error (j is unchanged)
 */
static int
parse_gecos (const char *gecos, job_run_t *j)
{
	mode_t umsk;
	rlim_t ulim;
	char *path;
	int ret, pri;
	char *ptr, *tok;

#if HAVE_ASSERT_H
	assert (j != NULL);
#endif

	/* Get the values from /etc/login.defs or /etc/default/login */
	ret = parse_login_defaults (&umsk, &ulim, &path);
	if (ret == -1) {
		return -1;
	}
	/* Do not parse the GECOS field (QUOTAS_ENAB = no) */
	if (ret != 0 || gecos == NULL) {
		/*
		 * Set the PATH environment variable (if defined in
		 * /etc/login.defs or /etc/default/login)
		 */
		if (	   path != NULL
			&& add_env (&(j->environ), "PATH", path) != 0)
		{
			return -1;
		}
		j->mask = umsk;
		j->lim  = ulim;
		j->nice = SCHEDWI_DEFAULT_NICE;
		return 0;
	}

	/* Parse the GECOS field */
	ptr = (char *) malloc (schedwi_strlen (gecos) + 1);
	if (ptr == NULL) {
		return -1;
	}
	strcpy (ptr, gecos);

	/*
	 * Set the PATH environment variable (if defined in
	 * /etc/login.defs or /etc/default/login)
	 */
	if (	   path != NULL
		&& add_env (&(j->environ), "PATH", path) != 0)
	{
		free (ptr);
		return -1;
	}
	j->mask = umsk;
	j->lim  = ulim;
	j->nice = SCHEDWI_DEFAULT_NICE;

	/* Parse the GECOS field */
	tok = strtok (ptr, ",");
	while (tok != NULL) {
		while (isspace (*tok)) {
			tok++;
		}
		if (schedwi_strncasecmp (tok, "pri=", 4) == 0) {
			pri = atoi (tok + 4);
			if (pri >= -20 && pri <= 19) {
				j->nice = pri;
			}
		}
		else {
		if (schedwi_strncasecmp (tok, "umask=", 6) == 0) {
			j->mask = strtol (tok + 6, NULL, 0);
		}
		else {
		if (schedwi_strncasecmp (tok, "ulimit=", 7) == 0) {
			j->lim = strtol (tok + 7, NULL, 0);
		}}}
		tok = strtok (NULL, ",");
	}

	free (ptr);
	return 0;
}


/*
 * Given a user name, fill the following members of the job_run structure:
 *
 *          work --> Working directory (a chdir(2) will be done in this
 *                   directory before running (exec) the user command)
 *         shell --> User default shell
 *           uid --> User ID
 *           gid --> User group ID
 *          mask --> umask for this user
 *           lim --> ulimit for this user
 *          nice --> scheduling priority (nice) for this user
 *       environ --> Initial environment for this user (SHELL, PATH, LOGNAME,
 *                   USER, HOME)
 *
 * Return:
 *    0 --> No error
 *    1 --> Unknow user (the job_run_t structure is unmodified)
 *   -1 --> Memory allocation error
 */
int
add_username_job_run (job_run_t *j, const char *username)
{
	struct passwd *p;
	char *shell, *workdir;
	environment_t env;

	if (j == NULL) {
		return 0;
	}

	if (username == NULL || username[0] == '\0') {
		return 1;
	}

	setpwent ();
	while ((p = getpwent ()) != NULL) {
		if (strcmp (username, p->pw_name) == 0) {

			init_environment (&env);

			/* User name (LOGNAME and USER) */
			if (	   add_env (&env, "LOGNAME", username) != 0
				|| add_env (&env, "USER", username) != 0)
			{
				destroy_environment (&env);
				endpwent ();
				return -1;
			}

			/* HOME directory */
			if (p->pw_dir != NULL) {
				if (add_env (&env, "HOME", p->pw_dir) != 0) {
					destroy_environment (&env);
					endpwent ();
					return -1;
				}
			}

			/*
			 * Working directory (a chdir(2) will be done in this
			 * directory before running (exec) the user command)
			 */
			workdir = get_workdir (p->pw_dir);
			if (workdir == NULL) {
				destroy_environment (&env);
				endpwent ();
				return -1;
			}

			/* Shell */
			if (p->pw_shell == NULL || (p->pw_shell)[0] == '\0') {
				shell = (char *)malloc (
					schedwi_strlen (SCHEDWI_DEFAULT_SHELL)
					+ 1);
				if (shell == NULL) {
					free (workdir);
					destroy_environment (&env);
					endpwent ();
					return -1;
				}
				strcpy (shell, SCHEDWI_DEFAULT_SHELL);
			}
			else {
				shell = (char *)malloc (
					schedwi_strlen (p->pw_shell) + 1);
				if (shell == NULL) {
					free (workdir);
					destroy_environment (&env);
					endpwent ();
					return -1;
				}
				strcpy (shell, p->pw_shell);
			}
			if (add_env (&env, "SHELL", shell) != 0) {
				free (shell);
				free (workdir);
				destroy_environment (&env);
				endpwent ();
				return -1;
			}

			/* ulimit, umask, nice and PATH */
			if (parse_gecos (p->pw_gecos, j) != 0) {
				free (shell);
				free (workdir);
				destroy_environment (&env);
				endpwent ();
				return -1;
			}

			if (concat_env (&(j->environ), &env) != 0) {
				free (shell);
				free (workdir);
				destroy_environment (&env);
				endpwent ();
				return -1;
			}

			j->work = workdir;
			j->shell = shell;
			j->uid = p->pw_uid;
			j->gid = p->pw_gid;

			endpwent ();
			return 0;
		}
	}
	endpwent ();
	return 1;
}


/*
 * Set the file_stdout and file_stderr members.  Parameter expansion is done
 * in those strings as well as date/time expansion (see strftime(3))
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 *   -2 --> Syntax error during parameter expansion in the file stdout names
 *   -3 --> Syntax error during parameter expansion in the file stderr names
 */
int
add_outfile_job_run (	job_run_t *j, const char *file_stdout,
			const char *file_stderr)
{
	char *out, *out1, *err1;
	size_t out_len, max, time_len;
	int ret;
#if HAVE_LOCALTIME && HAVE_TIME
	time_t curr;
	struct tm *curr_tm;
#endif

	if (j == NULL) {
		return 0;
	}

#if HAVE_LOCALTIME && HAVE_TIME
	curr = time (NULL);
	curr_tm = localtime (&curr);
#endif

	out = NULL;
	out_len = 0;
	out1 = NULL;
	err1 = NULL;


	/* Parse the stdout file name */
	ret = parse_string (	file_stdout, schedwi_strlen (file_stdout),
				&(j->environ),
				&out, &out_len);
	if (ret != 0) {
		return ret;
	}

	/* Convert the time/date format specification in the stdout file */
	max = out_len;
	time_len = 0;
	/*
	 * Return the result of the format in out1.  If out1 is not big
	 * enough, strftime will return 0 (or max).  In this case we will
	 * try again with a bigger (+100 characters) buffer
	 */
	do {
		if (out1 != NULL) {
			free (out1);
		}
		out1 = (char *)malloc (max + 100);
		if (out1 == NULL) {
			if (out != NULL && out_len > 0) {
				free (out);
			}
			return -1;
		}
		max += 100;
#if HAVE_STRFTIME && HAVE_LOCALTIME && HAVE_TIME
		time_len = strftime (out1, max, out, curr_tm);
#else
		strncpy (out1, out, max);
		time_len = max - 1;
		out1[time_len] = '\0';
#endif
	} while ((time_len == 0 && out[0] != '\0') || time_len == max);


	/* Parse the stderr file name */	
	ret = parse_string (	file_stderr, schedwi_strlen (file_stderr),
				&(j->environ),
				&out, &out_len);
	if (ret != 0) {
		if (out != NULL && out_len > 0) {
			free (out);
		}
		free (out1);
		if (ret == -2) {
			return -3;
		}
		return ret;
	}

	/* Convert the time/date format specification in the stderr file */
	max = out_len;
	time_len = 0;
	/*
	 * Return the result of the format in err1.  If err1 is not big
	 * enough, strftime will return 0 (or max).  In this case we will
	 * try again with a bigger (+100 characters) buffer
	 */
	do {
		if (err1 != NULL) {
			free (err1);
		}
		err1 = (char *)malloc (max + 100);
		if (err1 == NULL) {
			if (out != NULL && out_len > 0) {
				free (out);
			}
			free (out1);
			return -1;
		}
		max += 100;

#if HAVE_STRFTIME && HAVE_LOCALTIME && HAVE_TIME
		time_len = strftime (err1, max, out, curr_tm);
#else
		strncpy (err1, out, max);
		time_len = max - 1;
		err1[time_len] = '\0';
#endif
	} while ((time_len == 0 && out[0] != '\0') || time_len == max);

	if (out != NULL && out_len > 0) {
		free (out);
	}


	/* Copy the result */
	if (j->file_stdout != NULL) {
		free (j->file_stdout);
	}
	if (j->file_stderr != NULL) {
		free (j->file_stderr);
	}
	j->file_stdout = out1;
	j->file_stderr = err1;
	return 0;
}


/*
 * Store the provided Linux Control Group name in the structure.
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 */
int
add_cgroup_job_run (job_run_t *j, const char *cgroup_name)
{
	char *s;

	if (j == NULL) {
		return 0;
	}
	if (cgroup_name == NULL || cgroup_name[0] == '\0') {
		if (j->cgroup_name != NULL) {
			free (j->cgroup_name);
			j->cgroup_name = NULL;
		}
		return 0;
	}
	s = (char *) malloc (schedwi_strlen (cgroup_name) + 1);
	if (s == NULL) {
		return -1;
	}
	strcpy (s, cgroup_name);
	if (j->cgroup_name != NULL) {
		free (j->cgroup_name);
	}
	j->cgroup_name = s;
	return 0;
}


/*
 * Compose the program string to run and the arguments.
 * If `load_env' is 0, the provided command will be directly executed.
 * If `load_env' is 1, the user environment (/etc/profile, .profile, ...)
 * must be loaded.  For that, the user shell will be executed insteed of
 * the command.  It will load the user environment.  The command will be
 * then run by the shell (the command is provided to the shell using the
 * -c option: see bash(1))
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 *   -2 --> Syntax error during parameter expansion in the command string
 *          (idx is set to -1) or the arguments (idx >= 0)
 *   -3 --> The command is missing
 */
int
add_command_job_run (	job_run_t *j, const char *command,
			const argument_t *args, int load_env, int *idx)
{
	int ret;
	char *shell;
	char *cmd;
	char *args_str;
	size_t cmd_len;
	argument_t args_sup, args_tmp;
	struct stat stat_cmd;

	if (j == 0 || command == NULL || command[0] == '\0') {
		return 0;
	}

	if (idx != NULL) {
		*idx = -1;
	}

	cmd = NULL;
	cmd_len = 0;
	ret = parse_string (	command, schedwi_strlen (command),
				&(j->environ),
				&cmd, &cmd_len);
	if (ret != 0) {
		return ret;
	}

	/* Test if the command exists */
	if (stat (cmd, &stat_cmd) != 0) {
		if (cmd != NULL && cmd_len > 0) {
			free (cmd);
		}
		return -3;
	}

	init_argument (&args_sup);

	/* No user envireonment to load */
	if (load_env == 0) {

		/* argv0 */
		if (add_arg (&args_sup, cmd) != 0) {
			if (cmd != NULL && cmd_len > 0) {
				free (cmd);
			}
			destroy_argument (&args_sup);
			return -1;
		}

		ret = parse_args (args, &args_sup, &(j->environ), idx);
		if (ret != 0) {
			if (cmd != NULL && cmd_len > 0) {
				free (cmd);
			}
			destroy_argument (&args_sup);
			return ret;
		}

		set_arg (&(j->arguments), &args_sup);

		if (j->command != NULL) {
			free (j->command);
		}
		j->command = cmd;
		return 0;
	}

	/*
	 * User environment has to be loaded.  The user shell will be run
	 * with the -c parameter and the first character of arrv[0] must
	 * be `-' (see bash(1))
	 */

	/* The command is the user's shell */
	shell = (char *) malloc (schedwi_strlen (j->shell) + 2);
	if (shell == NULL) {
		if (cmd != NULL && cmd_len > 0) {
			free (cmd);
		}
		destroy_argument (&args_sup);
		return -1;
	}
	shell[0] = '-';
	strcpy (shell + 1, j->shell);
	/* argv0 (shell) and argv1 (-c) */
	if (add_arg (&args_sup, shell) != 0 || add_arg (&args_sup, "-c") != 0)
	{
		if (cmd != NULL && cmd_len > 0) {
			free (cmd);
		}
		free (shell);
		destroy_argument (&args_sup);
		return -1;
	}

	/* Command */
	init_argument (&args_tmp);
	if (add_arg (&args_tmp, cmd) != 0) {
		if (cmd != NULL && cmd_len > 0) {
			free (cmd);
		}
		free (shell);
		destroy_argument (&args_tmp);
		destroy_argument (&args_sup);
		return -1;
	}
	if (cmd != NULL && cmd_len > 0) {
		free (cmd);
	}

	/* Parse all the arguments to substitute variables (eg. $FOO) */
	ret = parse_args (args, &args_tmp, &(j->environ), idx);
	if (ret != 0) {
		free (shell);
		destroy_argument (&args_tmp);
		destroy_argument (&args_sup);
		return ret;
	}

	/* Concatenate all the arguments */
	args_str = args_to_string (&args_tmp);
	if (args_str == NULL) {
		free (shell);
		destroy_argument (&args_tmp);
		destroy_argument (&args_sup);
		return -1;
	}

	/* Add the concatenated argument list to the argument list */
	if (add_arg (&args_sup, args_str) != 0) {
		free (shell);
		free (args_str);
		destroy_argument (&args_tmp);
		destroy_argument (&args_sup);
		return -1;
	}
	free (args_str);

	set_arg (&(j->arguments), &args_sup);

	if (j->command != NULL) {
		free (j->command);
	}
	strcpy (shell, j->shell); /* Remove the leading - */
	j->command = shell;
	return 0;
}


/*
 * Parse and copy the environment
 *
 * Return:
 *    0 --> No error
 *   -1 --> Memory allocation error
 *   -2 --> Syntax error in the key part (variable name)
 *   -3 --> Syntax error in the value part
 */
int
add_env_job_run (job_run_t *j, const environment_t *e, int *idx)
{
	return parse_param (e, &(j->environ), idx);
}

/*-----------------============== End Of File ==============-----------------*/
