/* Schedwi
   Copyright (C) 2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* reg_getca_clnt.c -- Get the CA certificate from the server */

#include <schedwi.h>

#if HAVE_LIBGNUTLS

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#if HAVE_STDIO_H
#include <stdio.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <utils.h>
#include <net_utils_sock.h>
#include <cert_utils.h>
#include <parse_json_result.h>
#include <reg_getca_clnt.h>


/*
 * Get the CA certificate from the server (schedwireg) and store it in the
 * provided file.
 *
 * Return:
 *   0 --> No error
 *  -1 --> Error (a message has been logged)
 *   1 --> Error on the server side.  In that case, and if not NULL, `srv_msg'
 *         contains the error message from the server (`srv_msg' may be NULL
 *         or empty if the server didn't send any message). It must be freed by
 *         the caller.
 */
int
getCA (int sock, const char *outfile, char **srv_msg)
{
	char *s, *data;
	char *buff;
	size_t len;
	ssize_t nb_read;
	int ret;
	FILE *file;

#if HAVE_ASSERT_H
	assert (outfile != NULL);
#endif

	/* Ask for the CA certificate */
	s = "[ \"getca\" ]";
	net_write_sock (sock, s, schedwi_strlen (s));

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_sock (sock, &buff, &len);
	if (nb_read < 0) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	data = NULL;
	ret = parse_json_result (buff, nb_read, &data, NULL);
	if (buff != NULL) {
		free (buff);
	}
	/* Local error */
	if (ret < 0) {
		return -1;
	}

	/* Server error */
	if (ret > 0) {
		if (srv_msg != NULL) {
			*srv_msg = data;
		}
		else {
			if (data != NULL) {
				free (data);
			}
		}
		return 1;
	}

	/* Convert the string to PEM */
	s = string2PEM (data);
	free (data);
	if (s == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}

	/* Write the file */
	file = fopen (outfile, "wb");
	if (file == NULL) {
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		free (s);
		return -1;
	}
	len = schedwi_strlen (s);
	if (fwrite (s, 1, len, file) != len) {
		lwc_writeLog (	LOG_ERR, _("%s: write error: %s"),
				outfile, strerror (errno));
		fclose (file);
		my_unlink (outfile);
		free (s);
		return -1;
	}
	free (s);
	if (fclose(file) != 0) {
		my_unlink (outfile);
		lwc_writeLog (LOG_ERR, _("%s: %s"), outfile, strerror (errno));
		return -1;
	}

	return 0;
}

#endif /* HAVE_LIBGNUTLS */

/*-----------------============== End Of File ==============-----------------*/
