/* Schedwi
   Copyright (C) 2011 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* reg_whoami_clnt.c -- Get the names and IP of the current agent */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#else
#if HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if HAVE_STRING_H
#include <string.h>
#endif
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <net_utils_sock.h>
#include <JSON_parser.h>
#include <reg_whoami_clnt.h>


enum steps {
	S_START,
	S_DATA,
	S_KEY_SUCCESS,
	S_KEY_DATA,
	S_KEY_REASON,
	S_RESULT,
	S_KEY_IP,
	S_KEY_DNS,
	S_DNS_ARRAY,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	error_reason_t reason;
	int failure;
	char *message;
	char *ip;
	char **names;
	int names_size;
	int names_idx;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;
	char *v;

	switch (type) {
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_START) {
				s->step = S_DATA;
			}
			else
			if (s->step == S_KEY_DATA) {
				s->step = S_RESULT;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				if (schedwi_strcasecmp (value->vu.str.value,
							"success") == 0)
				{
					s->step = S_KEY_SUCCESS;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"data") == 0)
				{
					s->step = S_KEY_DATA;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"reason") == 0)
				{
					s->step = S_KEY_REASON;
				}
			}
			else
			if (s->step == S_RESULT) {
				if (schedwi_strcasecmp (value->vu.str.value,
							"ip") == 0)
				{
					s->step = S_KEY_IP;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"dns") == 0)
				{
					s->step = S_KEY_DNS;
				}
			}
			break;
		case JSON_T_TRUE:
			if (s->step == S_KEY_SUCCESS) {
				s->failure = 0;
				s->step = S_DATA;
			}
			break;
		case JSON_T_FALSE:
			if (s->step == S_KEY_SUCCESS) {
				s->failure = 1;
				s->step = S_DATA;
			}
			break;
		case JSON_T_INTEGER:
			if (s->step == S_KEY_REASON) {
				s->reason = int_to_reg_error (
						value->vu.integer_value);
				s->step = S_DATA;
			}
			break;
		case JSON_T_ARRAY_BEGIN:
			if (s->step == S_KEY_DNS) {
				s->step = S_DNS_ARRAY;
			}
			break;
		case JSON_T_ARRAY_END:
			if (s->step == S_DNS_ARRAY) {
				s->step = S_RESULT;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_KEY_DATA) {
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->message != NULL) {
					free (s->message);
				}
				s->message = v;
				s->step = S_DATA;
			}
			else
			if (s->step == S_KEY_IP) {
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->ip != NULL) {
					free (s->ip);
				}
				s->ip = v;
				s->step = S_RESULT;
			}
			else
			if (	   s->step == S_DNS_ARRAY
				&& s->names_idx + 1 < s->names_size) 
			{
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				s->names[(s->names_idx)++] = v;
				s->names[s->names_idx] = NULL;
			}
			break;
		case JSON_T_OBJECT_END:
			if (s->step == S_RESULT) {
				s->step = S_DATA;
			}
			else
			if (s->step == S_DATA) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}


/*
 * Parse the JSON string which contains the IP/names details.
 *
 * The format of the string should be as follow:
 *     {
 *         "success" : false,
 *          "data" : "Database unavailable"
 *     }
 * OR
 *     {
 *         "success" : false,
 *         "reason" : 2,
 *          "data" : "Certificate still not ready"
 *     }
 * OR (for a success)
 *     {
 *         "success" : false,
 *          "data" : {
 *                      "ip" : "192.168.0.2",
 *                      "dns" : [ "flower.example.com" , "apple.example.com" ]
 *                   }
 *     }
 *
 * Return:
 *   0 --> The reply means the remote action was successfull.  `names' contains
 *         a NULL terminated array of names for this agent (the last item is
 *         the IP address).  Each item and the array itself must be freed by
 *         the caller (free()).  If not NULL, `names_size' contains the
 *         number of items in the array (final NULL not included).
 *   1 --> The reply means the remote action failed. If result_msg is not NULL,
 *         it is set with the message sent by the remote socket. It may be NULL
 *         if no message was sent. It must be freed by the caller.
 *  -1 --> Error (a message is displayed using lwc_writeLog())
 */
static int
parse_json_whoami (const char *json_string, ssize_t json_string_length,
			char ***names, int *names_size,
			char **result_msg, error_reason_t *reason)
{
	JSON_config config;
	struct JSON_parser_struct *jc;
	struct json_cb cb_data;
	int i;


#if HAVE_ASSERT_H
	assert (   json_string != NULL && json_string_length > 0
		&& names != NULL);
#endif

	cb_data.names_size = 15;
	cb_data.names = (char **)malloc (	  sizeof (char *)
						* (cb_data.names_size + 1));
	if (cb_data.names == NULL) {
		lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
		return -1;
	}
	cb_data.names_idx = 0;
	cb_data.step      = S_START;
	cb_data.error     = 0;
	cb_data.reason    = REG_OK;
	cb_data.failure   = 1;
	cb_data.message   = NULL;
	cb_data.ip        = NULL;

	/* JSON parser initialization */
	init_JSON_config (&config);
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;

	jc = new_JSON_parser (&config);

	for (i = 0; i < json_string_length && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse.c
		 */
		JSON_parser_char (jc, json_string[i]);
		if (cb_data.error != 0) {
			delete_JSON_parser (jc);
			if (cb_data.message != NULL) {
				free (cb_data.message);
			}
			if (cb_data.ip != NULL) {
				free (cb_data.ip);
			}
			while (--(cb_data.names_idx) >= 0) {
				free (cb_data.names[cb_data.names_idx]);
			}
			free (cb_data.names);
			lwc_writeLog (LOG_CRIT, _("Memory allocation error"));
			return -1;
		}
	}
	delete_JSON_parser (jc);

	/* Add the IP address at the end of the array */
	if (cb_data.ip != NULL) {
		cb_data.names[(cb_data.names_idx)++] = cb_data.ip;
		cb_data.names[cb_data.names_idx] = NULL;
	}

	/* Success ! */
	if (cb_data.failure == 0) {
		*names = cb_data.names;
		if (names_size != NULL) {
			*names_size = cb_data.names_idx;
		}
		if (cb_data.message != NULL) {
			free (cb_data.message);
		}
		return 0;
	}

	/* Failure */
	while (--(cb_data.names_idx) >= 0) {
		free (cb_data.names[cb_data.names_idx]);
	}
	free (cb_data.names);

	if (reason != NULL) {
		*reason = cb_data.reason;
	}
	if (result_msg != NULL) {
		*result_msg = cb_data.message;
	}
	else {
		if (cb_data.message != NULL) {
			free (cb_data.message);
		}
	}
	return cb_data.failure;
}


/*
 * Get the know names and IP address of the current agent.
 *
 * Return:
 *   0 --> No error.  `names' contains a NULL terminated array of names for
 *         this agent (the last item is the IP address).  Each item and the
 *         array itself must be freed by the caller (free()).  If not NULL,
 *         `names_size' contains the number of items in the array (final NULL
 *         not included).
 *  -1 --> Error (a message has been logged)
 *   1 --> Error on the server side.  In that case, and if not NULL,
 *         `result_msg' contains the error message from the server
 *         (`result_msg' may be NULL or empty if the server didn't send any
 *         message). It must be freed by the caller.  Also, `reason' is set if
 *         not NULL.  It contains the error code which indentify the error.
 */
int
whoami (int sock, char ***names, int *names_size,
	char **result_msg, error_reason_t *reason)
{
	char *s;
	char *buff;
	size_t len;
	ssize_t nb_read;
	int ret;

#if HAVE_ASSERT_H
	assert (names != NULL);
#endif

	/* Ask the server who am I */
	s = "[ \"whoami\" ]";
	net_write_sock (sock, s, schedwi_strlen (s));

	/* Read the reply */
	buff = NULL;
	len = 0;
	nb_read = net_read_sock (sock, &buff, &len);
	if (nb_read < 0) {
		if (buff != NULL) {
			free (buff);
		}
		return -1;
	}

	ret = parse_json_whoami (buff, nb_read, names, names_size,
				 result_msg, reason);
	if (buff != NULL) {
		free (buff);
	}
	if (ret == 0) {
		return 0;
	}
	if (ret < 0) {
		return -1;
	}
	return 1;
}

/*-----------------============== End Of File ==============-----------------*/
