/* Schedwi
   Copyright (C) 2007-2010 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* net_module_testfile.c -- Module to test the presence of a file */

#include <schedwi.h>

#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif

#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#if HAVE_CTYPE_H
#include <ctype.h>
#endif

#if HAVE_UNISTD_H
#include <unistd.h>
#endif

#if HAVE_ASSERT_H
#include <assert.h>
#endif

#if HAVE_ERRNO_H
#include <errno.h>
#endif
#ifndef errno
extern int errno;
#endif

#include <lib_functions.h>
#include <lwc_log.h>
#include <parsevar.h>
#include <net_module_testfile.h>
#include <env_utils.h>
#include <JSON_parser.h>

enum steps {
	S_START,
	S_OUTER_ARRAY,
	S_OUTER_ARRAY_MODULE,
	S_DATA,
	S_KEY_FILE,
	S_KEY_JOBID,
	S_KEY_ENV,
	S_ENV_ARRAY_START,
	S_END
};

struct json_cb {
	enum steps step;
	char error;
	char *file, *jobid;
	environment_t env;
};


/*
 * JSON callback function
 */
static int
json_char_callback (void* ctx, int type, const JSON_value* value)
{
	struct json_cb *s = (struct json_cb *)ctx;
	char *v;

	switch (type) {
		case JSON_T_ARRAY_BEGIN:
			if (s->step == S_START) {
				s->step = S_OUTER_ARRAY;
			}
			else if (s->step == S_KEY_ENV) {
				s->step = S_ENV_ARRAY_START;
			}
			break;
		case JSON_T_STRING:
			if (s->step == S_OUTER_ARRAY) {
				s->step = S_OUTER_ARRAY_MODULE;
			}
			else if (s->step == S_ENV_ARRAY_START) {
				v = strchr (value->vu.str.value, '=');
				if (	   v != NULL
					&& add_env_s (&(s->env),
						value->vu.str.value,
						v - value->vu.str.value,
						v + 1,
					value->vu.str.length
					- (v + 1 - value->vu.str.value)) != 0)
				{
					s->error = 1;
					return 1;
				}
			}
			else if (s->step == S_KEY_FILE) {
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->file != NULL) {
					free (s->file);
				}
				s->file = v;
				s->step = S_DATA;
			}
			else if (s->step == S_KEY_JOBID) {
				v = (char *)malloc (value->vu.str.length + 1);
				if (v == NULL) {
					s->error = 1;
					return 1;
				}
				strncpy (v, value->vu.str.value,
						value->vu.str.length);
				v[value->vu.str.length] = '\0';
				if (s->jobid != NULL) {
					free (s->jobid);
				}
				s->jobid = v;
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_BEGIN:
			if (s->step == S_OUTER_ARRAY_MODULE) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_KEY:
			if (s->step == S_DATA) {
				if (schedwi_strcasecmp (value->vu.str.value,
							"file") == 0)
				{
					s->step = S_KEY_FILE;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"job id") == 0)
				{
					s->step = S_KEY_JOBID;
				}
				else
				if (schedwi_strcasecmp (value->vu.str.value,
							"env") == 0)
				{
					s->step = S_KEY_ENV;
				}
			}
			break;
		case JSON_T_ARRAY_END:
			if (s->step == S_ENV_ARRAY_START) {
				s->step = S_DATA;
			}
			break;
		case JSON_T_OBJECT_END:
			if (s->step == S_DATA) {
				s->step = S_END;
			}
			break;
	}

	return 1;
}


/*
 * Test the presence of a file/directory/socket/... and send the result to
 * the network client
 *
 * Return:
 *   0 --> No error
 *  -1 --> Memory allocation error (a message is displayed by using
 *         lwc_writeLog())
 *  -2 --> Syntax error (a message is displayed by using lwc_writeLog())
 */
int
test_file (net_id *sock, char *buffer, size_t size)
{
	char *s, *err_msg, *out;
	size_t out_len;
	struct stat file_info;
	int i, idx;
        JSON_config config;
        struct JSON_parser_struct *jc;
	struct json_cb cb_data;
	environment_t parsed_env;


#if HAVE_ASSERT_H
	assert (sock != NULL && buffer != NULL);
#endif

	/* JSON parser initialization */
	init_JSON_config (&config);

	init_environment (&(cb_data.env));
	cb_data.step                  = S_START;
	cb_data.error                 = 0;
	cb_data.file                  = NULL;
	cb_data.jobid                 = NULL;
	config.depth                  = 20;
	config.callback               = &json_char_callback;
	config.allow_comments         = 0;
	config.handle_floats_manually = 0;
	config.callback_ctx           = &cb_data;

	jc = new_JSON_parser (&config);

	/* Parse the JSON string */
	for (i = 0; i < size && cb_data.step != S_END; i++) {
		/*
		 * No need to check the return code as the JSON string has
		 * already been parsed once in net_parse.c
		 */
		JSON_parser_char (jc, buffer[i]);
		if (cb_data.error != 0) {
			delete_JSON_parser (jc);
			destroy_environment (&(cb_data.env));
			if (cb_data.file != NULL) {
				free (cb_data.file);
			}
			if (cb_data.jobid != NULL) {
				free (cb_data.jobid);
			}
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Memory allocation error");
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_CRIT, s);
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			return -1;
		}
	}
	delete_JSON_parser (jc);

	/* Check that the file name has been provided */
	if (cb_data.file == NULL) {
		destroy_environment (&(cb_data.env));
		if (cb_data.jobid != NULL) {
			free (cb_data.jobid);
		}
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (sock, s, schedwi_strlen (s));
		s = _("Syntax error: file name not provided");
		net_write (sock, s, schedwi_strlen (s));
		lwc_writeLog (LOG_ERR, s);
		s = "\" }";
		net_write (sock, s, schedwi_strlen (s));
		return -2;
	}

	/* Parse the provided environment and create a new one */
	init_environment (&parsed_env);
	if (cb_data.jobid != NULL) {
		if (add_env (&parsed_env, "SCHEDWI_JOBID", cb_data.jobid) != 0)
		{
			destroy_environment (&parsed_env);
			destroy_environment (&(cb_data.env));
			free (cb_data.file);
			free (cb_data.jobid);
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Memory allocation error");
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_CRIT, s);
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			return -1;
		}
		free (cb_data.jobid);
	}

	switch (parse_param (&(cb_data.env), &parsed_env, &idx)) {
		case -1:
			destroy_environment (&parsed_env);
			destroy_environment (&(cb_data.env));
			free (cb_data.file);
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Memory allocation error");
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_CRIT, s);
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			return -1;

		case -2: case -3:
			destroy_environment (&parsed_env);
			free (cb_data.file);
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Syntax error in the environment");
			net_write (sock, s, schedwi_strlen (s));
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_ERR,
				_("Syntax error in the environment: %s"),
				cb_data.env.env[idx]);
			destroy_environment (&(cb_data.env));
			return -2;
	}
	destroy_environment (&(cb_data.env));

	/* Parse the file name */
	out = NULL;
	out_len = 0;
	switch (parse_string (	cb_data.file, schedwi_strlen (cb_data.file),
				&parsed_env, &out, &out_len))
	{
		case -1:
			destroy_environment (&parsed_env);
			free (cb_data.file);
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Memory allocation error");
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_CRIT, s);
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			return -1;

		case -2:
			destroy_environment (&parsed_env);
			s = "{ \"success\" : false, \"data\" : \"";
			net_write (sock, s, schedwi_strlen (s));
			s = _("Syntax error in the file name");
			net_write (sock, s, schedwi_strlen (s));
			s = "\" }";
			net_write (sock, s, schedwi_strlen (s));
			lwc_writeLog (LOG_ERR,
				_("Syntax error in the file name: %s"),
				cb_data.file);
			free (cb_data.file);
			return -2;
	}
	destroy_environment (&parsed_env);
	free (cb_data.file);

	/* Check the file */
	if (out == NULL || out_len == 0 || stat (out, &file_info) != 0) {
		err_msg = strerror (errno);
		s = "{ \"success\" : false, \"data\" : \"";
		net_write (sock, s, schedwi_strlen (s));
		net_write (sock, err_msg, schedwi_strlen (err_msg));
		s = "\" }";
		net_write (sock, s, schedwi_strlen (s));
	}
	else {
		s = "{ \"success\" : true }";
		net_write (sock, s, schedwi_strlen (s));
	}
	if (out != NULL) {
		free (out);
	}
	return 0;
}

/*-----------------============== End Of File ==============-----------------*/
