/* Schedwi
   Copyright (C) 2013 Herve Quatremain

   This file is part of Schedwi.

   Schedwi is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3 of the License, or
   (at your option) any later version.

   Schedwi is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/* graph_node_position.js -- Compute the position of the link graph nodes */


/*
 * Requires the `graph' and `job_details' collections provided by the
 * schedwi GUI server (see web/link_graph.py)
 */

var MAX_ITERATIONS = 24;
var leaves = [];
var rank_by_node = {};
var rank_min = 0;
var rank_max = 0;
var column_by_node = {};
var num_columns = 0;


/*
 * Return the list of parent nodes
 *
 * @param node:
 *      The node ID for which the parents should be returned.
 * @return:
 *      The array that contains the parents ID.
 */
function get_parents(node)
{
    if (node in graph) {
        return graph[node];
    }
    else {
        return [];
    }
}


/*
 * Return the list of child nodes
 *
 * @param node:
 *      The node ID for which the children should be returned.
 * @return:
 *      The array that contains the child IDs.
 */
function get_children(node)
{
    var ret = [];
    var k;
    var i;

    for (k in graph) {
        if (graph.hasOwnProperty(k)) {  /* For Lint */
            for (i = 0; i < graph[k].length; i++) {
                if (node == graph[k][i].id) {
                    ret.push({id: k, status: graph[k][i].status});
                    break;
                }
            }
        }
    }
    return ret;
}


/*
 * Recursively compute the rank (Y-coordinate or level) of the nodes in the
 * graph dictionnary.
 * The global rank_by_node, rank_min, rank_max and leaves variables are
 * updated by this function.
 *
 * @param node:
 *       The node ID for which the rank must be set.
 * @param Rank:
 *       the rank of the node.
 */
function compute_rank_recursive(node, rank)
{
    var i;
    var parents;
    var children;

    rank_by_node[node] = rank;
    parents = get_parents(node);
    if (parents.length === 0) {
        leaves.push(node);
    }
    for (i = 0; i < parents.length; i++) {
        if (! (parents[i].id in rank_by_node)) {
            compute_rank_recursive(parents[i].id, rank - 1);
        }
        else if (rank <= rank_by_node[parents[i].id]) {
            rank = rank_by_node[parents[i].id] + 1;
            rank_by_node[node] = rank;
        }
    }
    rank = rank_by_node[node];

    children = get_children(node);
    for (i = 0; i < children.length; i++) {
        if (! (children[i].id in rank_by_node)) {
            compute_rank_recursive(children[i].id, rank + 1);
        }
        else if (rank >= rank_by_node[children[i].id]) {
            rank_by_node[children[i].id] = rank + 1;
            if (rank + 1 > rank_max) {
                rank_max = rank + 1;
            }
        }
    }
    if (rank < rank_min) {
        rank_min = rank;
    }
    if (rank > rank_max) {
        rank_max = rank;
    }
}


/*
 * Compute the rank (or Y-coordinate or level) of the nodes in the graph.
 * The global rank_by_node, rank_min, rank_max and leaves variables are
 * updated by this function.
 */
function compute_rank()
{
    var key;
    var i;
    var j;
    var min;
    var children;

    /* Get the first key */
    for (key in graph) {
        if (graph.hasOwnProperty(key)) {  /* For Lint */
            break;
        }
    }

    /* Only nodes without links */
    if (! key) {
        for (key in job_details) {
            rank_by_node[key] = 0;
        }
    }
    else {
        /*
         * Start the rank computation by the first node (in fact any node
         * will do must a first one must be choosen)
         * In the process the leaves are stored in the `leaves' array.
         */
        compute_rank_recursive(key, 0);

        /*
         * The rank of leaves is recalcultated because in some cases, it can
         * be wrong.
         */
        for (i = 0; i < leaves.length; i++) {
            min = rank_max;
            children = get_children(leaves[i]);
            for (j = 0; j < children.length; j++) {
                if (min > rank_by_node[children[j].id]) {
                    min = rank_by_node[children[j].id];
                }
            }
            rank_by_node[leaves[i]] = min - 1;
        }
    }
}


/*
 * Add fake nodes in the graph when there are more than one level between
 * linked nodes.
 */
function add_fake_nodes()
{
    var start_fake_id = -1;
    var node;

    for (node in graph) {
        if (graph.hasOwnProperty(node)) {  /* For Lint */
            var parents;
            var i;

            parents = get_parents(node);
            for (i = 0; i < parents.length; i++) {
                var r;
                var n;
                var p;
                var idx;

                r = rank_by_node[node];
                n = node;
                p = parents[i];
                idx = i;

                /*
                 * While there is more than one level between the node
                 * and its parent, add a fake node between them.
                 */
                while (r - 1 > rank_by_node[p.id]) {
                    r--;
                    graph[n].splice(idx, 1);       // Remove parent node
                    graph[n].push({id: start_fake_id,
                                   status: p.status});  // Add new fake node
                    graph[start_fake_id] = [ p ];  // Add old node to fake node
                    rank_by_node[start_fake_id] = r;
                    job_details[start_fake_id] = {
                                            name: job_details[p.id].name,
                                            type: job_details[p.id].type,
                                            state: job_details[p.id].state,
                                            fake: true };
                    n = start_fake_id;
                    start_fake_id--;
                    idx = 0;
                }
            }
        }
    }
}


/*
 * Return all the node at a specific rank/level
 *
 * @param r:
 *      The rank number.
 * @return:
 *      An array that contains all the nodes ID at that rank.
 */
function get_nodes_at_rank(r)
{
    var ret = [];
    var node;

    for (node in rank_by_node) {
        if (rank_by_node.hasOwnProperty(node)) {  /* For Lint */
            if (rank_by_node[node] === r) {
                ret.push(node);
            }
        }
    }
    return ret;
}


/*
 * Initialize the column_by_node array with an arbitrary order
 */
function init_column_by_node()
{
    var next_column_by_rank = {};
    var node;
    for (node in rank_by_node) {
        if (rank_by_node.hasOwnProperty(node)) {  /* For Lint */
            if (rank_by_node[node] in next_column_by_rank) {
                column_by_node[node] = next_column_by_rank[rank_by_node[node]];
                next_column_by_rank[rank_by_node[node]]++;
            }
            else {
                column_by_node[node] = 0;
                next_column_by_rank[rank_by_node[node]] = 1;
            }
            if (next_column_by_rank[rank_by_node[node]] > num_columns) {
                num_columns = next_column_by_rank[rank_by_node[node]];
            }
        }
    }
}


/*
 * Return the median value of the positions of the parent of
 * the provided node
 *
 * @param node:
 *       The node ID for which the rank must be set.
 * @return:
 *       The median value
 */
function median_value(node)
{
    var parents;
    var i;
    var len;
    var parent_values = [];

    parents = get_parents(node);
    for (i = 0; i < parents.length; i++) {
        parent_values.push(column_by_node[parents[i].id]);
    }
    len = parent_values.length;
    if (len === 0) {
        return -1.0;
    }
    if (len === 2) {
        return (parent_values[0] + parent_values[1]) / 2.0;
    }
    parent_values.sort(function(a,b){return a - b;});
    return parent_values[Math.floor(len / 2)];
}


/*
 * Sort function used by sort() bellow
 */
function sort_func(a, b)
{
    return a[1] - b[1];
}


/*
 * Compute the X-axis position of each node
 */
function compute_colum_by_node()
{
    var iteration;
    var r;
    var i;
    var nodes_at_rank;
    var num_nodes_at_this_rank;
    var offset;

    init_column_by_node();

    for (iteration = 0; iteration < MAX_ITERATIONS; iteration++) {
        /* Loop from the second rank up to the last one */
        for (r = rank_min + 1; r <= rank_max; r++) {
            var score = [];

            nodes_at_rank = get_nodes_at_rank(r);
            num_nodes_at_this_rank = nodes_at_rank.length;
            for (i = 0; i < num_nodes_at_this_rank; i++) {
                /* Score for each node at that rank */
                score.push([nodes_at_rank[i],
                            median_value(nodes_at_rank[i])]);
            }
            if (num_nodes_at_this_rank < num_columns) {
                /* Center nodes on the X-axis */
                offset = (num_columns - num_nodes_at_this_rank) / 2;
            }
            else {
                offset = 0;
            }
            /* Update the X position of the nodes at that rank */
            score.sort(sort_func);
            for (i = 0; i < num_nodes_at_this_rank; i++) {
                column_by_node[score[i][0]] = offset++;
            }
        }
    }
    /* Center first-rank nodes on the X-axis */
    nodes_at_rank = get_nodes_at_rank(rank_min);
    num_nodes_at_this_rank = nodes_at_rank.length;
    nodes_at_rank.sort(function(a, b) {
                           return column_by_node[a] - column_by_node[b];});
    if (num_nodes_at_this_rank < num_columns) {
        offset = (num_columns - num_nodes_at_this_rank) / 2;
        for (i = 0; i < num_nodes_at_this_rank; i++) {
            column_by_node[nodes_at_rank[i]] = offset++;
        }
    }
}

