# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to copy a calendar or a calendar directory."""

import sys
import getopt

import simple_queries_cal
import path_cal
import rm
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: cp [OPTION]... SOURCE DEST
          or:  cp [OPTION]... SOURCE... DIR
        Copy SOURCE to DEST, or multiple SOURCE(s) to DIR.

        Options:
          -f, --force          never prompt for confirmation and overwrite the
                               destination if it already exists.
          -r, -R, --recursive  copy directories recursively.
          -h, --help  display this help.
    """))


def cp_recursive(session, src_cal, dest_cal, base=None):
    """Recursively copy calendars in a calendar directory.

    Arguments:
    session -- SQLAlchemy session
    src_cal -- calendar or calendar directory to copy (source)
    dest_cal -- destination calendar directory
    base -- new name for the copied calendar (if not set the destination
            calendar will have the same name as the source one)
    """
    new_cal = src_cal.copy()
    new_cal.parent = dest_cal.id
    if base is not None:
        new_cal.name = base.decode('utf-8')
    session.add(new_cal)
    if src_cal.entry_type == 0:
        return 0
    for cal in simple_queries_cal.sql_get_cal_children(session, src_cal.id):
        cp_recursive(session, cal, new_cal)
    return 0


def _rename(sql_session, cwd, src_path, dst, recursive):
    """Rename a calendar or directory.

    Arguments:
    sql_session -- SQLAlchemy session
    cwd -- current working directory (a path_cal.PathCal object)
    src_path -- path of the calendar or directory to copy
    dst -- path of the destination
    recursive -- whether the copy is recursive or not

    """
    (base, dest_path) = path_cal.get_new_paths(sql_session, dst, cwd)
    lp = len(dest_path)
    if lp == 0:
        sys.stderr.write(_("cp: no such destination directory\n"))
        return 1
    if lp > 1:
        sys.stderr.write(_("cp: `%s/%s': no such calendar or directory\n") %
                         (dest_path[0], base))
        return 1
    session = sql_session.open_session()
    try:
        dcal = simple_queries_cal.sql_get_cal(session, dest_path[0].id[-1])
    except:
        sys.stderr.write(_("cp: no such destination directory\n"))
        sql_session.cancel_session(session)
        return 1
    try:
        scal = simple_queries_cal.sql_get_cal(session, src_path.id[-1])
    except:
        sys.stderr.write(_("cp: no such calendar or directory\n"))
        sql_session.cancel_session(session)
        return 1
    # The source is a directory and the recursive option (-r) is not set
    if scal.entry_type != 0 and not recursive:
        sys.stderr.write(_("cp: omitting directory `%s'\n") % src_path)
        sql_session.cancel_session(session)
        return 1
    cp_recursive(session, scal, dcal, base)
    sql_session.close_session(session)
    return 0


def _cp_multi(sql_session, cwd, paths, recursive, force):
    """Copy several source calendars/directory.

    Arguments:
    sql_session -- SQLAlchemy session
    cwd -- current working directory (a path_cal.PathCal object)
    paths -- list of calendars/directory
    recursive -- whether the copy is recursive or not
    force -- whether to overwrite the destination if it exists

    """
    dest = paths.pop()
    session = sql_session.open_session()
    try:
        dest_cal = simple_queries_cal.sql_get_cal(session, dest.id[-1])
    except:
        sys.stderr.write(_("cp: no such destination directory\n"))
        sql_session.cancel_session(session)
        return 1
    sql_session.close_session(session)
    if len(paths) > 1:
        if dest_cal.entry_type == 0:
            sys.stderr.write(_("cp: target `%s' is not a directory\n") % dest)
            return 1
    else:
        if dest.id[-1] == paths[0].id[-1]:
            sys.stderr.write(_("cp: cannot copy `%s' to itself\n") % dest)
            return 1
        if dest_cal.entry_type == 0:
            if not force:
                sys.stderr.write(_("cp: target `%s' already exists\n") % dest)
                return 1
            else:
                session = sql_session.open_session()
                rm.rm_recursive(session, dest, force, recursive=True)
                sql_session.close_session(session)
                return _rename(sql_session, cwd, paths[0],
                               str(dest), recursive)
    session = sql_session.open_session()
    dest_content = simple_queries_cal.sql_get_cal_children(session,
                                                           dest_cal.id)
    for p in paths:
        try:
            cal = simple_queries_cal.sql_get_cal(session, p.id[-1])
        except:
            sys.stderr.write(_("cp: no such calendar or directory\n"))
            sql_session.cancel_session(session)
            return 1
        # Check to see if the source calendar name is not already used by
        # an other calendar in the destination directory
        for dest_child in dest_content:
            if dest_child.name == cal.name:
                if not force:
                    sys.stderr.write(_("cp: `%s' already exists in `%s'\n") %
                                     (cal.name, dest))
                    sql_session.cancel_session(session)
                    return 1
                else:
                    if recursive or dest_child.entry_type == 0:
                        rm.rm_recursive(
                            session,
                            path_cal.PathCal(session, id=dest_child.id),
                            True,
                            recursive=True)
                        sql_session.close_session(session)
                        session = sql_session.open_session()
        if cal.entry_type != 0:
            if not recursive:
                sys.stderr.write(_("cp: omitting directory `%s'\n") % p)
                sql_session.cancel_session(session)
                return 1
            try:
                dest.id.index(cal.id)
            except:
                cp_recursive(session, cal, dest_cal)
            else:
                sys.stderr.write(
                    _("cp: cannot copy `%s' to a sub-directory of itself\n") %
                    p)
                sql_session.cancel_session(session)
                return 1
        else:
            cp_recursive(session, cal, dest_cal)
    sql_session.close_session(session)
    return 0


def cp(sql_session, current_cwd, arguments):
    """Copy a calendar or directory.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the cp command (list of calendars)

    """
    try:
        optlist, args = getopt.getopt(arguments, "frRh",
                                      ["force", "recursive", "help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("cp: ") + str(err) + "\n")
        return 1
    force = False
    recursive = False
    for o, a in optlist:
        if o in ("-f", "--force"):
            force = True
        elif o in ("-r", "-R", "--recursive"):
            recursive = True
        elif o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    num_args = len(args)
    dest_rename = None
    if args:
        for i in range(0, num_args):
            p = path_cal.get_paths(sql_session, args[i], current_cwd)
            # An argument is an unknow path (if it's not the last one)
            if not p:
                if i + 1 != num_args:
                    sys.stderr.write(
                        _("cp: `%s': no such calendar or directory\n") %
                        args[i])
                    return 1
                else:
                    dest_rename = args[i]
            paths.extend(p)
    else:
        sys.stderr.write(_("cp: missing calendar or directory operand\n"))
        return 1
    if not paths:
        sys.stderr.write(_("cp: `%s': no such calendar or directory\n") %
                         args[0])
        return 1
    lp = len(paths)
    # Copy and rename
    if lp == 1:
        if num_args < 2:
            sys.stderr.write(
                _("cp: no enough operand or unknown calendar/directory\n"))
            return 1
        return _rename(sql_session, current_cwd, paths[0], args[1], recursive)
    # Copy
    if dest_rename is not None:
        sys.stderr.write(_("cp: target `%s' does not exist\n") % dest_rename)
        return 1
    return _cp_multi(sql_session, current_cwd, paths, recursive, force)
