# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to rename or move a calendar or calendar directory."""

import sys
import getopt

import simple_queries_cal
import path_cal
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: mv SOURCE DEST
          or:  mv SOURCE... DIR
        Rename SOURCE to DEST, or move SOURCE(s) to DIR.

        Option:
          -h, --help  display this help.
    """))


def rename(sql_session, cwd, src_path, dst, workload=None):
    """Rename a calendar or directory.

    Arguments:
    sql_session -- SQLAlchemy session
    cwd -- current working directory (a path_cal.PathCal object)
    src_path -- calendar/directory to rename
    dst -- destination
    workload -- workload to use

    """
    (base, dest_path) = path_cal.get_new_paths(sql_session, dst, cwd,
                                               workload=workload)
    lp = len(dest_path)
    if lp == 0:
        sys.stderr.write(_("mv: no such destination directory\n"))
        return 1
    if lp > 1:
        sys.stderr.write(_("mv: `%s/%s': no such calendar or directory\n") %
                         (dest_path[0], base))
        return 1
    session = sql_session.open_session()
    dest_dir = simple_queries_cal.sql_get_cal(session, dest_path[0].id[-1],
                                              workload)
    try:
        src_cal = simple_queries_cal.sql_get_cal(session, src_path.id[-1],
                                                 workload)
    except:
        sys.stderr.write(_("mv: `%s': no such calendar or directory\n") %
                         src_path)
        sql_session.cancel_session(session)
        return 1
    src_cal.parent = dest_dir.id
    src_cal.name = base.decode('utf-8')
    sql_session.close_session(session)
    return 0


def move(sql_session, paths, workload=None):
    """Move calendars or directories.

    Arguments:
    sql_session -- SQLAlchemy session
    paths -- list of calendars/directories
    workload -- workload to use

    """
    dest = paths.pop()
    session = sql_session.open_session()
    try:
        dest_cal = simple_queries_cal.sql_get_cal(session, dest.id[-1],
                                                  workload)
    except:
        sys.stderr.write(_("mv: no such destination calendar or directory\n"))
        sql_session.cancel_session(session)
        return 1
    if len(paths) > 1:
        if dest_cal.entry_type == 0:
            sys.stderr.write(_("mv: target `%s' is not a directory\n") % dest)
            sql_session.cancel_session(session)
            return 1
        for p in paths:
            try:
                cal = simple_queries_cal.sql_get_cal(session, p.id[-1],
                                                     workload)
            except:
                sys.stderr.write(_("mv: `%s': no such calendar or directory\n")
                                 % p)
                sql_session.cancel_session(session)
                return 1
            try:
                dest.id.index(cal.id)
            except:
                cal.parent = dest_cal.id
            else:
                sys.stderr.write(
                    _("mv: cannot move `%s' to a sub-directory of itself\n")
                    % cal.name)
                sql_session.cancel_session(session)
                return 1
    else:
        try:
                cal = simple_queries_cal.sql_get_cal(session, paths[0].id[-1],
                                                     workload)
        except:
            sys.stderr.write(_("mv: `%s': no such calendar or directory\n")
                             % paths[0])
            sql_session.cancel_session(session)
            return 1
        if dest_cal.entry_type != 0:
            try:
                dest.id.index(cal.id)
            except:
                cal.parent = dest_cal.id
            else:
                sys.stderr.write(
                    _("mv: cannot move `%s' to a sub-directory of itself\n")
                    % cal.name)
                sql_session.cancel_session(session)
                return 1
        else:
            sys.stderr.write(_("mv: target `%s' already exists\n") % dest)
            sql_session.cancel_session(session)
            return 1
    sql_session.close_session(session)
    return 0


def mv(sql_session, current_cwd, arguments, workload=None):
    """Rename or move calendars and directories.

    Arguments:
    sql_session -- SQLAlchemy session
    current_cwd -- current working directory (a path_cal.PathCal object)
    arguments -- list of arguments given to the mv command (list of calendars)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("mv: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    paths = list()
    num_args = len(args)
    dest_rename = None
    if args:
        for i in range(0, num_args):
            p = path_cal.get_paths(sql_session, args[i], current_cwd,
                                   workload=workload)
            # An argument is an unknow path (if it's not the last one)
            if not p:
                if i + 1 != num_args:
                    sys.stderr.write(
                        _("mv: `%s': no such calendar or directory\n")
                        % args[i])
                    return 1
                else:
                    dest_rename = args[i]
            paths.extend(p)
    else:
        sys.stderr.write(_("mv: missing calendar or directory operand\n"))
        return 1
    if not paths:
        sys.stderr.write(_("mv: `%s': no such calendar or directory\n")
                         % args[0])
        return 1
    lp = len(paths)
    # Rename
    if lp == 1:
        if num_args < 2:
            sys.stderr.write(
                _("mv: no enough operand or unknown calendar/directory\n"))
            return 1
        return rename(sql_session, current_cwd, paths[0], args[1], workload)
    # Move
    if dest_rename is not None:
        sys.stderr.write(_("mv: target `%s' does not exist\n") % dest_rename)
        return 1
    return move(sql_session, paths, workload)
