# Schedwi
# Copyright (C) 2011, 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to print the parameters of environment groups."""

import sys
import getopt

import environments_utils
from help import print_trim


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: cat ENV...
        Print the parameters of the provided environment groups.

        Options:
          -h, --help  display this help.
    """))

_FIELD_WIDTH = 12


def _print_val(title, fmt=None, val=None):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    fmt -- format string for the parameter value
    val -- value

    """
    if fmt is not None:
        if val is not None:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt % val
        else:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt
    else:
        print "%s :" % title.rjust(_FIELD_WIDTH)


def _print_env(session, env):
    """Print the environment group parameters.

    Arguments:
    session -- SQLAlchemy session
    env -- environment group object to print
    workload -- workload to use

    """
    _print_val(_("Name"), "%s", env.name)
    pos = 1
    for e in env.environment_var:
        _print_val("Var %d" % pos, "%s=%s", (e.env_key, e.env_value))
        pos += 1
    _print_val(_("Description"), "%s", env.description)


def cat(sql_session, arguments, workload=None):
    """Print environment groups parameters.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the cat command
                 (list of env groups)
    workload -- workload to use

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("cat: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write(_("cat: missing operand\n"))
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        env_list = environments_utils.name2env_list(session, a, workload)
        if not env_list:
            sys.stderr.write(_("cat: `%s': no such environment group\n") % a)
        else:
            first = True
            for e in env_list:
                if not first:
                    print
                _print_env(session, e)
                first = False
    sql_session.close_session(session)
    return 0 if not error else 1
