# Schedwi
# Copyright (C) 2011, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Module to print the parameters of hosts."""

import sys
import getopt

from tables.host_environment import host_environment
from tables.environments import environments
from tables.clusters import clusters
from tables.host_clusters import host_clusters
import host_utils
from help import print_trim
import cert


def usage():
    """Print a usage message on STDOUT."""
    print_trim(_("""Usage: cat HOST...
        Print the parameters of the provided HOSTs.
        HOST must have the following format:
           HOSTNAME[:PORT]
        For IPv6, the address must be enclosed between square brackets (ie.
        [fe80::210:a2ff:fa1d:aabb:ccdd]:2006)

        Option:
          -h, --help  display this help.
    """))

_FIELD_WIDTH = 17


def _print_val(title, fmt=None, val=None):
    """Print a parameter value.

    Arguments:
    title -- parameter name
    fmt -- format string for the parameter value
    val -- value

    """
    if fmt is not None:
        if val is not None:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt % val
        else:
            print "%s :" % title.rjust(_FIELD_WIDTH), fmt
    else:
        print "%s :" % title.rjust(_FIELD_WIDTH)


def _print_host(session, host):
    """Print a host parameters.

    Arguments:
    session -- SQLAlchemy session
    host -- host object to print

    """
    _print_val(_("Hostname"), "%s", host.hostname)
    _print_val(_("Port"), "%s", host.portnum)
    _print_val(_("SSL"), "%s", host.sslenable)
    if host.sslcert:
        certX509 = cert.schedwiX509(host.sslcert.encode('ascii'))
        _print_val(_("Certificate"), "%s", certX509.get_subject())
        dns = certX509.get_dns()
        if dns:
            _print_val(_("Cert alt names"), "%s", dns)
        _print_val(_("Cert activation"), "%s", certX509.get_not_before())
        _print_val(_("Cert expiration"), "%s", certX509.get_not_after())
    else:
        _print_val(_("Certificate"))
    q = session.query(environments).filter(host_environment.host_id == host.id)
    q = q.filter(host_environment.env_id == environments.id)
    for e in q.order_by(host_environment.position):
        _print_val(_("Environment group"), "%s", e.name)
    q = session.query(clusters).filter(host_clusters.host_id == host.id)
    q = q.filter(host_clusters.cluster_id == clusters.id)
    for c in q.order_by(clusters.name):
        _print_val(_("Cluster"), "%s", c.name)
    _print_val(_("Description"), "%s", host.description)


def cat(sql_session, arguments):
    """Print host parameters.

    Arguments:
    sql_session -- SQLAlchemy session
    arguments -- list of arguments given to the cat command (list of hosts)

    """
    try:
        optlist, args = getopt.getopt(arguments, "h", ["help"])
    except getopt.GetoptError, err:
        sys.stderr.write(_("cat: ") + str(err) + "\n")
        return 1
    for o, a in optlist:
        if o in ("-h", "--help"):
            usage()
            return 0
    if not args:
        sys.stderr.write(_("cat: missing operand\n"))
        return 1
    session = sql_session.open_session()
    error = False
    for a in args:
        host_list = host_utils.name2host_list(session, a)
        if not host_list:
            sys.stderr.write(_("cat: `%s': no such host\n") % a)
            error = True
        else:
            first = True
            for h in host_list:
                if not first:
                    print
                _print_host(session, h)
                first = False
    sql_session.close_session(session)
    return 0 if not error else 1
