# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Simple database queries for the acknowledge_manual table."""

from datetime import datetime
import sqlalchemy.orm.session

from tables.acknowledge_manual import acknowledge_manual


def sql_get_acknowledge_request_by_job_id(sql_session, job_id, workload):
    """Retrieve an entry from the acknowledge_manual table by job ID.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param job_id:
                the job ID to look for in the database.
    @param workload:
                workload to consider.
    @return:    the L{tables.acknowledge_manual} object.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID/workload is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    query = session.query(acknowledge_manual)
    query = query.filter(acknowledge_manual.workload_date == workload)
    query = query.filter_by(job_id=job_id)
    try:
        ack_entry = query.one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return ack_entry


def sql_get_acknowledge_request_by_urn_and_password(sql_session, urn, passwd):
    """Retrieve an entry from the acknowledge_manual table by urn and password.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param urn:
                the Web urn to look for in the database.
    @param passwd:
                the associated password.
    @return:    the L{tables.acknowledge_manual} object.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given urn/password is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    query = session.query(acknowledge_manual)
    query = query.filter(acknowledge_manual.urn == urn)
    query = query.filter(acknowledge_manual.password == passwd)
    try:
        ack_entry = query.one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return ack_entry


def sql_get_acknowledge_request_by_workload(sql_session, workload):
    """Retrieve all the entries from the acknowledge_manual table for the
    specified workload.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param workload:
                workload to look for.
    @return:    the list of L{tables.acknowledge_manual} objects.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    query = session.query(acknowledge_manual)
    query = query.filter(acknowledge_manual.workload_date == workload)
    query = query.order_by(acknowledge_manual.status)
    query = query.order_by(acknowledge_manual.ack_time)
    lst = query.all()
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return lst


def sql_update_acknowledge_request(sql_session, job_id, workload,
                                   username="web admin"):
    """Update an entry from the acknowledge_manual table.

    @param sql_session:
                SQLAlchemy session.
    @param job_id:
                the job ID to update for in the database.
    @param workload:
                the associated workload.
    @param username:
                the name of the user who acknowledged the job/jobset.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given job ID/workload is not in the database.
    """
    session = sql_session.open_session()
    try:
        ack = sql_get_acknowledge_request_by_job_id(session, job_id, workload)
    except:
        sql_session.close_session(session)
        raise
    ack.status = 1
    ack.username = username
    ack.ack_time = datetime.now()
    sql_session.close_session(session)
