# Schedwi
# Copyright (C) 2012 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Simple database queries for calendars."""

import sqlalchemy.orm.session
from sqlalchemy import desc

from tables.calendars import calendars
from tables.calendars_s import calendars_s


def sql_get_cal(sql_session, cal_id, workload=None, only_cal=False):
    """Retrieve a calendar or a calendar directory from the database.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param cal_id:
                the calendar ID to look for in the database.
    @param workload:
                workload to consider.
    @param only_cal:
                only look for a calendar if True, otherwise, look for
                a calendar or a directoy.
    @return:    the calendar.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given calendar ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    if workload is None:
        query = session.query(calendars)
    else:
        query = session.query(calendars_s)
        query = query.filter(calendars_s.workload_date == workload)
    if only_cal:
        query = query.filter_by(entry_type=0)
    query = query.filter_by(id=cal_id)
    try:
        cal = query.one()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise

    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return cal


def sql_get_cal_children(session, caldir_id, only_dir=False, workload=None,
                         sort_by_dirs=True):
    """Return the list of children of the specified directory.

    @param session:
                an opened SQLAlchemy session.
    @param caldir_id:
                parent calendar directory ID.
    @param only_dir:
                only return the children that are directories.
    @param workload:
                workload to consider.
    @param sort_by_dirs:
                list the directories first if True, otherwise, only sort
                by name.
    @return:    the list of children calendars/calendar directories.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given calendar directory ID is not in the database.
    """
    if workload is None:
        query = session.query(calendars)
        if sort_by_dirs:
            query = query.order_by(desc(calendars.entry_type))
        query = query.order_by(calendars.name)
    else:
        query = session.query(calendars_s)
        query = query.filter(calendars_s.workload_date == workload)
        if sort_by_dirs:
            query = query.order_by(desc(calendars_s.entry_type))
        query = query.order_by(calendars_s.name)
    if only_dir:
        query = query.filter_by(entry_type=1)
    return query.filter_by(parent=caldir_id).all()


def sql_count_cal_children(sql_session, caldir_id, only_dir=False,
                           workload=None):
    """Return the number of children of the specified directory.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param caldir_id:
                parent calendar directory ID.
    @param only_dir:
                only count the children that are directories.
    @param workload:
                workload to consider.
    @return:    the number of children.
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given calendar directory ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    if workload is None:
        query = session.query(calendars)
    else:
        query = session.query(calendars_s)
        query = query.filter(calendars_s.workload_date == workload)
    if only_dir:
        query = query.filter_by(entry_type=1)
    try:
        n = query.filter_by(parent=caldir_id).count()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return n


def sql_count_cal_children2(sql_session, caldir_id, workload=None):
    """Return the number of children of the specified directory.

    @param sql_session:
                SQLAlchemy session (it can be an opened session)
    @param caldir_id:
                parent calendar directory ID.
    @param workload:
                workload to consider.
    @return:    the tuple (#cals, #dirs)
    @raise sqlalchemy.orm.exc.NoResultFound:
                the given calendar directory ID is not in the database.
    """
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session

    # Number of sub-directories
    try:
        nb_dir = sql_count_cal_children(session, caldir_id, True, workload)
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise
    # Number of Jobs
    if workload is None:
        query = session.query(calendars)
    else:
        query = session.query(calendars_s)
        query = query.filter(calendars_s.workload_date == workload)
    query = query.filter_by(entry_type=0)
    try:
        nb_cal = query.filter_by(parent=caldir_id).count()
    except:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        raise
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (nb_cal, nb_dir)
