# Schedwi
# Copyright (C) 2011, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.


"""Functions to get parameters from the job/jobset hierarchy."""

from sqlalchemy.orm.exc import NoResultFound
import sqlalchemy.orm.session

from simple_queries_job import sql_get_job
from tables.job_cluster import job_cluster
from tables.job_host import job_host
from tables.job_cluster_s import job_cluster_s
from tables.job_host_s import job_host_s


def get_by_id(sql_session, ids, obj, workload=None):
    """Get the parameter from the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param ids:
                    the list of all the jobset IDs from the root jobset up to
                    the job/jobset for which parameters have to be retrieved.
                    If empty, then (False, None) is returned.
    @param obj:
                    database object (see L{tables}) to retrieve.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj) with from_obj to True if the
                    parameter has been retrieved from the object itself or
                    False if it is not defined in the object but has been
                    retrieved from a parent (inherited)
    """
    if not ids:
        return (False, None)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
        session.expire_on_commit = False
    else:
        session = sql_session
    # First, try to get the data from the jobs/jobset itself
    query = session.query(obj).filter_by(job_id=ids[-1])
    if workload is not None:
        query = query.filter_by(workload_date=workload)
    try:
        query = query.one()
    except NoResultFound:
        pass
    else:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, query)
    # Otherwise, look in the parent hierarchy
    for i in range(len(ids) - 2, -1, -1):
        query = session.query(obj).filter_by(job_id=ids[i])
        if workload is not None:
            query = query.filter_by(workload_date=workload)
        try:
            query = query.one()
        except NoResultFound:
            pass
        else:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, query)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (False, None)


def get(sql_session, path, obj, workload=None):
    """Get the parameter from the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param path:
                    L{path.Path} object.
    @param obj:
                    database object (see L{tables}) to retrieve.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj) with from_obj to True if the
                    parameter has been retrieved from the object itself or
                    False if it is not defined in the object but has been
                    retrieved from a parent (inherited)
    """
    return get_by_id(sql_session, path.id, obj, workload)


def get_list_by_id(sql_session, ids, obj, order=None, workload=None):
    """Get parameters from the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param ids:
                    the list of all the jobset IDs from the root jobset up to
                    the job/jobset for which parameters have to be retrieved.
                    If empty, the (False, None) is returned.
    @param obj:
                    database object (see L{tables}) to retrieve.
    @param order:
                    column name to sort the result.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj_list) with from_obj to True if the
                    parameters have been retrieved from the object itself or
                    False if they are not defined in the object but have been
                    retrieved from a parent (inherited)
    """
    if not ids:
        return (False, None)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
    else:
        session = sql_session
    # First, try to get the data from the ID of the object
    query = session.query(obj).filter_by(job_id=ids[-1])
    if workload is not None:
        query = query.filter_by(workload_date=workload)
    if order is not None:
        query = query.order_by(order)
    res = query.all()
    if res:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, res)
    # Otherwise, look in the parent hierarchy
    for i in range(len(ids) - 2, -1, -1):
        query = session.query(obj).filter_by(job_id=ids[i])
        if workload is not None:
            query = query.filter_by(workload_date=workload)
        if order is not None:
            query = query.order_by(order)
        res = query.all()
        if res:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, res)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (False, None)


def get_list(sql_session, path, obj, order=None, workload=None):
    """Get parameters from the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param path:
                    L{path.Path} object.
    @param obj:
                    database object (see L{tables}) to retrieve.
    @param order:
                    column name to sort the result.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj_list) with from_obj to True if the
                    parameters have been retrieved from the object itself or
                    False if they are not defined in the object but have been
                    retrieved from a parent (inherited)
    """
    return get_list_by_id(sql_session, path.id, obj, order, workload)


def get_start_time_id(sql_session, ids, workload=None):
    """Get the start time parameter from the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param ids:
                    the list of all the jobset IDs from the root jobset up to
                    the job/jobset for which the start time has to be
                    retrieved.  If empty, (False, None) is returned.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj) with from_obj to True if the
                    start time has been retrieved from the object itself or
                    False if it is not defined in the object but has been
                    retrieved from a parent (inherited).
                    obj is the L{tables.job_main.job_main} or
                    L{tables.job_main_s.job_main_s} object.
    """
    if not ids:
        return (False, None)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
        session.expire_on_commit = False
    else:
        session = sql_session
    # First, try to get the start time from the ID of the object
    job = sql_get_job(session, ids[-1], workload)
    if job.start_time >= 0:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, job)
    # Otherwise, look in the parent hierarchy
    for i in range(len(ids) - 2, -1, -1):
        job = sql_get_job(session, ids[i], workload)
        if job.start_time >= 0:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, job)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (False, None)


def get_calendar_id(sql_session, ids, workload=None):
    """Get the calendar ID parameter from the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param ids:
                    the list of all the jobset IDs from the root jobset up to
                    the job/jobset for which the calendar ID has to be
                    retrieved.  If empty, (False, None) is returned.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj) with from_obj to True if the
                    calendar has been retrieved from the object itself or
                    False if it is not defined in the object but has been
                    retrieved from a parent (inherited).
                    obj is the L{tables.job_main.job_main} or
                    L{tables.job_main_s.job_main_s} object.
    """
    if not ids:
        return (False, None)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
        session.expire_on_commit = False
    else:
        session = sql_session
    # First, try to get the calendar from the ID of the object
    job = sql_get_job(session, ids[-1], workload)
    if job.cal_id > 0:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, job)
    # Otherwise, look in the parent hierarchy
    for i in range(len(ids) - 2, -1, -1):
        job = sql_get_job(session, ids[i], workload)
        if job.cal_id > 0:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, job)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (False, None)


def get_cluster_or_host_by_id(sql_session, ids, workload=None):
    """Get the cluster or host associated with a the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param ids:
                    the list of all the jobset IDs from the root jobset up to
                    the job/jobset for which parameters have to be retrieved.
                    If empty, then (False, None) is returned.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj) with from_obj to True if the
                    cluster/host has been retrieved from the object itself or
                    False if it is not defined in the object but has been
                    retrieved from a parent (inherited).  The type of obj
                    is L{tables.job_cluster.job_cluster},
                    L{tables.job_host.job_host},
                    L{tables.job_cluster_s.job_cluster_s} or
                    L{tables.job_host_s.job_host_s}. This can be checked by
                    the caller by isinstance(obj, job_cluster) and
                    isinstance(obj, job_host) for instance.
    """
    if not ids:
        return (False, None)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        session = sql_session.open_session()
        session.expire_on_commit = False
    else:
        session = sql_session
    if workload is not None:
        table_cluster = job_cluster_s
        table_host = job_host_s
    else:
        table_cluster = job_cluster
        table_host = job_host
    # First, try to get the cluster/host from the jobs/jobset itself
    query = session.query(table_cluster).filter_by(job_id=ids[-1])
    if workload is not None:
        query = query.filter_by(workload_date=workload)
    try:
        query = query.one()
    except NoResultFound:
        pass
    else:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, query)
    query = session.query(table_host).filter_by(job_id=ids[-1])
    if workload is not None:
        query = query.filter_by(workload_date=workload)
    try:
        query = query.one()
    except NoResultFound:
        pass
    else:
        if not isinstance(sql_session, sqlalchemy.orm.session.Session):
            sql_session.close_session(session)
        return (True, query)
    # Otherwise, look in the parent hierarchy
    for i in range(len(ids) - 2, -1, -1):
        # Look for a cluster first
        query = session.query(table_cluster).filter_by(job_id=ids[i])
        if workload is not None:
            query = query.filter_by(workload_date=workload)
        try:
            query = query.one()
        except NoResultFound:
            pass
        else:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, query)
        # Look for a host
        query = session.query(table_host).filter_by(job_id=ids[i])
        if workload is not None:
            query = query.filter_by(workload_date=workload)
        try:
            query = query.one()
        except NoResultFound:
            pass
        else:
            if not isinstance(sql_session, sqlalchemy.orm.session.Session):
                sql_session.close_session(session)
            return (False, query)
    if not isinstance(sql_session, sqlalchemy.orm.session.Session):
        sql_session.close_session(session)
    return (False, None)


def get_cluster_or_host(sql_session, path, workload=None):
    """Get the cluster or host associated with a the job/jobset hierarchy.

    @param sql_session:
                    SQLAlchemy session (it can be an opened session)
    @param path:
                    L{path.Path} object.
    @param workload:
                    workload to consider.
    @return:
                    the tuple (from_obj, obj) with from_obj to True if the
                    cluster/host has been retrieved from the object itself or
                    False if it is not defined in the object but has been
                    retrieved from a parent (inherited).  The type of obj
                    is L{tables.job_cluster.job_cluster},
                    L{tables.job_host.job_host},
                    L{tables.job_cluster_s.job_cluster_s} or
                    L{tables.job_host_s.job_host_s}. This can be checked by
                    the caller by isinstance(obj, job_cluster) and
                    isinstance(obj, job_host) for instance.
    """
    return get_cluster_or_host_by_id(sql_session, path.id, workload)
