# Schedwi
# Copyright (C) 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""System calls to change the owner of the current process."""

import grp
import pwd
import os


def get_group_id(name):
    """
    Get the group id of the given group ID or name.

    @param name:
                group ID or group name. If None, the current process group ID
                is returned.
    @raise KeyError:
                the provided group name is not found.
    """
    if name is None:
        return os.getgid()
    try:
        gid = int(name)
    except:
        gid = grp.getgrnam(name).gr_gid
    return gid


def get_user_id(name):
    """
    Get the user id of the given user ID or name.

    @param name:
                user ID or user name. If None, the current process user ID
                is returned.
    @raise KeyError:
                the provided user name is not found.
    """
    if name is None:
        return os.getuid()
    try:
        uid = int(name)
    except:
        uid = pwd.getpwnam(name).pw_uid
    return uid


def change_group(gid):
    """
    Change the group id of the current process.

    @param gid:
                group ID.
    """
    if gid is None:
        return
    try:
        os.setregid(gid, gid)
    except:
        pass


def change_user(uid):
    """
    Change the user id of the current process.

    @param uid:
                user ID.
    """
    if uid is None:
        return
    try:
        os.setreuid(uid, uid)
    except:
        pass
