# Schedwi
# Copyright (C) 2011-2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from sqlalchemy.schema import Column, ForeignKeyConstraint
from sqlalchemy.types import Integer, String

from base import Base, convert2unicode

# Reasons for a waiting job/jobset
PARENT, TIME, LINKS, CONSTRAINT_FILE, MODULE, MANUAL = 1, 2, 4, 8, 16, 32


class job_status(Base):
    __tablename__ = 'job_status'
    __table_args__ = (
        ForeignKeyConstraint(['job_id', 'workload_date'],
                             ['job_main_s.id', 'job_main_s.workload_date']),
        {})

    job_id = Column(Integer, primary_key=True)
    status = Column(Integer)
    start_time = Column(Integer)  # Number of seconds since epoch
    retry_num = Column(Integer)
    wait_reason = Column(Integer)
    error_msg = Column(String(255))
    duration = Column(Integer)  # In seconds
    host_id = Column(Integer)
    workload_date = Column(Integer, primary_key=True)

    def __init__(self, status, start_time, retry_num, error_msg, duration,
                 host_id, workload_date):
        self.status = status
        self.start_time = start_time
        self.retry_num = retry_num
        self.error_msg = convert2unicode(error_msg)
        self.duration = duration
        self.host_id = host_id
        self.workload_date = workload_date

    def copy(self):
        return job_status(self.status, self.start_time, self.retry_num,
                          self.error_msg, self.duration, self.host_id,
                          self.workload_date)

    def is_waiting_for_parent(self):
        if self.status == 1 and self.wait_reason & PARENT:
            return True
        return False

    def is_waiting_for_time(self):
        if self.status == 1 and self.wait_reason & TIME:
            return True
        return False

    def is_waiting_for_links(self):
        if self.status == 1 and self.wait_reason & LINKS:
            return True
        return False

    def is_waiting_for_constraint_file(self):
        if self.status == 1 and self.wait_reason & CONSTRAINT_FILE:
            return True
        return False

    def is_waiting_for_module(self):
        if self.status == 1 and self.wait_reason & MODULE:
            return True
        return False

    def is_waiting_for_manual(self):
        if self.status == 1 and self.wait_reason & MANUAL:
            return True
        return False

    def get_waiting_message(self):
        """Return a text message for the waiting state reason.

        @return:
                The message (may be an empty string)
        """
        lst = list()
        if self.is_waiting_for_parent():
            lst.append(_("the start of parent jobset"))
        if self.is_waiting_for_time():
            lst.append(_("start time"))
        if self.is_waiting_for_links():
            lst.append(_("a dependency link"))
        if self.is_waiting_for_constraint_file():
            lst.append(_("a file"))
        if self.is_waiting_for_module():
            lst.append(_("an external module"))
        if self.is_waiting_for_manual():
            lst.append(_("manual acknowledgement"))
        if lst:
            return _("Waiting for ") + _(", ").join(lst)
        else:
            return ''
