# Schedwi
# Copyright (C) 2012, 2013 Herve Quatremain
#
# This file is part of Schedwi.
#
# Schedwi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Schedwi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Constraint file UI."""

from sqlalchemy import desc

from muntjac.api import (VerticalLayout, Table, HorizontalLayout, Button,
                         Window, Alignment, TextField, OptionGroup)
from muntjac.ui.window import Notification
from muntjac.data.util.indexed_container import IndexedContainer
from muntjac.terminal.theme_resource import ThemeResource
from muntjac.data.property import IValueChangeListener
from muntjac.ui.button import IClickListener

from web.jobtable import JobTableListener
from web.cutcopypastemenu import GenericContextMenu
from web.selecthostwidget import SelectHost
from web.autorefresh import AutoRefresh

from boxes import DeleteBox, BoxCloseListener
from tables.constraint_file import constraint_file
from tables.constraint_file_s import constraint_file_s
from tables.hosts import hosts
from tables.job_main_s import job_main_s
from simple_queries_job import sql_get_job


class Files(HorizontalLayout, JobTableListener):

    """Constraint file UI."""

    def __init__(self, jobTable, sql_session, workload=None):
        """Build the constraint file table.

        @param jobTable:
                    the associated L{web.jobtable.JobTable} object.
        @param sql_session:
                    SQLAlchemy session.
        @param workload:
                    workload to consider.
        """
        super(Files, self).__init__()
        self._sql_session = sql_session
        self._workload = workload
        self._selected = None
        self._job = None
        self._jobsetIds = None

        self.setMargin(True)
        self.setSpacing(True)
        self.setSizeFull()

        self._table = Table()
        self._table.setSizeFull()
        self._table.setSelectable(False)
        self._table.setMultiSelect(False)
        self._table.setColumnReorderingAllowed(False)
        self._table.setColumnCollapsingAllowed(False)
        self._table.setImmediate(True)
        self._container = self._newContainer()
        self._table.setContainerDataSource(self._container)
        self._table.setVisibleColumns(['name', 'host', 'must'])
        self._table.setColumnHeaders([_('File/Directory'), _('Host'),
                                      _('Must')])
        self._table.setColumnExpandRatio('name', 1)
        self.addComponent(self._table)
        self.setExpandRatio(self._table, 1.0)

        if workload is None or job_main_s.rw_constraint_file is True:
            self._table.setSelectable(True)
            self._table.addListener(SelectionHandler(self),
                                    IValueChangeListener)
            self._table.addActionHandler(FileContextMenu(self))

            btbox = VerticalLayout()
            btbox.setSpacing(True)
            btbox.setMargin(False)
            btbox.setWidth('110px')
            b = Button(_('Edit'))
            b.setIcon(ThemeResource('icons/edit.png'))
            b.setDescription(_("Edit the constraint"))
            b.setEnabled(False)
            b.setWidth('100%')
            b.addListener(EditBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btedit = b

            b = Button(_('Add'))
            b.setIcon(ThemeResource('icons/add.png'))
            b.setDescription(_("Add a new constraint file"))
            b.setEnabled(False)
            b.setWidth('100%')
            b.addListener(AddBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btadd = b

            b = Button(_('Delete'))
            b.setIcon(ThemeResource('icons/delete.png'))
            b.setDescription(_("Remove a constraint file"))
            b.setEnabled(False)
            b.setWidth('100%')
            b.addListener(DeleteBtHandler(self), IClickListener)
            btbox.addComponent(b)
            self._btdel = b

            self.addComponent(btbox)

        jobTable.add_listener(self)

    def _newContainer(self):
        """Return a new container for the job constraint table.

        @return:    the new container.
        """
        jContainer = IndexedContainer()
        jContainer.addContainerProperty('name', str, None)
        jContainer.addContainerProperty('host', str, None)
        jContainer.addContainerProperty('must', str, None)
        jContainer.addContainerProperty('must_bool', int, None)
        jContainer.addContainerProperty('host_id', long, None)
        return jContainer

    def getFileDetails(self, item_id):
        """Get the entry details from a container item ID.

        @param item_id:     the container item ID.
        @return:            a dictionnary containing the item details.
        """
        item = self._container.getItem(item_id)
        r = dict()
        r['name'] = item.getItemProperty('name').getValue()
        r['host'] = item.getItemProperty('host').getValue()
        r['must'] = item.getItemProperty('must').getValue()
        r['must_bool'] = item.getItemProperty('must_bool').getValue()
        r['host_id'] = item.getItemProperty('host_id').getValue()
        r['container_id'] = item_id
        return r

    def select_item(self, item_id):
        """Select the provided container ID.

        @param item_id:     the container ID of the item to select.
        """
        if item_id is not None:
            self._table.select(item_id)

    def refresh(self, job, jobsetIds):
        """Show the constraints in the table for the given job.

        @param job:
                    the job ID for which the constraint files must be shown.
        @param jobsetIds:
                    an array of the IDs the parent jobsets (the last element
                    is the ID of the jobset that contains `job')
        """
#        # Get the details of the selected item first so we will be able
#        # to re-select it at the end of the refresh
#        if job != self._job or self._selected is None:
#            self._selected = None  # A new job
#            filename = host_id = None
#        else:
#            item = self._container.getItem(self._selected)
#            filename = item.getItemProperty('name').getValue()
#            host_id = item.getItemProperty('host_id').getValue()

        self._container = self._newContainer()
        if job:
            session = self._sql_session.open_session()
            if self._workload is None:
                query = session.query(constraint_file, hosts)
                query = query.filter_by(job_id=job)
                query = query.filter(constraint_file.host_id == hosts.id)
                query = query.order_by(desc(constraint_file.exist))
            else:
                query = session.query(constraint_file_s, hosts)
                query = query.filter(constraint_file_s.workload_date ==
                                     self._workload)
                query = query.filter_by(job_id=job)
                query = query.filter(constraint_file_s.host_id == hosts.id)
                query = query.order_by(desc(constraint_file_s.exist))
#            new_selection_id = None
            for f, h in query.all():
                i = self._container.generateId()
                item = self._container.addItem(i)
                name = f.filename.encode('utf-8')
                item.getItemProperty('name').setValue(name)
                s = '%s (SSL)' if h.sslenable else '%s'
                item.getItemProperty('host').setValue(s % str(h))
                item.getItemProperty('must').setValue(
                    _('exist') if f.exist else _('not exist'))
                item.getItemProperty('must_bool').setValue(f.exist)
                item.getItemProperty('host_id').setValue(long(f.host_id))
    #            if f.filename == filename and f.host_id == host_id:
    #                new_selection_id = i
            self._sql_session.close_session(session)
            if self._workload is None or job_main_s.rw_constraint_file is True:
                self._btadd.setEnabled(True)
        else:
            if self._workload is None or job_main_s.rw_constraint_file is True:
                self._btadd.setEnabled(False)

        self._table.setContainerDataSource(self._container)
#        if new_selection_id:
#            self._table.select(new_selection_id)
        self._table.setVisibleColumns(['name', 'host', 'must'])
        self._job = job
        self._jobsetIds = jobsetIds[:]


class SelectionHandler(IValueChangeListener):

    """Manage the selection in the constraint table."""

    def __init__(self, file_obj):
        """Initialize the object.

        @param file_obj:  the associated L{Files} object.
        """
        super(SelectionHandler, self).__init__()
        self._c = file_obj

    def valueChange(self, event):
        """Called when an item has been selected/un-selected.

        @param event:  the listener event.
        """
        AutoRefresh.reset()
        v = event.getProperty().getValue()
        if v is not None:
            self._c._selected = v
            self._c._btedit.setEnabled(True)
            self._c._btdel.setEnabled(True)
        else:
            self._c._selected = None
            self._c._btedit.setEnabled(False)
            self._c._btdel.setEnabled(False)


class AddBtHandler(IClickListener):
    def __init__(self, file_obj):
        super(AddBtHandler, self).__init__()
        self._c = file_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        WindowFiles(self._c)


class EditBtHandler(IClickListener):
    def __init__(self, file_obj):
        super(EditBtHandler, self).__init__()
        self._c = file_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        if self._c._selected is not None:
            r = self._c.getFileDetails(self._c._selected)
            WindowFiles(self._c, r)


class DeleteBtHandler(IClickListener):
    def __init__(self, file_obj):
        super(DeleteBtHandler, self).__init__()
        self._c = file_obj

    def buttonClick(self, event):
        AutoRefresh.reset()
        if self._c._selected is not None:
            r = self._c.getFileDetails(self._c._selected)
            d = DeleteBox(
                self._c,
                _("Delete"),
                _('Are you sure you want to delete this constraint?'),
                _('%s on %s') % (r['name'], str(r['host'])))
            d.addListener(DeleteConfirmed(self._c, r['host_id'], r['name']))


class DeleteConfirmed(BoxCloseListener):

    """Manage constraint file deletion."""

    def __init__(self, file_obj, host_id, filename):
        """Initialize the object.

        @param file_obj:
                    the associated L{Files} object.
        @param host_id:
                    database ID of the host part of the constraint file.
        @param filename:
                    file name part of the constraint file.
        """
        self._c = file_obj
        self._host_id = host_id
        self._filename = filename

    def boxClose(self, event):
        """Called when the user confirmed the deletion."""
        AutoRefresh.reset()
        session = self._c._sql_session.open_session()
        try:
            job = sql_get_job(session, self._c._job, self._c._workload)
        except:
            self._c._sql_session.close_session(session)
            self._c.getWindow().showNotification(
                _("Cannot get the selected job details"),
                _("<br/>Maybe someone else just removed it."),
                Notification.TYPE_ERROR_MESSAGE)
            return
        # Retrieve the constraint file to remove
        for cf in job.constraint_file:
            if cf.host_id == self._host_id and \
               cf.filename.encode('utf-8') == self._filename:
                try:
                    job.constraint_file.remove(cf)
                except:
                    pass
                break
        self._c._sql_session.close_session(session)
        self._c.refresh(self._c._job, self._c._jobsetIds)


class FileContextMenu(GenericContextMenu):

    """Right button context menu."""

    def __init__(self, file_obj):
        """Initialize the object.

        @param file_obj:
                    the associated L{Files} object.
        """
        super(FileContextMenu, self).__init__()
        self._c = file_obj

    def getActions(self, target, sender):
        if self._c._job:
            if not target:
                # Background
                return [self._ACTION_ADD]
            return [self._ACTION_EDIT, self._ACTION_ADD, self._ACTION_DELETE]
        else:
            return []

    def handleAction(self, a, sender, target):
        AutoRefresh.reset()
        if a.getCaption() == self._ACTION_EDIT.getCaption():
            r = self._c.getFileDetails(target)
            WindowFiles(self._c, r)

        elif a.getCaption() == self._ACTION_ADD.getCaption():
            WindowFiles(self._c)

        elif a.getCaption() == self._ACTION_DELETE.getCaption():
            r = self._c.getFileDetails(target)
            d = DeleteBox(
                self._c,
                _("Delete"),
                _('Are you sure you want to delete this constraint?'),
                _('%s on %s') % (r['name'], str(r['host'])))
            d.addListener(DeleteConfirmed(self._c, r['host_id'], r['name']))


class WindowFiles(IClickListener):

    """Edit/Add window for a constraint file."""

    _must_labels = [_('Exist'), _('NOT Exist')]
    _bt_captions = [_('Cancel'), _('OK')]

    def __init__(self, file_obj, params=None):
        """Create and display the Edit/Add window.

        @param file_obj:
                    the associated L{Files} object.
        @param params:
                    when the window is used to edit an existing constraint,
                    this is the constraint details.  This parameter is
                    a dictionnary as returned by L{Files.getFileDetails}.
        """
        super(WindowFiles, self).__init__()

        self._c = file_obj
        self._params = params

        if params is None:
            title = _('New File constraint')
        else:
            title = _('Edit File constraint')
        self._w = Window(title)
        self._w.setWidth("360px")

        # VerticalLayout as content by default
        v = self._w.getContent()
        v.setSizeFull()
        v.setMargin(True)
        v.setSpacing(True)

        # Form
        t = TextField(_('File/Directory Name:'))
        t.setWidth('100%')
        t.setDescription(_("Full path to the file or directory to monitor"))
        if params and 'name' in params:
            t.setValue(params['name'])
        self._name = t
        v.addComponent(t)

        s = SelectHost(self._c._sql_session, _("Host:"),
                       _("Host on which the file will be monitored"),
                       params['host_id'] if params and 'host_id' in params
                       else None)
        self._host = s
        s.setWidth('100%')
        v.addComponent(s)

        o = OptionGroup(_("Must:"), map(lambda i: _(i), self._must_labels))
        o.setNullSelectionAllowed(False)
        if params is None:
            o.select(_(self._must_labels[0]))
        elif 'must_bool' in params:
            o.select(_(self._must_labels[0]) if params['must_bool']
                     else _(self._must_labels[1]))
        self._must = o
        v.addComponent(o)

        # Button box
        h_bt = HorizontalLayout()
        h_bt.setMargin(False)
        h_bt.setSpacing(True)

        for caption in self._bt_captions:
            b = Button(_(caption))
            b.addListener(self, IClickListener)
            h_bt.addComponent(b)

        v.addComponent(h_bt)
        v.setComponentAlignment(h_bt, Alignment.BOTTOM_RIGHT)

        self._c.getWindow().addWindow(self._w)
        self._w.center()

    def buttonClick(self, event):
        AutoRefresh.reset()
        # First button is Cancel
        if event.getButton().getCaption() != _(self._bt_captions[0]):
            # Retrieve the values from the form
            name = self._name.getValue().strip()
            if not name:
                self._c.getWindow().showNotification(
                    _("File/Directory name is empty"),
                    _("<br/>The file or directory name cannot \
                       be empty or contain just spaces."),
                    Notification.TYPE_ERROR_MESSAGE)
                self._name.focus()
                return
            host_id = self._host.getValue()
            if self._must.getValue() == _(self._must_labels[0]):
                must_bool = 1
            else:
                must_bool = 0

            # Get the job details from the database
            session = self._c._sql_session.open_session()
            try:
                job = sql_get_job(session, self._c._job, self._c._workload)
            except:
                self._c._sql_session.close_session(session)
                self._c.getWindow().showNotification(
                    _("Cannot get the selected job details"),
                    _("<br/>Maybe someone else just removed it."),
                    Notification.TYPE_ERROR_MESSAGE)
                # Close the window
                self._c.getWindow().removeWindow(self._w)
                return

            # Retrieve the edited constraint file from the database
            for cf in job.constraint_file:
                if cf.host_id == host_id and \
                   cf.filename.encode('utf-8') == name:
                    # Already in the database
                    # If it was an Add window, print an error message.
                    if self._params is None:
                        self._c._sql_session.close_session(session)
                        self._c.getWindow().showNotification(
                            _("Constraint file already exists"),
                            _("<br/>This new constraint file %s is already \
                            defined in the database.") % name,
                            Notification.TYPE_ERROR_MESSAGE)
                        return
                    break
            else:
                cf = None

            # Add a new record
            if self._params is None:
                if self._c._workload is None:
                    c = constraint_file(host_id, name, must_bool)
                else:
                    c = constraint_file_s(host_id, name, must_bool,
                                          self._c._workload)
                job.constraint_file.append(c)

            # Edit a record
            else:
                # Nothing has been changed (maybe the `Exist' flag)
                if host_id == self._params['host_id'] and \
                   name == self._params['name']:
                    if cf:
                        if cf.exist != must_bool:
                            cf.exist = must_bool
                    else:
                        # Humm, it should have been in the database...
                        # So recreating the constraint file
                        if self._c._workload is None:
                            c = constraint_file(host_id, name, must_bool)
                        else:
                            c = constraint_file_s(host_id, name, must_bool,
                                                  self._c._workload)
                        job.constraint_file.append(c)

                # The file name or the host_id have changed
                else:
                    if cf:
                        self._c._sql_session.close_session(session)
                        self._c.getWindow().showNotification(
                            _("Constraint file already exists"),
                            _("<br/>This new constraint file %s is already \
                            defined in the database.") % name,
                            Notification.TYPE_ERROR_MESSAGE)
                        return

                    # Find and edit the old record in the database
                    for c in job.constraint_file:
                        if c.host_id == self._params['host_id'] and \
                           c.filename.encode('utf-8') == self._params['name']:
                            c.host_id = host_id
                            c.filename = name.decode('utf-8')
                            c.exist = must_bool
                            break
                    else:
                        # Humm, it should have been in the database...
                        # So creating the constraint file
                        if self._c._workload is None:
                            c = constraint_file(host_id, name, must_bool)
                        else:
                            c = constraint_file_s(host_id, name, must_bool,
                                                  self._c._workload)
                        job.constraint_file.append(c)

            self._c._sql_session.close_session(session)
            self._c.refresh(self._c._job, self._c._jobsetIds)

        # Close the window
        self._c.getWindow().removeWindow(self._w)
